

package space.yizhu.record.plugin.activerecord.generator;

import java.util.List;
import javax.sql.DataSource;

import space.yizhu.record.plugin.activerecord.dialect.Dialect;


/**
 * <p>Generator class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class Generator {

    protected Dialect dialect = null;
    protected MetaBuilder metaBuilder;
    protected BaseModelGenerator baseModelGenerator;
    protected ModelGenerator modelGenerator;
    protected MappingKitGenerator mappingKitGenerator;
    protected DataDictionaryGenerator dataDictionaryGenerator;
    protected boolean generateDataDictionary = false;

    
    /**
     * <p>Constructor for Generator.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param baseModelPackageName a {@link java.lang.String} object.
     * @param baseModelOutputDir a {@link java.lang.String} object.
     * @param modelPackageName a {@link java.lang.String} object.
     * @param modelOutputDir a {@link java.lang.String} object.
     */
    public Generator(DataSource dataSource, String baseModelPackageName, String baseModelOutputDir, String modelPackageName, String modelOutputDir) {
        this(dataSource, new BaseModelGenerator(baseModelPackageName, baseModelOutputDir), new ModelGenerator(modelPackageName, baseModelPackageName, modelOutputDir));
    }

    
    /**
     * <p>Constructor for Generator.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param baseModelPackageName a {@link java.lang.String} object.
     * @param baseModelOutputDir a {@link java.lang.String} object.
     */
    public Generator(DataSource dataSource, String baseModelPackageName, String baseModelOutputDir) {
        this(dataSource, new BaseModelGenerator(baseModelPackageName, baseModelOutputDir));
    }

    /**
     * <p>Constructor for Generator.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param baseModelGenerator a {@link space.yizhu.record.plugin.activerecord.generator.BaseModelGenerator} object.
     */
    public Generator(DataSource dataSource, BaseModelGenerator baseModelGenerator) {
        if (dataSource == null) {
            throw new IllegalArgumentException("dataSource can not be null.");
        }
        if (baseModelGenerator == null) {
            throw new IllegalArgumentException("baseModelGenerator can not be null.");
        }

        this.metaBuilder = new MetaBuilder(dataSource);
        this.baseModelGenerator = baseModelGenerator;
        this.modelGenerator = null;
        this.mappingKitGenerator = null;
        this.dataDictionaryGenerator = null;
    }

    
    /**
     * <p>Constructor for Generator.</p>
     *
     * @param dataSource a {@link javax.sql.DataSource} object.
     * @param baseModelGenerator a {@link space.yizhu.record.plugin.activerecord.generator.BaseModelGenerator} object.
     * @param modelGenerator a {@link space.yizhu.record.plugin.activerecord.generator.ModelGenerator} object.
     */
    public Generator(DataSource dataSource, BaseModelGenerator baseModelGenerator, ModelGenerator modelGenerator) {
        if (dataSource == null) {
            throw new IllegalArgumentException("dataSource can not be null.");
        }
        if (baseModelGenerator == null) {
            throw new IllegalArgumentException("baseModelGenerator can not be null.");
        }
        if (modelGenerator == null) {
            throw new IllegalArgumentException("modelGenerator can not be null.");
        }

        this.metaBuilder = new MetaBuilder(dataSource);
        this.baseModelGenerator = baseModelGenerator;
        this.modelGenerator = modelGenerator;
        this.mappingKitGenerator = new MappingKitGenerator(modelGenerator.modelPackageName, modelGenerator.modelOutputDir);
        this.dataDictionaryGenerator = new DataDictionaryGenerator(dataSource, modelGenerator.modelOutputDir);
    }

    
    /**
     * <p>Setter for the field <code>metaBuilder</code>.</p>
     *
     * @param metaBuilder a {@link space.yizhu.record.plugin.activerecord.generator.MetaBuilder} object.
     */
    public void setMetaBuilder(MetaBuilder metaBuilder) {
        if (metaBuilder != null) {
            this.metaBuilder = metaBuilder;
        }
    }

    
    /**
     * <p>setGenerateRemarks.</p>
     *
     * @param generateRemarks a boolean.
     */
    public void setGenerateRemarks(boolean generateRemarks) {
        if (metaBuilder != null) {
            metaBuilder.setGenerateRemarks(generateRemarks);
        }
    }

    /**
     * <p>setTypeMapping.</p>
     *
     * @param typeMapping a {@link space.yizhu.record.plugin.activerecord.generator.TypeMapping} object.
     */
    public void setTypeMapping(TypeMapping typeMapping) {
        this.metaBuilder.setTypeMapping(typeMapping);
    }

    
    /**
     * <p>Setter for the field <code>mappingKitGenerator</code>.</p>
     *
     * @param mappingKitGenerator a {@link space.yizhu.record.plugin.activerecord.generator.MappingKitGenerator} object.
     */
    public void setMappingKitGenerator(MappingKitGenerator mappingKitGenerator) {
        if (mappingKitGenerator != null) {
            this.mappingKitGenerator = mappingKitGenerator;
        }
    }

    
    /**
     * <p>Setter for the field <code>dataDictionaryGenerator</code>.</p>
     *
     * @param dataDictionaryGenerator a {@link space.yizhu.record.plugin.activerecord.generator.DataDictionaryGenerator} object.
     */
    public void setDataDictionaryGenerator(DataDictionaryGenerator dataDictionaryGenerator) {
        if (dataDictionaryGenerator != null) {
            this.dataDictionaryGenerator = dataDictionaryGenerator;
        }
    }

    
    /**
     * <p>Setter for the field <code>dialect</code>.</p>
     *
     * @param dialect a {@link space.yizhu.record.plugin.activerecord.dialect.Dialect} object.
     */
    public void setDialect(Dialect dialect) {
        this.dialect = dialect;
    }

    
    /**
     * <p>setBaseModelTemplate.</p>
     *
     * @param baseModelTemplate a {@link java.lang.String} object.
     */
    public void setBaseModelTemplate(String baseModelTemplate) {
        baseModelGenerator.setTemplate(baseModelTemplate);
    }

    
    /**
     * <p>setGenerateChainSetter.</p>
     *
     * @param generateChainSetter a boolean.
     */
    public void setGenerateChainSetter(boolean generateChainSetter) {
        baseModelGenerator.setGenerateChainSetter(generateChainSetter);
    }

    
    /**
     * <p>setRemovedTableNamePrefixes.</p>
     *
     * @param removedTableNamePrefixes a {@link java.lang.String} object.
     */
    public void setRemovedTableNamePrefixes(String... removedTableNamePrefixes) {
        metaBuilder.setRemovedTableNamePrefixes(removedTableNamePrefixes);
    }

    
    /**
     * <p>addExcludedTable.</p>
     *
     * @param excludedTables a {@link java.lang.String} object.
     */
    public void addExcludedTable(String... excludedTables) {
        metaBuilder.addExcludedTable(excludedTables);
    }

    
    /**
     * <p>setModelTemplate.</p>
     *
     * @param modelTemplate a {@link java.lang.String} object.
     */
    public void setModelTemplate(String modelTemplate) {
        if (modelGenerator != null) {
            modelGenerator.setTemplate(modelTemplate);
        }
    }

    
    /**
     * <p>setGenerateDaoInModel.</p>
     *
     * @param generateDaoInModel a boolean.
     */
    public void setGenerateDaoInModel(boolean generateDaoInModel) {
        if (modelGenerator != null) {
            modelGenerator.setGenerateDaoInModel(generateDaoInModel);
        }
    }

    
    /**
     * <p>Setter for the field <code>generateDataDictionary</code>.</p>
     *
     * @param generateDataDictionary a boolean.
     */
    public void setGenerateDataDictionary(boolean generateDataDictionary) {
        this.generateDataDictionary = generateDataDictionary;
    }

    
    /**
     * <p>setMappingKitTemplate.</p>
     *
     * @param mappingKitTemplate a {@link java.lang.String} object.
     */
    public void setMappingKitTemplate(String mappingKitTemplate) {
        if (this.mappingKitGenerator != null) {
            this.mappingKitGenerator.setTemplate(mappingKitTemplate);
        }
    }

    
    /**
     * <p>setMappingKitOutputDir.</p>
     *
     * @param mappingKitOutputDir a {@link java.lang.String} object.
     */
    public void setMappingKitOutputDir(String mappingKitOutputDir) {
        if (this.mappingKitGenerator != null) {
            this.mappingKitGenerator.setMappingKitOutputDir(mappingKitOutputDir);
        }
    }

    
    /**
     * <p>setMappingKitPackageName.</p>
     *
     * @param mappingKitPackageName a {@link java.lang.String} object.
     */
    public void setMappingKitPackageName(String mappingKitPackageName) {
        if (this.mappingKitGenerator != null) {
            this.mappingKitGenerator.setMappingKitPackageName(mappingKitPackageName);
        }
    }

    
    /**
     * <p>setMappingKitClassName.</p>
     *
     * @param mappingKitClassName a {@link java.lang.String} object.
     */
    public void setMappingKitClassName(String mappingKitClassName) {
        if (this.mappingKitGenerator != null) {
            this.mappingKitGenerator.setMappingKitClassName(mappingKitClassName);
        }
    }

    
    /**
     * <p>setDataDictionaryOutputDir.</p>
     *
     * @param dataDictionaryOutputDir a {@link java.lang.String} object.
     */
    public void setDataDictionaryOutputDir(String dataDictionaryOutputDir) {
        if (this.dataDictionaryGenerator != null) {
            this.dataDictionaryGenerator.setDataDictionaryOutputDir(dataDictionaryOutputDir);
        }
    }

    
    /**
     * <p>setDataDictionaryFileName.</p>
     *
     * @param dataDictionaryFileName a {@link java.lang.String} object.
     */
    public void setDataDictionaryFileName(String dataDictionaryFileName) {
        if (dataDictionaryGenerator != null) {
            dataDictionaryGenerator.setDataDictionaryFileName(dataDictionaryFileName);
        }
    }

    /**
     * <p>generate.</p>
     */
    public void generate() {
        if (dialect != null) {
            metaBuilder.setDialect(dialect);
        }

        long start = System.currentTimeMillis();
        List<TableMeta> tableMetas = metaBuilder.build();
        if (tableMetas.size() == 0) {
            System.out.println("TableMeta 数量为 0，不生成任何文件");
            return;
        }

        baseModelGenerator.generate(tableMetas);

        if (modelGenerator != null) {
            modelGenerator.generate(tableMetas);
        }

        if (mappingKitGenerator != null) {
            mappingKitGenerator.generate(tableMetas);
        }

        if (dataDictionaryGenerator != null && generateDataDictionary) {
            dataDictionaryGenerator.generate(tableMetas);
        }

        long usedTime = (System.currentTimeMillis() - start) / 1000;
        System.out.println("Generate complete in " + usedTime + " seconds.");
    }
}



