

package space.yizhu.record.plugin.druid;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;

import com.alibaba.druid.filter.Filter;
import com.alibaba.druid.pool.DruidDataSource;
import space.yizhu.kits.StrKit;
import space.yizhu.record.plugin.IPlugin;
import space.yizhu.record.plugin.activerecord.IDataSourceProvider;


/**
 * <p>DruidPlugin class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class DruidPlugin implements IPlugin, IDataSourceProvider {
    
    private String name = null;

    
    private String url;
    private String username;
    private String password;
    private String publicKey;
    private String driverClass = null;    

    
    private int initialSize = 1;
    private int minIdle = 10;
    private int maxActive = 32;

    
    private long maxWait = DruidDataSource.DEFAULT_MAX_WAIT;

    
    private long timeBetweenEvictionRunsMillis = DruidDataSource.DEFAULT_TIME_BETWEEN_EVICTION_RUNS_MILLIS;
    
    private long minEvictableIdleTimeMillis = DruidDataSource.DEFAULT_MIN_EVICTABLE_IDLE_TIME_MILLIS;
    
    private long timeBetweenConnectErrorMillis = DruidDataSource.DEFAULT_TIME_BETWEEN_CONNECT_ERROR_MILLIS;

    
    private String validationQuery = "select 1";
    private String connectionInitSql = null;
    private String connectionProperties = null;
    private boolean testWhileIdle = true;
    private boolean testOnBorrow = false;
    private boolean testOnReturn = false;

    
    private boolean removeAbandoned = false;
    
    private long removeAbandonedTimeoutMillis = 300 * 1000;
    
    private boolean logAbandoned = false;

    
    

    
    private int maxPoolPreparedStatementPerConnectionSize = -1;

    
    private String filters;    
    private List<Filter> filterList;

    private DruidDataSource ds;
    private volatile boolean isStarted = false;

    /**
     * <p>Constructor for DruidPlugin.</p>
     *
     * @param url a {@link java.lang.String} object.
     * @param username a {@link java.lang.String} object.
     * @param password a {@link java.lang.String} object.
     */
    public DruidPlugin(String url, String username, String password) {
        this.url = url;
        this.username = username;
        this.password = password;
        this.validationQuery = autoCheckValidationQuery(url);
    }

    /**
     * <p>Constructor for DruidPlugin.</p>
     *
     * @param url a {@link java.lang.String} object.
     * @param username a {@link java.lang.String} object.
     * @param password a {@link java.lang.String} object.
     * @param driverClass a {@link java.lang.String} object.
     */
    public DruidPlugin(String url, String username, String password, String driverClass) {
        this.url = url;
        this.username = username;
        this.password = password;
        this.driverClass = driverClass;
        this.validationQuery = autoCheckValidationQuery(url);
    }

    /**
     * <p>Constructor for DruidPlugin.</p>
     *
     * @param url a {@link java.lang.String} object.
     * @param username a {@link java.lang.String} object.
     * @param password a {@link java.lang.String} object.
     * @param driverClass a {@link java.lang.String} object.
     * @param filters a {@link java.lang.String} object.
     */
    public DruidPlugin(String url, String username, String password, String driverClass, String filters) {
        this.url = url;
        this.username = username;
        this.password = password;
        this.driverClass = driverClass;
        this.filters = filters;
        this.validationQuery = autoCheckValidationQuery(url);
    }

    
    private static String autoCheckValidationQuery(String url) {
        if (url.startsWith("jdbc:oracle")) {
            return "select 1 from dual";
        } else if (url.startsWith("jdbc:db2")) {
            return "select 1 from sysibm.sysdummy1";
        } else if (url.startsWith("jdbc:hsqldb")) {
            return "select 1 from INFORMATION_SCHEMA.SYSTEM_USERS";
        } else if (url.startsWith("jdbc:derby")) {
            return "select 1 from INFORMATION_SCHEMA.SYSTEM_USERS";
        }
        return "select 1";
    }

    
    /**
     * <p>Setter for the field <code>connectionInitSql</code>.</p>
     *
     * @param sql a {@link java.lang.String} object.
     */
    public void setConnectionInitSql(String sql) {
        this.connectionInitSql = sql;
    }

    /**
     * <p>Getter for the field <code>name</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public final String getName() {
        return name;
    }

    
    /**
     * <p>Setter for the field <code>name</code>.</p>
     *
     * @param name a {@link java.lang.String} object.
     */
    public final void setName(String name) {
        this.name = name;
    }

    
    /**
     * <p>Setter for the field <code>filters</code>.</p>
     *
     * @param filters a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setFilters(String filters) {
        this.filters = filters;
        return this;
    }

    /**
     * <p>addFilter.</p>
     *
     * @param filter a {@link com.alibaba.druid.filter.Filter} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public synchronized DruidPlugin addFilter(Filter filter) {
        if (filterList == null)
            filterList = new ArrayList<Filter>();
        filterList.add(filter);
        return this;
    }

    /**
     * <p>start.</p>
     *
     * @return a boolean.
     */
    public boolean start() {
        if (isStarted)
            return true;

        ds = new DruidDataSource();
        if (this.name != null) {
            ds.setName(this.name);
        }
        ds.setUrl(url);
        ds.setUsername(username);
        ds.setPassword(password);
        if (driverClass != null)
            ds.setDriverClassName(driverClass);
        ds.setInitialSize(initialSize);
        ds.setMinIdle(minIdle);
        ds.setMaxActive(maxActive);
        ds.setMaxWait(maxWait);
        ds.setTimeBetweenConnectErrorMillis(timeBetweenConnectErrorMillis);
        ds.setTimeBetweenEvictionRunsMillis(timeBetweenEvictionRunsMillis);
        ds.setMinEvictableIdleTimeMillis(minEvictableIdleTimeMillis);

        ds.setValidationQuery(validationQuery);
        if (StrKit.notBlank(connectionInitSql)) {
            List<String> connectionInitSqls = new ArrayList<String>();
            connectionInitSqls.add(this.connectionInitSql);
            ds.setConnectionInitSqls(connectionInitSqls);
        }
        ds.setTestWhileIdle(testWhileIdle);
        ds.setTestOnBorrow(testOnBorrow);
        ds.setTestOnReturn(testOnReturn);

        ds.setRemoveAbandoned(removeAbandoned);
        ds.setRemoveAbandonedTimeoutMillis(removeAbandonedTimeoutMillis);
        ds.setLogAbandoned(logAbandoned);

        
        ds.setMaxPoolPreparedStatementPerConnectionSize(maxPoolPreparedStatementPerConnectionSize);

        boolean hasSetConnectionProperties = false;
        if (StrKit.notBlank(filters)) {
            try {
                ds.setFilters(filters);
                
                if (filters.contains("config")) {
                    
                    if (StrKit.isBlank(this.publicKey)) {
                        throw new RuntimeException("Druid连接池的filter设定了config时，必须设定publicKey");
                    }
                    String decryptStr = "config.decrypt=true;config.decrypt.key=" + this.publicKey;
                    String cp = this.connectionProperties;
                    if (StrKit.isBlank(cp)) {
                        cp = decryptStr;
                    } else {
                        cp = cp + ";" + decryptStr;
                    }
                    ds.setConnectionProperties(cp);
                    hasSetConnectionProperties = true;
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
        
        if (!hasSetConnectionProperties && StrKit.notBlank(this.connectionProperties)) {
            ds.setConnectionProperties(this.connectionProperties);
        }
        addFilterList(ds);

        isStarted = true;
        return true;
    }

    private void addFilterList(DruidDataSource ds) {
        if (filterList != null) {
            List<Filter> targetList = ds.getProxyFilters();
            for (Filter add : filterList) {
                boolean found = false;
                for (Filter target : targetList) {
                    if (add.getClass().equals(target.getClass())) {
                        found = true;
                        break;
                    }
                }
                if (!found)
                    targetList.add(add);
            }
        }
    }

    /**
     * <p>stop.</p>
     *
     * @return a boolean.
     */
    public boolean stop() {
        if (ds != null)
            ds.close();

        ds = null;
        isStarted = false;
        return true;
    }

    /**
     * <p>getDataSource.</p>
     *
     * @return a {@link javax.sql.DataSource} object.
     */
    public DataSource getDataSource() {
        return ds;
    }

    /**
     * <p>set.</p>
     *
     * @param initialSize a int.
     * @param minIdle a int.
     * @param maxActive a int.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin set(int initialSize, int minIdle, int maxActive) {
        this.initialSize = initialSize;
        this.minIdle = minIdle;
        this.maxActive = maxActive;
        return this;
    }

    /**
     * <p>Setter for the field <code>driverClass</code>.</p>
     *
     * @param driverClass a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setDriverClass(String driverClass) {
        this.driverClass = driverClass;
        return this;
    }

    /**
     * <p>Setter for the field <code>initialSize</code>.</p>
     *
     * @param initialSize a int.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setInitialSize(int initialSize) {
        this.initialSize = initialSize;
        return this;
    }

    /**
     * <p>Setter for the field <code>minIdle</code>.</p>
     *
     * @param minIdle a int.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setMinIdle(int minIdle) {
        this.minIdle = minIdle;
        return this;
    }

    /**
     * <p>Setter for the field <code>maxActive</code>.</p>
     *
     * @param maxActive a int.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setMaxActive(int maxActive) {
        this.maxActive = maxActive;
        return this;
    }

    /**
     * <p>Setter for the field <code>maxWait</code>.</p>
     *
     * @param maxWait a long.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setMaxWait(long maxWait) {
        this.maxWait = maxWait;
        return this;
    }

    /**
     * <p>Setter for the field <code>timeBetweenEvictionRunsMillis</code>.</p>
     *
     * @param timeBetweenEvictionRunsMillis a long.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setTimeBetweenEvictionRunsMillis(long timeBetweenEvictionRunsMillis) {
        this.timeBetweenEvictionRunsMillis = timeBetweenEvictionRunsMillis;
        return this;
    }

    /**
     * <p>Setter for the field <code>minEvictableIdleTimeMillis</code>.</p>
     *
     * @param minEvictableIdleTimeMillis a long.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setMinEvictableIdleTimeMillis(long minEvictableIdleTimeMillis) {
        this.minEvictableIdleTimeMillis = minEvictableIdleTimeMillis;
        return this;
    }

    
    /**
     * <p>Setter for the field <code>validationQuery</code>.</p>
     *
     * @param validationQuery a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setValidationQuery(String validationQuery) {
        this.validationQuery = validationQuery;
        return this;
    }

    /**
     * <p>Setter for the field <code>testWhileIdle</code>.</p>
     *
     * @param testWhileIdle a boolean.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setTestWhileIdle(boolean testWhileIdle) {
        this.testWhileIdle = testWhileIdle;
        return this;
    }

    /**
     * <p>Setter for the field <code>testOnBorrow</code>.</p>
     *
     * @param testOnBorrow a boolean.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setTestOnBorrow(boolean testOnBorrow) {
        this.testOnBorrow = testOnBorrow;
        return this;
    }

    /**
     * <p>Setter for the field <code>testOnReturn</code>.</p>
     *
     * @param testOnReturn a boolean.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setTestOnReturn(boolean testOnReturn) {
        this.testOnReturn = testOnReturn;
        return this;
    }

    /**
     * <p>Setter for the field <code>maxPoolPreparedStatementPerConnectionSize</code>.</p>
     *
     * @param maxPoolPreparedStatementPerConnectionSize a int.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public DruidPlugin setMaxPoolPreparedStatementPerConnectionSize(int maxPoolPreparedStatementPerConnectionSize) {
        this.maxPoolPreparedStatementPerConnectionSize = maxPoolPreparedStatementPerConnectionSize;
        return this;
    }

    /**
     * <p>Setter for the field <code>timeBetweenConnectErrorMillis</code>.</p>
     *
     * @param timeBetweenConnectErrorMillis a long.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setTimeBetweenConnectErrorMillis(long timeBetweenConnectErrorMillis) {
        this.timeBetweenConnectErrorMillis = timeBetweenConnectErrorMillis;
        return this;
    }

    /**
     * <p>Setter for the field <code>removeAbandoned</code>.</p>
     *
     * @param removeAbandoned a boolean.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setRemoveAbandoned(boolean removeAbandoned) {
        this.removeAbandoned = removeAbandoned;
        return this;
    }

    /**
     * <p>Setter for the field <code>removeAbandonedTimeoutMillis</code>.</p>
     *
     * @param removeAbandonedTimeoutMillis a long.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setRemoveAbandonedTimeoutMillis(long removeAbandonedTimeoutMillis) {
        this.removeAbandonedTimeoutMillis = removeAbandonedTimeoutMillis;
        return this;
    }

    /**
     * <p>Setter for the field <code>logAbandoned</code>.</p>
     *
     * @param logAbandoned a boolean.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setLogAbandoned(boolean logAbandoned) {
        this.logAbandoned = logAbandoned;
        return this;
    }

    /**
     * <p>Setter for the field <code>connectionProperties</code>.</p>
     *
     * @param connectionProperties a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setConnectionProperties(String connectionProperties) {
        this.connectionProperties = connectionProperties;
        return this;
    }

    /**
     * <p>Setter for the field <code>publicKey</code>.</p>
     *
     * @param publicKey a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.plugin.druid.DruidPlugin} object.
     */
    public final DruidPlugin setPublicKey(String publicKey) {
        this.publicKey = publicKey;
        return this;
    }
}
