

package space.yizhu.record.template;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import space.yizhu.kits.HashKit;
import space.yizhu.kits.StrKit;
import space.yizhu.kits.SyncWriteMap;
import space.yizhu.record.template.expr.ast.FieldGetter;
import space.yizhu.record.template.expr.ast.FieldKeyBuilder;
import space.yizhu.record.template.expr.ast.FieldKit;
import space.yizhu.record.template.expr.ast.MethodKit;
import space.yizhu.record.template.source.ClassPathSourceFactory;
import space.yizhu.record.template.source.ISource;
import space.yizhu.record.template.source.ISourceFactory;
import space.yizhu.record.template.source.StringSource;
import space.yizhu.record.template.stat.OutputDirectiveFactory;
import space.yizhu.record.template.stat.Parser;
import space.yizhu.record.template.stat.ast.Stat;


/**
 * <p>Engine class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class Engine {

    /** Constant <code>MAIN_ENGINE_NAME="main"</code> */
    public static final String MAIN_ENGINE_NAME = "main";

    private static Engine MAIN_ENGINE;
    private static Map<String, Engine> engineMap = new HashMap<String, Engine>(64, 0.5F);

    
    static {
        MAIN_ENGINE = new Engine(MAIN_ENGINE_NAME);
        engineMap.put(MAIN_ENGINE_NAME, MAIN_ENGINE);
    }

    private String name;
    private boolean devMode = false;
    private EngineConfig config = new EngineConfig();
    private ISourceFactory sourceFactory = config.getSourceFactory();

    private Map<String, Template> templateCache = new SyncWriteMap<String, Template>(2048, 0.5F);

    
    /**
     * <p>Constructor for Engine.</p>
     */
    public Engine() {
        this.name = "NO_NAME";
    }

    
    /**
     * <p>Constructor for Engine.</p>
     *
     * @param engineName a {@link java.lang.String} object.
     */
    public Engine(String engineName) {
        this.name = engineName;
    }

    
    /**
     * <p>use.</p>
     *
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public static Engine use() {
        return MAIN_ENGINE;
    }

    
    /**
     * <p>use.</p>
     *
     * @param engineName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public static Engine use(String engineName) {
        return engineMap.get(engineName);
    }

    
    /**
     * <p>create.</p>
     *
     * @param engineName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public synchronized static Engine create(String engineName) {
        if (StrKit.isBlank(engineName)) {
            throw new IllegalArgumentException("Engine name can not be blank");
        }
        engineName = engineName.trim();
        if (engineMap.containsKey(engineName)) {
            throw new IllegalArgumentException("Engine already exists : " + engineName);
        }
        Engine newEngine = new Engine(engineName);
        engineMap.put(engineName, newEngine);
        return newEngine;
    }

    
    /**
     * <p>remove.</p>
     *
     * @param engineName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public synchronized static Engine remove(String engineName) {
        Engine removed = engineMap.remove(engineName);
        if (removed != null && MAIN_ENGINE_NAME.equals(removed.name)) {
            Engine.MAIN_ENGINE = null;
        }
        return removed;
    }

    
    /**
     * <p>setMainEngine.</p>
     *
     * @param engine a {@link space.yizhu.record.template.Engine} object.
     */
    public synchronized static void setMainEngine(Engine engine) {
        if (engine == null) {
            throw new IllegalArgumentException("Engine can not be null");
        }
        engine.name = Engine.MAIN_ENGINE_NAME;
        engineMap.put(Engine.MAIN_ENGINE_NAME, engine);
        Engine.MAIN_ENGINE = engine;
    }

    
    /**
     * <p>getTemplate.</p>
     *
     * @param fileName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Template} object.
     */
    public Template getTemplate(String fileName) {
        if (fileName.charAt(0) != '/') {
            char[] arr = new char[fileName.length() + 1];
            fileName.getChars(0, fileName.length(), arr, 1);
            arr[0] = '/';
            fileName = new String(arr);
        }

        Template template = templateCache.get(fileName);
        if (template == null) {
            template = buildTemplateBySourceFactory(fileName);
            templateCache.put(fileName, template);
        } else if (devMode) {
            if (template.isModified()) {
                template = buildTemplateBySourceFactory(fileName);
                templateCache.put(fileName, template);
            }
        }
        return template;
    }

    private Template buildTemplateBySourceFactory(String fileName) {
        
        ISource source = sourceFactory.getSource(config.getBaseTemplatePath(), fileName, config.getEncoding());
        Env env = new Env(config);
        Parser parser = new Parser(env, source.getContent(), fileName);
        if (devMode) {
            env.addSource(source);
        }
        Stat stat = parser.parse();
        Template template = new Template(env, stat);
        return template;
    }

    
    /**
     * <p>getTemplateByString.</p>
     *
     * @param content a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Template} object.
     */
    public Template getTemplateByString(String content) {
        return getTemplateByString(content, false);
    }

    
    /**
     * <p>getTemplateByString.</p>
     *
     * @param content a {@link java.lang.String} object.
     * @param cache a boolean.
     * @return a {@link space.yizhu.record.template.Template} object.
     */
    public Template getTemplateByString(String content, boolean cache) {
        if (!cache) {
            return buildTemplateBySource(new StringSource(content, cache));
        }

        String cacheKey = HashKit.md5(content);
        Template template = templateCache.get(cacheKey);
        if (template == null) {
            template = buildTemplateBySource(new StringSource(content, cache));
            templateCache.put(cacheKey, template);
        } else if (devMode) {
            if (template.isModified()) {
                template = buildTemplateBySource(new StringSource(content, cache));
                templateCache.put(cacheKey, template);
            }
        }
        return template;
    }

    
    /**
     * <p>getTemplate.</p>
     *
     * @param source a {@link space.yizhu.record.template.source.ISource} object.
     * @return a {@link space.yizhu.record.template.Template} object.
     */
    public Template getTemplate(ISource source) {
        String cacheKey = source.getCacheKey();
        if (cacheKey == null) {    
            return buildTemplateBySource(source);
        }

        Template template = templateCache.get(cacheKey);
        if (template == null) {
            template = buildTemplateBySource(source);
            templateCache.put(cacheKey, template);
        } else if (devMode) {
            if (template.isModified()) {
                template = buildTemplateBySource(source);
                templateCache.put(cacheKey, template);
            }
        }
        return template;
    }

    private Template buildTemplateBySource(ISource source) {
        Env env = new Env(config);
        Parser parser = new Parser(env, source.getContent(), null);
        if (devMode) {
            env.addSource(source);
        }
        Stat stat = parser.parse();
        Template template = new Template(env, stat);
        return template;
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param fileName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedFunction(String fileName) {
        config.addSharedFunction(fileName);
        return this;
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param source a {@link space.yizhu.record.template.source.ISource} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedFunction(ISource source) {
        config.addSharedFunction(source);
        return this;
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param fileNames a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedFunction(String... fileNames) {
        config.addSharedFunction(fileNames);
        return this;
    }

    
    /**
     * <p>addSharedFunctionByString.</p>
     *
     * @param content a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedFunctionByString(String content) {
        config.addSharedFunctionByString(content);
        return this;
    }

    
    /**
     * <p>addSharedObject.</p>
     *
     * @param name a {@link java.lang.String} object.
     * @param object a {@link java.lang.Object} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedObject(String name, Object object) {
        config.addSharedObject(name, object);
        return this;
    }

    
    /**
     * <p>setOutputDirectiveFactory.</p>
     *
     * @param outputDirectiveFactory a {@link space.yizhu.record.template.stat.OutputDirectiveFactory} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setOutputDirectiveFactory(OutputDirectiveFactory outputDirectiveFactory) {
        config.setOutputDirectiveFactory(outputDirectiveFactory);
        return this;
    }

    
    /**
     * <p>addDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @param directiveClass a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addDirective(String directiveName, Class<? extends Directive> directiveClass) {
        config.addDirective(directiveName, directiveClass);
        return this;
    }

    
    /**
     * <p>addDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @param directive a {@link space.yizhu.record.template.Directive} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    @Deprecated
    public Engine addDirective(String directiveName, Directive directive) {
        return addDirective(directiveName, directive.getClass());
    }

    
    /**
     * <p>removeDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine removeDirective(String directiveName) {
        config.removeDirective(directiveName);
        return this;
    }

    
    /**
     * <p>addSharedMethod.</p>
     *
     * @param sharedMethodFromObject a {@link java.lang.Object} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedMethod(Object sharedMethodFromObject) {
        config.addSharedMethod(sharedMethodFromObject);
        return this;
    }

    
    /**
     * <p>addSharedMethod.</p>
     *
     * @param sharedMethodFromClass a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedMethod(Class<?> sharedMethodFromClass) {
        config.addSharedMethod(sharedMethodFromClass);
        return this;
    }

    
    /**
     * <p>addSharedStaticMethod.</p>
     *
     * @param sharedStaticMethodFromClass a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine addSharedStaticMethod(Class<?> sharedStaticMethodFromClass) {
        config.addSharedStaticMethod(sharedStaticMethodFromClass);
        return this;
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param methodName a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine removeSharedMethod(String methodName) {
        config.removeSharedMethod(methodName);
        return this;
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param clazz a {@link java.lang.Class} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine removeSharedMethod(Class<?> clazz) {
        config.removeSharedMethod(clazz);
        return this;
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param method a {@link java.lang.reflect.Method} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine removeSharedMethod(Method method) {
        config.removeSharedMethod(method);
        return this;
    }

    
    /**
     * <p>removeTemplateCache.</p>
     *
     * @param cacheKey a {@link java.lang.String} object.
     */
    public void removeTemplateCache(String cacheKey) {
        templateCache.remove(cacheKey);
    }

    
    /**
     * <p>removeAllTemplateCache.</p>
     */
    public void removeAllTemplateCache() {
        templateCache.clear();
    }

    /**
     * <p>getTemplateCacheSize.</p>
     *
     * @return a int.
     */
    public int getTemplateCacheSize() {
        return templateCache.size();
    }

    /**
     * <p>Getter for the field <code>name</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getName() {
        return name;
    }

    /**
     * <p>toString.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString() {
        return "Template Engine: " + name;
    }

    

    /**
     * <p>getEngineConfig.</p>
     *
     * @return a {@link space.yizhu.record.template.EngineConfig} object.
     */
    public EngineConfig getEngineConfig() {
        return config;
    }

    
    /**
     * <p>Setter for the field <code>devMode</code>.</p>
     *
     * @param devMode a boolean.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setDevMode(boolean devMode) {
        this.devMode = devMode;
        this.config.setDevMode(devMode);
        if (this.devMode) {
            removeAllTemplateCache();
        }
        return this;
    }

    /**
     * <p>Getter for the field <code>devMode</code>.</p>
     *
     * @return a boolean.
     */
    public boolean getDevMode() {
        return devMode;
    }

    
    /**
     * <p>Setter for the field <code>sourceFactory</code>.</p>
     *
     * @param sourceFactory a {@link space.yizhu.record.template.source.ISourceFactory} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setSourceFactory(ISourceFactory sourceFactory) {
        this.config.setSourceFactory(sourceFactory);    
        this.sourceFactory = sourceFactory;
        return this;
    }

    
    /**
     * <p>setToClassPathSourceFactory.</p>
     *
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setToClassPathSourceFactory() {
        return setSourceFactory(new ClassPathSourceFactory());
    }

    /**
     * <p>Getter for the field <code>sourceFactory</code>.</p>
     *
     * @return a {@link space.yizhu.record.template.source.ISourceFactory} object.
     */
    public ISourceFactory getSourceFactory() {
        return sourceFactory;
    }

    /**
     * <p>setBaseTemplatePath.</p>
     *
     * @param baseTemplatePath a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setBaseTemplatePath(String baseTemplatePath) {
        config.setBaseTemplatePath(baseTemplatePath);
        return this;
    }

    /**
     * <p>getBaseTemplatePath.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getBaseTemplatePath() {
        return config.getBaseTemplatePath();
    }

    /**
     * <p>setDatePattern.</p>
     *
     * @param datePattern a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setDatePattern(String datePattern) {
        config.setDatePattern(datePattern);
        return this;
    }

    /**
     * <p>getDatePattern.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getDatePattern() {
        return config.getDatePattern();
    }

    /**
     * <p>setEncoding.</p>
     *
     * @param encoding a {@link java.lang.String} object.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setEncoding(String encoding) {
        config.setEncoding(encoding);
        return this;
    }

    /**
     * <p>getEncoding.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getEncoding() {
        return config.getEncoding();
    }

    /**
     * <p>setWriterBufferSize.</p>
     *
     * @param bufferSize a int.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setWriterBufferSize(int bufferSize) {
        config.setWriterBufferSize(bufferSize);
        return this;
    }

    
    /**
     * <p>setReloadModifiedSharedFunctionInDevMode.</p>
     *
     * @param reloadModifiedSharedFunctionInDevMode a boolean.
     * @return a {@link space.yizhu.record.template.Engine} object.
     */
    public Engine setReloadModifiedSharedFunctionInDevMode(boolean reloadModifiedSharedFunctionInDevMode) {
        config.setReloadModifiedSharedFunctionInDevMode(reloadModifiedSharedFunctionInDevMode);
        return this;
    }

    /**
     * <p>addExtensionMethod.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param objectOfExtensionClass a {@link java.lang.Object} object.
     */
    public static void addExtensionMethod(Class<?> targetClass, Object objectOfExtensionClass) {
        MethodKit.addExtensionMethod(targetClass, objectOfExtensionClass);
    }

    /**
     * <p>addExtensionMethod.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param extensionClass a {@link java.lang.Class} object.
     */
    public static void addExtensionMethod(Class<?> targetClass, Class<?> extensionClass) {
        MethodKit.addExtensionMethod(targetClass, extensionClass);
    }

    /**
     * <p>removeExtensionMethod.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param objectOfExtensionClass a {@link java.lang.Object} object.
     */
    public static void removeExtensionMethod(Class<?> targetClass, Object objectOfExtensionClass) {
        MethodKit.removeExtensionMethod(targetClass, objectOfExtensionClass);
    }

    /**
     * <p>removeExtensionMethod.</p>
     *
     * @param targetClass a {@link java.lang.Class} object.
     * @param extensionClass a {@link java.lang.Class} object.
     */
    public static void removeExtensionMethod(Class<?> targetClass, Class<?> extensionClass) {
        MethodKit.removeExtensionMethod(targetClass, extensionClass);
    }

    
    /**
     * <p>addFieldGetter.</p>
     *
     * @param index a int.
     * @param fieldGetter a {@link space.yizhu.record.template.expr.ast.FieldGetter} object.
     */
    public static void addFieldGetter(int index, FieldGetter fieldGetter) {
        FieldKit.addFieldGetter(index, fieldGetter);
    }

    /**
     * <p>addFieldGetterToLast.</p>
     *
     * @param fieldGetter a {@link space.yizhu.record.template.expr.ast.FieldGetter} object.
     */
    public static void addFieldGetterToLast(FieldGetter fieldGetter) {
        FieldKit.addFieldGetterToLast(fieldGetter);
    }

    /**
     * <p>addFieldGetterToFirst.</p>
     *
     * @param fieldGetter a {@link space.yizhu.record.template.expr.ast.FieldGetter} object.
     */
    public static void addFieldGetterToFirst(FieldGetter fieldGetter) {
        FieldKit.addFieldGetterToFirst(fieldGetter);
    }

    /**
     * <p>removeFieldGetter.</p>
     *
     * @param fieldGetterClass a {@link java.lang.Class} object.
     */
    public static void removeFieldGetter(Class<? extends FieldGetter> fieldGetterClass) {
        FieldKit.removeFieldGetter(fieldGetterClass);
    }

    /**
     * <p>setToFastFieldKeyBuilder.</p>
     */
    public static void setToFastFieldKeyBuilder() {
        FieldKeyBuilder.setToFastFieldKeyBuilder();
    }
}





