

package space.yizhu.record.template;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import space.yizhu.kits.StrKit;
import space.yizhu.record.template.expr.ast.ExprList;
import space.yizhu.record.template.expr.ast.SharedMethodKit;
import space.yizhu.record.template.ext.directive.*;
import space.yizhu.record.template.ext.sharedmethod.SharedMethodLib;
import space.yizhu.record.template.io.EncoderFactory;
import space.yizhu.record.template.io.WriterBuffer;
import space.yizhu.record.template.source.FileSource;
import space.yizhu.record.template.source.FileSourceFactory;
import space.yizhu.record.template.source.ISource;
import space.yizhu.record.template.source.ISourceFactory;
import space.yizhu.record.template.source.StringSource;
import space.yizhu.record.template.stat.Location;
import space.yizhu.record.template.stat.OutputDirectiveFactory;
import space.yizhu.record.template.stat.Parser;
import space.yizhu.record.template.stat.ast.Define;
import space.yizhu.record.template.stat.ast.Output;


/**
 * <p>EngineConfig class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class EngineConfig {

    /** Constant <code>DEFAULT_ENCODING="UTF-8"</code> */
    public static final String DEFAULT_ENCODING = "UTF-8";

    WriterBuffer writerBuffer = new WriterBuffer();

    private Map<String, Define> sharedFunctionMap = createSharedFunctionMap();        
    private List<ISource> sharedFunctionSourceList = new ArrayList<ISource>();        

    Map<String, Object> sharedObjectMap = null;

    private OutputDirectiveFactory outputDirectiveFactory = OutputDirectiveFactory.me;
    private ISourceFactory sourceFactory = new FileSourceFactory();
    private Map<String, Class<? extends Directive>> directiveMap = new HashMap<String, Class<? extends Directive>>(64, 0.5F);
    private SharedMethodKit sharedMethodKit = new SharedMethodKit();

    private boolean devMode = false;
    private boolean reloadModifiedSharedFunctionInDevMode = true;
    private String baseTemplatePath = null;
    private String encoding = DEFAULT_ENCODING;
    private String datePattern = "yyyy-MM-dd HH:mm";

    /**
     * <p>Constructor for EngineConfig.</p>
     */
    public EngineConfig() {
        
        addDirective("render", RenderDirective.class);
        addDirective("date", DateDirective.class);
        addDirective("escape", EscapeDirective.class);
        addDirective("string", StringDirective.class);
        addDirective("random", RandomDirective.class);
        addDirective("number", NumberDirective.class);
        addDirective("call", CallDirective.class);

        
        addSharedMethod(new SharedMethodLib());
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param fileName a {@link java.lang.String} object.
     */
    public void addSharedFunction(String fileName) {
        fileName = fileName.replace("\\", "/");
        
        ISource source = sourceFactory.getSource(baseTemplatePath, fileName, encoding);
        doAddSharedFunction(source, fileName);
    }

    private synchronized void doAddSharedFunction(ISource source, String fileName) {
        Env env = new Env(this);
        new Parser(env, source.getContent(), fileName).parse();
        addToSharedFunctionMap(sharedFunctionMap, env);
        if (devMode) {
            sharedFunctionSourceList.add(source);
            env.addSource(source);
        }
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param fileNames a {@link java.lang.String} object.
     */
    public void addSharedFunction(String... fileNames) {
        for (String fileName : fileNames) {
            addSharedFunction(fileName);
        }
    }

    
    /**
     * <p>addSharedFunctionByString.</p>
     *
     * @param content a {@link java.lang.String} object.
     */
    public void addSharedFunctionByString(String content) {
        
        
        StringSource stringSource = new StringSource(content, false);
        doAddSharedFunction(stringSource, null);
    }

    
    /**
     * <p>addSharedFunction.</p>
     *
     * @param source a {@link space.yizhu.record.template.source.ISource} object.
     */
    public void addSharedFunction(ISource source) {
        String fileName = source instanceof FileSource ? ((FileSource) source).getFileName() : null;
        doAddSharedFunction(source, fileName);
    }

    private void addToSharedFunctionMap(Map<String, Define> sharedFunctionMap, Env env) {
        Map<String, Define> funcMap = env.getFunctionMap();
        for (Entry<String, Define> e : funcMap.entrySet()) {
            if (sharedFunctionMap.containsKey(e.getKey())) {
                throw new IllegalArgumentException("Template function already exists : " + e.getKey());
            }
            Define func = e.getValue();
            if (devMode) {
                func.setEnvForDevMode(env);
            }
            sharedFunctionMap.put(e.getKey(), func);
        }
    }

    
    Define getSharedFunction(String functionName) {
        Define func = sharedFunctionMap.get(functionName);
        if (func == null) {
            
            return null;
        }

        if (devMode && reloadModifiedSharedFunctionInDevMode) {
            if (func.isSourceModifiedForDevMode()) {
                synchronized (this) {
                    func = sharedFunctionMap.get(functionName);
                    if (func.isSourceModifiedForDevMode()) {
                        reloadSharedFunctionSourceList();
                        func = sharedFunctionMap.get(functionName);
                    }
                }
            }
        }
        return func;
    }

    
    private synchronized void reloadSharedFunctionSourceList() {
        Map<String, Define> newMap = createSharedFunctionMap();
        for (int i = 0, size = sharedFunctionSourceList.size(); i < size; i++) {
            ISource source = sharedFunctionSourceList.get(i);
            String fileName = source instanceof FileSource ? ((FileSource) source).getFileName() : null;

            Env env = new Env(this);
            new Parser(env, source.getContent(), fileName).parse();
            addToSharedFunctionMap(newMap, env);
            if (devMode) {
                env.addSource(source);
            }
        }
        this.sharedFunctionMap = newMap;
    }

    private Map<String, Define> createSharedFunctionMap() {
        return new HashMap<String, Define>(512, 0.25F);
    }

    /**
     * <p>addSharedObject.</p>
     *
     * @param name a {@link java.lang.String} object.
     * @param object a {@link java.lang.Object} object.
     */
    public synchronized void addSharedObject(String name, Object object) {
        if (sharedObjectMap == null) {
            sharedObjectMap = new HashMap<String, Object>(64, 0.25F);
        } else if (sharedObjectMap.containsKey(name)) {
            throw new IllegalArgumentException("Shared object already exists: " + name);
        }
        sharedObjectMap.put(name, object);
    }

    Map<String, Object> getSharedObjectMap() {
        return sharedObjectMap;
    }

    
    /**
     * <p>Setter for the field <code>outputDirectiveFactory</code>.</p>
     *
     * @param outputDirectiveFactory a {@link space.yizhu.record.template.stat.OutputDirectiveFactory} object.
     */
    public void setOutputDirectiveFactory(OutputDirectiveFactory outputDirectiveFactory) {
        if (outputDirectiveFactory == null) {
            throw new IllegalArgumentException("outputDirectiveFactory can not be null");
        }
        this.outputDirectiveFactory = outputDirectiveFactory;
    }

    /**
     * <p>getOutputDirective.</p>
     *
     * @param exprList a {@link space.yizhu.record.template.expr.ast.ExprList} object.
     * @param location a {@link space.yizhu.record.template.stat.Location} object.
     * @return a {@link space.yizhu.record.template.stat.ast.Output} object.
     */
    public Output getOutputDirective(ExprList exprList, Location location) {
        return outputDirectiveFactory.getOutputDirective(exprList, location);
    }

    
    void setDevMode(boolean devMode) {
        this.devMode = devMode;
    }

    /**
     * <p>isDevMode.</p>
     *
     * @return a boolean.
     */
    public boolean isDevMode() {
        return devMode;
    }

    
    void setSourceFactory(ISourceFactory sourceFactory) {
        if (sourceFactory == null) {
            throw new IllegalArgumentException("sourceFactory can not be null");
        }
        this.sourceFactory = sourceFactory;
    }

    /**
     * <p>Getter for the field <code>sourceFactory</code>.</p>
     *
     * @return a {@link space.yizhu.record.template.source.ISourceFactory} object.
     */
    public ISourceFactory getSourceFactory() {
        return sourceFactory;
    }

    /**
     * <p>Setter for the field <code>baseTemplatePath</code>.</p>
     *
     * @param baseTemplatePath a {@link java.lang.String} object.
     */
    public void setBaseTemplatePath(String baseTemplatePath) {
        
        if (baseTemplatePath == null) {
            this.baseTemplatePath = null;
            return;
        }
        if (StrKit.isBlank(baseTemplatePath)) {
            throw new IllegalArgumentException("baseTemplatePath can not be blank");
        }
        baseTemplatePath = baseTemplatePath.trim();
        baseTemplatePath = baseTemplatePath.replace("\\", "/");
        if (baseTemplatePath.length() > 1) {
            if (baseTemplatePath.endsWith("/")) {
                baseTemplatePath = baseTemplatePath.substring(0, baseTemplatePath.length() - 1);
            }
        }
        this.baseTemplatePath = baseTemplatePath;
    }

    /**
     * <p>Getter for the field <code>baseTemplatePath</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getBaseTemplatePath() {
        return baseTemplatePath;
    }

    /**
     * <p>Setter for the field <code>encoding</code>.</p>
     *
     * @param encoding a {@link java.lang.String} object.
     */
    public void setEncoding(String encoding) {
        if (StrKit.isBlank(encoding)) {
            throw new IllegalArgumentException("encoding can not be blank");
        }
        this.encoding = encoding;

        writerBuffer.setEncoding(encoding);        
    }

    /**
     * <p>setEncoderFactory.</p>
     *
     * @param encoderFactory a {@link space.yizhu.record.template.io.EncoderFactory} object.
     */
    public void setEncoderFactory(EncoderFactory encoderFactory) {
        writerBuffer.setEncoderFactory(encoderFactory);
        writerBuffer.setEncoding(encoding);        
    }

    /**
     * <p>setWriterBufferSize.</p>
     *
     * @param bufferSize a int.
     */
    public void setWriterBufferSize(int bufferSize) {
        writerBuffer.setBufferSize(bufferSize);
    }

    /**
     * <p>Getter for the field <code>encoding</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * <p>Setter for the field <code>datePattern</code>.</p>
     *
     * @param datePattern a {@link java.lang.String} object.
     */
    public void setDatePattern(String datePattern) {
        if (StrKit.isBlank(datePattern)) {
            throw new IllegalArgumentException("datePattern can not be blank");
        }
        this.datePattern = datePattern;
    }

    /**
     * <p>Getter for the field <code>datePattern</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getDatePattern() {
        return datePattern;
    }

    /**
     * <p>Setter for the field <code>reloadModifiedSharedFunctionInDevMode</code>.</p>
     *
     * @param reloadModifiedSharedFunctionInDevMode a boolean.
     */
    public void setReloadModifiedSharedFunctionInDevMode(boolean reloadModifiedSharedFunctionInDevMode) {
        this.reloadModifiedSharedFunctionInDevMode = reloadModifiedSharedFunctionInDevMode;
    }

    /**
     * <p>addDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @param directive a {@link space.yizhu.record.template.Directive} object.
     */
    @Deprecated
    public void addDirective(String directiveName, Directive directive) {
        addDirective(directiveName, directive.getClass());
    }

    /**
     * <p>addDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @param directiveClass a {@link java.lang.Class} object.
     */
    public synchronized void addDirective(String directiveName, Class<? extends Directive> directiveClass) {
        if (StrKit.isBlank(directiveName)) {
            throw new IllegalArgumentException("directive name can not be blank");
        }
        if (directiveClass == null) {
            throw new IllegalArgumentException("directiveClass can not be null");
        }
        if (directiveMap.containsKey(directiveName)) {
            throw new IllegalArgumentException("directive already exists : " + directiveName);
        }
        directiveMap.put(directiveName, directiveClass);
    }

    /**
     * <p>getDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     * @return a {@link java.lang.Class} object.
     */
    public Class<? extends Directive> getDirective(String directiveName) {
        return directiveMap.get(directiveName);
    }

    /**
     * <p>removeDirective.</p>
     *
     * @param directiveName a {@link java.lang.String} object.
     */
    public void removeDirective(String directiveName) {
        directiveMap.remove(directiveName);
    }

    
    /**
     * <p>addSharedMethod.</p>
     *
     * @param sharedMethodFromObject a {@link java.lang.Object} object.
     */
    public void addSharedMethod(Object sharedMethodFromObject) {
        sharedMethodKit.addSharedMethod(sharedMethodFromObject);
    }

    
    /**
     * <p>addSharedMethod.</p>
     *
     * @param sharedMethodFromClass a {@link java.lang.Class} object.
     */
    public void addSharedMethod(Class<?> sharedMethodFromClass) {
        sharedMethodKit.addSharedMethod(sharedMethodFromClass);
    }

    
    /**
     * <p>addSharedStaticMethod.</p>
     *
     * @param sharedStaticMethodFromClass a {@link java.lang.Class} object.
     */
    public void addSharedStaticMethod(Class<?> sharedStaticMethodFromClass) {
        sharedMethodKit.addSharedStaticMethod(sharedStaticMethodFromClass);
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param methodName a {@link java.lang.String} object.
     */
    public void removeSharedMethod(String methodName) {
        sharedMethodKit.removeSharedMethod(methodName);
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param sharedClass a {@link java.lang.Class} object.
     */
    public void removeSharedMethod(Class<?> sharedClass) {
        sharedMethodKit.removeSharedMethod(sharedClass);
    }

    
    /**
     * <p>removeSharedMethod.</p>
     *
     * @param method a {@link java.lang.reflect.Method} object.
     */
    public void removeSharedMethod(Method method) {
        sharedMethodKit.removeSharedMethod(method);
    }

    /**
     * <p>Getter for the field <code>sharedMethodKit</code>.</p>
     *
     * @return a {@link space.yizhu.record.template.expr.ast.SharedMethodKit} object.
     */
    public SharedMethodKit getSharedMethodKit() {
        return sharedMethodKit;
    }
}





