

package space.yizhu.record.template;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.Map;

import space.yizhu.record.template.io.ByteWriter;
import space.yizhu.record.template.io.CharWriter;
import space.yizhu.record.template.io.FastStringWriter;
import space.yizhu.record.template.stat.Scope;
import space.yizhu.record.template.stat.ast.Stat;


/**
 * <p>Template class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class Template {

    private Env env;
    private Stat ast;

    /**
     * <p>Constructor for Template.</p>
     *
     * @param env a {@link space.yizhu.record.template.Env} object.
     * @param ast a {@link space.yizhu.record.template.stat.ast.Stat} object.
     */
    public Template(Env env, Stat ast) {
        if (env == null || ast == null) {
            throw new IllegalArgumentException("env and ast can not be null");
        }
        this.env = env;
        this.ast = ast;
    }

    
    /**
     * <p>render.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @param outputStream a {@link java.io.OutputStream} object.
     */
    public void render(Map<?, ?> data, OutputStream outputStream) {
        ByteWriter byteWriter = env.engineConfig.writerBuffer.getByteWriter(outputStream);
        try {
            ast.exec(env, new Scope(data, env.engineConfig.sharedObjectMap), byteWriter);
        } finally {
            byteWriter.close();
        }
    }

    
    /**
     * <p>render.</p>
     *
     * @param outputStream a {@link java.io.OutputStream} object.
     */
    public void render(OutputStream outputStream) {
        render(null, outputStream);
    }

    
    /**
     * <p>render.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @param writer a {@link java.io.Writer} object.
     */
    public void render(Map<?, ?> data, Writer writer) {
        CharWriter charWriter = env.engineConfig.writerBuffer.getCharWriter(writer);
        try {
            ast.exec(env, new Scope(data, env.engineConfig.sharedObjectMap), charWriter);
        } finally {
            charWriter.close();
        }
    }

    
    /**
     * <p>render.</p>
     *
     * @param writer a {@link java.io.Writer} object.
     */
    public void render(Writer writer) {
        render(null, writer);
    }

    
    /**
     * <p>renderToString.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @return a {@link java.lang.String} object.
     */
    public String renderToString(Map<?, ?> data) {
        FastStringWriter fsw = env.engineConfig.writerBuffer.getFastStringWriter();
        try {
            render(data, fsw);
            return fsw.toString();
        } finally {
            fsw.close();
        }
    }

    
    /**
     * <p>renderToStringBuilder.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @return a {@link java.lang.StringBuilder} object.
     */
    public StringBuilder renderToStringBuilder(Map<?, ?> data) {
        FastStringWriter fsw = new FastStringWriter();
        render(data, fsw);
        return fsw.getBuffer();
    }

    
    /**
     * <p>render.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @param file a {@link java.io.File} object.
     */
    public void render(Map<?, ?> data, File file) {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(file);
            render(data, fos);
        } catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        } finally {
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    e.printStackTrace(System.err);
                }
            }
        }
    }

    
    /**
     * <p>render.</p>
     *
     * @param data a {@link java.util.Map} object.
     * @param fileName a {@link java.lang.String} object.
     */
    public void render(Map<?, ?> data, String fileName) {
        render(data, new File(fileName));
    }

    /**
     * <p>isModified.</p>
     *
     * @return a boolean.
     */
    public boolean isModified() {
        return env.isSourceListModified();
    }
}





