

package space.yizhu.record.template.expr.ast;

import java.lang.reflect.InvocationTargetException;

import space.yizhu.record.template.TemplateException;
import space.yizhu.record.template.stat.Location;
import space.yizhu.record.template.stat.ParseException;
import space.yizhu.record.template.stat.Scope;
import space.yizhu.record.template.stat.Location;
import space.yizhu.record.template.stat.ParseException;
import space.yizhu.record.template.stat.Scope;


/**
 * <p>Method class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class Method extends Expr {

    private Expr expr;
    private String methodName;
    private ExprList exprList;

    /**
     * <p>Constructor for Method.</p>
     *
     * @param expr a {@link space.yizhu.record.template.expr.ast.Expr} object.
     * @param methodName a {@link java.lang.String} object.
     * @param exprList a {@link space.yizhu.record.template.expr.ast.ExprList} object.
     * @param location a {@link space.yizhu.record.template.stat.Location} object.
     */
    public Method(Expr expr, String methodName, ExprList exprList, Location location) {
        if (exprList == null || exprList.length() == 0) {
            throw new ParseException("The parameter of method can not be blank", location);
        }
        init(expr, methodName, exprList, location);
    }

    /**
     * <p>Constructor for Method.</p>
     *
     * @param expr a {@link space.yizhu.record.template.expr.ast.Expr} object.
     * @param methodName a {@link java.lang.String} object.
     * @param location a {@link space.yizhu.record.template.stat.Location} object.
     */
    public Method(Expr expr, String methodName, Location location) {
        init(expr, methodName, ExprList.NULL_EXPR_LIST, location);
    }

    private void init(Expr expr, String methodName, ExprList exprList, Location location) {
        if (expr == null) {
            throw new ParseException("The target for method invoking can not be blank", location);
        }
        if (MethodKit.isForbiddenMethod(methodName)) {
            throw new ParseException("Forbidden method: " + methodName, location);
        }
        this.expr = expr;
        this.methodName = methodName;
        this.exprList = exprList;
        this.location = location;
    }

    /** {@inheritDoc} */
    public Object eval(Scope scope) {
        Object target = expr.eval(scope);
        if (target == null) {
            if (scope.getCtrl().isNullSafe()) {
                return null;
            }
            throw new TemplateException("The target for method invoking can not be null, method name: " + methodName, location);
        }

        Object[] argValues = exprList.evalExprList(scope);
        try {

            MethodInfo methodInfo = MethodKit.getMethod(target.getClass(), methodName, argValues);
            if (methodInfo != null) {
                return methodInfo.invoke(target, argValues);
            }

            if (scope.getCtrl().isNullSafe()) {
                return null;
            }
            throw new TemplateException(buildMethodNotFoundSignature("public method not found: " + target.getClass().getName() + ".", methodName, argValues), location);

        } catch (TemplateException | ParseException e) {
            throw e;
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t == null) {
                t = e;
            }
            throw new TemplateException(t.getMessage(), location, t);
        } catch (Exception e) {
            throw new TemplateException(e.getMessage(), location, e);
        }
    }

    static String buildMethodNotFoundSignature(String preMsg, String methodName, Object[] argValues) {
        StringBuilder ret = new StringBuilder().append(preMsg).append(methodName).append("(");
        if (argValues != null) {
            for (int i = 0; i < argValues.length; i++) {
                if (i > 0) {
                    ret.append(", ");
                }
                ret.append(argValues[i] != null ? argValues[i].getClass().getName() : "null");
            }
        }
        return ret.append(")").toString();
    }
	
	
}





