

package space.yizhu.record.template.io;

import java.io.IOException;
import java.io.OutputStream;


/**
 * <p>ByteWriter class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class ByteWriter extends Writer {

    OutputStream out;
    Encoder encoder;

    char[] chars;
    byte[] bytes;

    /**
     * <p>Constructor for ByteWriter.</p>
     *
     * @param encoder a {@link space.yizhu.record.template.io.Encoder} object.
     * @param bufferSize a int.
     */
    public ByteWriter(Encoder encoder, int bufferSize) {
        this.encoder = encoder;
        this.chars = new char[bufferSize];
        this.bytes = new byte[bufferSize * ((int) encoder.maxBytesPerChar())];
    }

    /**
     * <p>init.</p>
     *
     * @param outputStream a {@link java.io.OutputStream} object.
     * @return a {@link space.yizhu.record.template.io.ByteWriter} object.
     */
    public ByteWriter init(OutputStream outputStream) {
        this.out = outputStream;
        return this;
    }

    /**
     * <p>flush.</p>
     *
     * @throws java.io.IOException if any.
     */
    public void flush() throws IOException {
        out.flush();
    }

    /**
     * <p>close.</p>
     */
    public void close() {
        out = null;
    }

    /** {@inheritDoc} */
    public void write(String str, int offset, int len) throws IOException {
        while (len > chars.length) {
            write(str, offset, chars.length);
            offset += chars.length;
            len -= chars.length;
        }

        str.getChars(offset, offset + len, chars, 0);
        int byteLen = encoder.encode(chars, 0, len, bytes);
        out.write(bytes, 0, byteLen);
    }

    /**
     * <p>write.</p>
     *
     * @param str a {@link java.lang.String} object.
     * @throws java.io.IOException if any.
     */
    public void write(String str) throws IOException {
        write(str, 0, str.length());
    }

    /** {@inheritDoc} */
    public void write(StringBuilder stringBuilder, int offset, int len) throws IOException {
        while (len > chars.length) {
            write(stringBuilder, offset, chars.length);
            offset += chars.length;
            len -= chars.length;
        }

        stringBuilder.getChars(offset, offset + len, chars, 0);
        int byteLen = encoder.encode(chars, 0, len, bytes);
        out.write(bytes, 0, byteLen);
    }

    /**
     * <p>write.</p>
     *
     * @param stringBuilder a {@link java.lang.StringBuilder} object.
     * @throws java.io.IOException if any.
     */
    public void write(StringBuilder stringBuilder) throws IOException {
        write(stringBuilder, 0, stringBuilder.length());
    }

    /**
     * <p>write.</p>
     *
     * @param writable a {@link space.yizhu.record.template.io.IWritable} object.
     * @throws java.io.IOException if any.
     */
    public void write(IWritable writable) throws IOException {
        byte[] data = writable.getBytes();
        out.write(data, 0, data.length);
    }

    /**
     * <p>write.</p>
     *
     * @param intValue a int.
     * @throws java.io.IOException if any.
     */
    public void write(int intValue) throws IOException {
        IntegerWriter.write(this, intValue);
    }

    /**
     * <p>write.</p>
     *
     * @param longValue a long.
     * @throws java.io.IOException if any.
     */
    public void write(long longValue) throws IOException {
        LongWriter.write(this, longValue);
    }

    /** {@inheritDoc} */
    public void write(double doubleValue) throws IOException {
        FloatingWriter.write(this, doubleValue);
    }

    /**
     * <p>write.</p>
     *
     * @param floatValue a float.
     * @throws java.io.IOException if any.
     */
    public void write(float floatValue) throws IOException {
        FloatingWriter.write(this, floatValue);
    }

    private static final byte[] TRUE_BYTES = "true".getBytes();
    private static final byte[] FALSE_BYTES = "false".getBytes();

    /**
     * <p>write.</p>
     *
     * @param booleanValue a boolean.
     * @throws java.io.IOException if any.
     */
    public void write(boolean booleanValue) throws IOException {
        out.write(booleanValue ? TRUE_BYTES : FALSE_BYTES);
    }
}



