

package space.yizhu.record.template.source;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import space.yizhu.kits.LogKit;
import space.yizhu.record.template.EngineConfig;


/**
 * <p>FileSource class.</p>
 *
 * @author yi
 * @version $Id: $Id
 */
public class FileSource implements ISource {

    private String finalFileName;
    private String fileName;
    private String encoding;

    private long lastModified;

    /**
     * <p>Constructor for FileSource.</p>
     *
     * @param baseTemplatePath a {@link java.lang.String} object.
     * @param fileName a {@link java.lang.String} object.
     * @param encoding a {@link java.lang.String} object.
     */
    public FileSource(String baseTemplatePath, String fileName, String encoding) {
        this.finalFileName = buildFinalFileName(baseTemplatePath, fileName);
        this.fileName = fileName;
        this.encoding = encoding;
    }

    /**
     * <p>Constructor for FileSource.</p>
     *
     * @param baseTemplatePath a {@link java.lang.String} object.
     * @param fileName a {@link java.lang.String} object.
     */
    public FileSource(String baseTemplatePath, String fileName) {
        this(baseTemplatePath, fileName, EngineConfig.DEFAULT_ENCODING);
    }

    /**
     * <p>isModified.</p>
     *
     * @return a boolean.
     */
    public boolean isModified() {
        return lastModified != new File(finalFileName).lastModified();
    }

    /**
     * <p>getCacheKey.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getCacheKey() {
        return fileName;
    }

    /**
     * <p>Getter for the field <code>encoding</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getEncoding() {
        return encoding;
    }

    /**
     * <p>Getter for the field <code>finalFileName</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getFinalFileName() {
        return finalFileName;
    }

    /**
     * <p>Getter for the field <code>fileName</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getFileName() {
        return fileName;
    }

    /**
     * <p>getContent.</p>
     *
     * @return a {@link java.lang.StringBuilder} object.
     */
    public StringBuilder getContent() {
        File file = new File(finalFileName);
        if (!file.exists()) {
            throw new RuntimeException("File not found : " + finalFileName);
        }

        
        this.lastModified = file.lastModified();

        return loadFile(file, encoding);
    }

    private String buildFinalFileName(String baseTemplatePath, String fileName) {
        if (baseTemplatePath == null) {
            return fileName;
        }
        char firstChar = fileName.charAt(0);
        String finalFileName;
        if (firstChar == '/' || firstChar == '\\') {
            finalFileName = baseTemplatePath + fileName;
        } else {
            finalFileName = baseTemplatePath + File.separator + fileName;
        }
        return finalFileName;
    }

    /**
     * <p>loadFile.</p>
     *
     * @param file a {@link java.io.File} object.
     * @param encoding a {@link java.lang.String} object.
     * @return a {@link java.lang.StringBuilder} object.
     */
    public static StringBuilder loadFile(File file, String encoding) {
        StringBuilder ret = new StringBuilder((int) file.length() + 3);
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(new FileInputStream(file), encoding));
            
            String line = br.readLine();
            if (line != null) {
                ret.append(line);
            } else {
                return ret;
            }

            while ((line = br.readLine()) != null) {
                ret.append('\n').append(line);
            }
            return ret;
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                    LogKit.error(e.getMessage(), e);
                }
            }
        }
    }

    /**
     * <p>toString.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("File name: ").append(fileName).append("\n");
        sb.append("Final file name: ").append(finalFileName).append("\n");
        sb.append("Last modified: ").append(lastModified).append("\n");
        return sb.toString();
    }
}




