/*
 * Decompiled with CFR 0.152.
 */
package studio.mevera.imperat.command.tree;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import studio.mevera.imperat.command.tree.CommandTree;
import studio.mevera.imperat.command.tree.ParameterNode;
import studio.mevera.imperat.context.Source;
import studio.mevera.imperat.util.ImperatDebugger;

@ApiStatus.Internal
public final class CommandTreeVisualizer<S extends Source> {
    private static final String HORIZONTAL_LINE = "\u2500";
    private static final String VERTICAL_LINE = "\u2502";
    private static final String CORNER_TOP_LEFT = "\u250c";
    private static final String CORNER_TOP_RIGHT = "\u2510";
    private static final String CORNER_BOTTOM_LEFT = "\u2514";
    private static final String CORNER_BOTTOM_RIGHT = "\u2518";
    private static final String T_JUNCTION = "\u252c";
    private static final String T_JUNCTION_UP = "\u2534";
    private static final String T_JUNCTION_RIGHT = "\u251c";
    private static final String T_JUNCTION_LEFT = "\u2524";
    private static final String CROSS_JUNCTION = "\u253c";
    @Nullable
    private final CommandTree<S> tree;
    private final boolean useColors;
    private final boolean showNodeTypes;
    private final int minNodeWidth;
    private final int nodeSpacing;
    private static final String RESET = "\u001b[0m";
    private static final String CYAN = "\u001b[36m";
    private static final String YELLOW = "\u001b[33m";
    private static final String GREEN = "\u001b[32m";
    private static final String GRAY = "\u001b[90m";
    private static final String BOLD = "\u001b[1m";
    private static final String WHITE = "\u001b[37m";

    CommandTreeVisualizer(@Nullable CommandTree<S> tree) {
        this(tree, true, true, 12, 2);
    }

    CommandTreeVisualizer(@Nullable CommandTree<S> tree, boolean useColors, boolean showNodeTypes, int minNodeWidth, int nodeSpacing) {
        this.tree = tree;
        this.useColors = useColors;
        this.showNodeTypes = showNodeTypes;
        this.minNodeWidth = minNodeWidth;
        this.nodeSpacing = nodeSpacing;
    }

    public static <S extends Source> CommandTreeVisualizer<S> of(@Nullable CommandTree<S> tree) {
        return new CommandTreeVisualizer<S>(tree);
    }

    public void visualize() {
        if (this.tree == null || !ImperatDebugger.isEnabled()) {
            return;
        }
        TreeNode root = this.buildTreeStructure(this.tree.rootNode(), null);
        this.calculateNodePositions(root);
        String visualization = this.renderTree(root);
        ImperatDebugger.debug(visualization, new Object[0]);
    }

    private TreeNode buildTreeStructure(ParameterNode<S, ?> node, TreeNode parent) {
        if (node == null) {
            return null;
        }
        NodeType type = parent == null ? NodeType.ROOT : this.determineNodeType(node);
        TreeNode treeNode = new TreeNode(node, type);
        for (ParameterNode parameterNode : node.getChildren()) {
            TreeNode childTreeNode = this.buildTreeStructure(parameterNode, treeNode);
            if (childTreeNode == null) continue;
            treeNode.children.add(childTreeNode);
        }
        return treeNode;
    }

    private void calculateNodePositions(TreeNode root) {
        this.calculateWidths(root);
        this.assignPositions(root, 0, 0);
    }

    private int calculateWidths(TreeNode node) {
        if (node.children.isEmpty()) {
            node.width = Math.max(this.minNodeWidth, node.getDisplayText().length() + 4);
            return node.width;
        }
        int totalChildrenWidth = 0;
        for (TreeNode child : node.children) {
            totalChildrenWidth += this.calculateWidths(child);
        }
        node.width = Math.max(Math.max(this.minNodeWidth, node.getDisplayText().length() + 4), totalChildrenWidth += this.nodeSpacing * (node.children.size() - 1));
        return node.width;
    }

    private void assignPositions(TreeNode node, int x, int y) {
        node.x = x;
        node.y = y;
        if (!node.children.isEmpty()) {
            int currentX = x;
            int totalChildrenWidth = 0;
            for (TreeNode child : node.children) {
                totalChildrenWidth += child.width;
            }
            if (node.width > (totalChildrenWidth += this.nodeSpacing * (node.children.size() - 1))) {
                currentX = x + (node.width - totalChildrenWidth) / 2;
            }
            for (TreeNode child : node.children) {
                this.assignPositions(child, currentX, y + 4);
                currentX += child.width + this.nodeSpacing;
            }
        }
    }

    private String renderTree(TreeNode root) {
        char[][] canvas;
        int maxX = this.findMaxX(root) + 5;
        int maxY = this.findMaxY(root) + 3;
        for (char[] row : canvas = new char[maxY][maxX]) {
            Arrays.fill(row, ' ');
        }
        this.drawNode(canvas, root);
        StringBuilder result = new StringBuilder();
        result.append("\n");
        if (this.useColors) {
            result.append(BOLD);
        }
        result.append("Command Tree Structure\n");
        if (this.useColors) {
            result.append(RESET);
        }
        result.append("=".repeat(maxX)).append("\n").append("\n");
        for (char[] row : canvas) {
            result.append(new String(row).replaceAll("\\s+$", "")).append("\n");
        }
        if (this.showNodeTypes) {
            result.append("\n").append("Legend:\n");
            if (this.useColors) {
                result.append("  ").append(CYAN).append("\u25a1").append(RESET).append(" Subcommands\n");
                result.append("  ").append(YELLOW).append("\u25a1").append(RESET).append(" Required Arguments\n");
                result.append("  ").append(GREEN).append("\u25a1").append(RESET).append(" Optional Flags\n");
                result.append("  ").append(GRAY).append("\u25a1").append(RESET).append(" Optional Arguments\n");
            } else {
                result.append("  [SUB] Subcommands\n");
                result.append("  [REQ] Required Arguments\n");
                result.append("  [FLAG] Optional Flags\n");
                result.append("  [OPT] Optional Arguments\n");
            }
        }
        return result.toString();
    }

    private void drawNode(char[][] canvas, TreeNode node) {
        String text = node.getDisplayText();
        int boxWidth = Math.max(text.length() + 2, 10);
        int boxX = node.x + (node.width - boxWidth) / 2;
        int boxY = node.y;
        this.drawBox(canvas, boxX, boxY, boxWidth, text);
        if (!node.children.isEmpty()) {
            int parentCenterX = boxX + boxWidth / 2;
            canvas[boxY + 2][parentCenterX] = VERTICAL_LINE.charAt(0);
            if (node.children.size() == 1) {
                TreeNode child = node.children.get(0);
                int childBoxWidth = Math.max(child.getDisplayText().length() + 2, 10);
                int childCenterX = child.x + (child.width - childBoxWidth) / 2 + childBoxWidth / 2;
                if (parentCenterX == childCenterX) {
                    canvas[boxY + 3][parentCenterX] = VERTICAL_LINE.charAt(0);
                } else {
                    canvas[boxY + 3][parentCenterX] = VERTICAL_LINE.charAt(0);
                    int startX = Math.min(parentCenterX, childCenterX);
                    int endX = Math.max(parentCenterX, childCenterX);
                    for (int x = startX; x <= endX; ++x) {
                        if (canvas[boxY + 3][x] != ' ') continue;
                        canvas[boxY + 3][x] = HORIZONTAL_LINE.charAt(0);
                    }
                    canvas[boxY + 3][parentCenterX] = parentCenterX < childCenterX ? CORNER_BOTTOM_LEFT.charAt(0) : CORNER_BOTTOM_RIGHT.charAt(0);
                    canvas[boxY + 3][childCenterX] = T_JUNCTION.charAt(0);
                }
            } else {
                int childCenterX;
                int childBoxWidth;
                canvas[boxY + 3][parentCenterX] = T_JUNCTION.charAt(0);
                int leftmostX = Integer.MAX_VALUE;
                int rightmostX = Integer.MIN_VALUE;
                for (TreeNode child : node.children) {
                    childBoxWidth = Math.max(child.getDisplayText().length() + 2, 10);
                    childCenterX = child.x + (child.width - childBoxWidth) / 2 + childBoxWidth / 2;
                    leftmostX = Math.min(leftmostX, childCenterX);
                    rightmostX = Math.max(rightmostX, childCenterX);
                }
                for (int x = leftmostX; x <= rightmostX; ++x) {
                    if (canvas[boxY + 3][x] != ' ') continue;
                    canvas[boxY + 3][x] = HORIZONTAL_LINE.charAt(0);
                }
                for (TreeNode child : node.children) {
                    childBoxWidth = Math.max(child.getDisplayText().length() + 2, 10);
                    childCenterX = child.x + (child.width - childBoxWidth) / 2 + childBoxWidth / 2;
                    if (canvas[boxY + 3][childCenterX] != HORIZONTAL_LINE.charAt(0)) continue;
                    canvas[boxY + 3][childCenterX] = T_JUNCTION.charAt(0);
                }
            }
        }
        for (TreeNode child : node.children) {
            this.drawNode(canvas, child);
        }
    }

    private void drawBox(char[][] canvas, int x, int y, int width, String text) {
        int i;
        if (y >= canvas.length || x + width >= canvas[0].length) {
            return;
        }
        canvas[y][x] = CORNER_TOP_LEFT.charAt(0);
        for (int i2 = 1; i2 < width - 1; ++i2) {
            canvas[y][x + i2] = HORIZONTAL_LINE.charAt(0);
        }
        canvas[y][x + width - 1] = CORNER_TOP_RIGHT.charAt(0);
        canvas[y + 1][x] = VERTICAL_LINE.charAt(0);
        int textStart = x + 1 + (width - 2 - text.length()) / 2;
        for (i = 0; i < text.length() && textStart + i < x + width - 1; ++i) {
            canvas[y + 1][textStart + i] = text.charAt(i);
        }
        canvas[y + 1][x + width - 1] = VERTICAL_LINE.charAt(0);
        canvas[y + 2][x] = CORNER_BOTTOM_LEFT.charAt(0);
        for (i = 1; i < width - 1; ++i) {
            canvas[y + 2][x + i] = HORIZONTAL_LINE.charAt(0);
        }
        canvas[y + 2][x + width - 1] = CORNER_BOTTOM_RIGHT.charAt(0);
        if (!this.useColors || this.showNodeTypes) {
            // empty if block
        }
    }

    private int findMaxX(TreeNode node) {
        int max = node.x + node.width;
        for (TreeNode child : node.children) {
            max = Math.max(max, this.findMaxX(child));
        }
        return max;
    }

    private int findMaxY(TreeNode node) {
        int max = node.y + 3;
        for (TreeNode child : node.children) {
            max = Math.max(max, this.findMaxY(child));
        }
        return max;
    }

    private NodeType determineNodeType(ParameterNode<S, ?> node) {
        String format = node.format().toLowerCase();
        if (format.startsWith("-") || format.startsWith("--")) {
            return NodeType.OPTIONAL_FLAG;
        }
        if (format.startsWith("<") && format.endsWith(">")) {
            return NodeType.REQUIRED_ARG;
        }
        if (format.startsWith("[") && format.endsWith("]")) {
            return NodeType.OPTIONAL_ARG;
        }
        return NodeType.SUBCOMMAND;
    }

    public void visualizeSimple() {
        if (this.tree == null || !ImperatDebugger.isEnabled()) {
            return;
        }
        StringBuilder builder = new StringBuilder();
        builder.append("\n==== Command Tree ====\n\n");
        this.visualizeSimpleNode(this.tree.rootNode(), builder, 0, new ArrayList<Boolean>(), true);
        ImperatDebugger.debug(builder.toString(), new Object[0]);
    }

    public void visualizeUniqueTreeSimple() {
        if (this.tree == null || !ImperatDebugger.isEnabled()) {
            return;
        }
        StringBuilder builder = new StringBuilder();
        builder.append("\n==== Command Tree ====\n\n");
        this.visualizeSimpleNode(this.tree.uniqueVersionedTree(), builder, 0, new ArrayList<Boolean>(), true);
        ImperatDebugger.debug(builder.toString(), new Object[0]);
    }

    private void visualizeSimpleNode(ParameterNode<S, ?> node, StringBuilder builder, int depth, List<Boolean> lastFlags, boolean isLast) {
        for (int i = 0; i < depth - 1; ++i) {
            builder.append(lastFlags.get(i) != false ? "     " : "  \u2502  ");
        }
        if (depth > 0) {
            builder.append(isLast ? "  \u2514\u2500 " : "  \u251c\u2500 ");
        }
        String nodeText = node.format();
        builder.append(nodeText).append("\n");
        LinkedList<ParameterNode<S, ?>> children = node.getChildren();
        ArrayList<Boolean> newLastFlags = new ArrayList<Boolean>(lastFlags);
        if (depth > 0) {
            newLastFlags.add(isLast);
        }
        for (int i = 0; i < children.size(); ++i) {
            this.visualizeSimpleNode((ParameterNode)children.get(i), builder, depth + 1, newLastFlags, i == children.size() - 1);
        }
    }

    private class TreeNode {
        ParameterNode<S, ?> node;
        NodeType type;
        List<TreeNode> children = new ArrayList<TreeNode>();
        int x;
        int y;
        int width;

        TreeNode(ParameterNode<S, ?> node, NodeType type) {
            this.node = node;
            this.type = type;
        }

        String getDisplayText() {
            String base = this.node.format();
            if (CommandTreeVisualizer.this.showNodeTypes && this.type != NodeType.ROOT) {
                switch (this.type) {
                    case SUBCOMMAND: {
                        return "[SUB] " + base;
                    }
                    case REQUIRED_ARG: {
                        return "[REQ] " + base;
                    }
                    case OPTIONAL_FLAG: {
                        return "[FLAG] " + base;
                    }
                    case OPTIONAL_ARG: {
                        return "[OPT] " + base;
                    }
                }
            }
            return base;
        }
    }

    private static enum NodeType {
        ROOT,
        SUBCOMMAND,
        REQUIRED_ARG,
        OPTIONAL_FLAG,
        OPTIONAL_ARG;

    }
}

