package team.bangbang.common;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import team.bangbang.common.config.Config;
import team.bangbang.common.data.KeyValue;
import team.bangbang.common.data.generator.Sequence;
import team.bangbang.common.exception.BizException;
import team.bangbang.common.file.FileReader;
import team.bangbang.common.file.FileWriter;
import team.bangbang.common.sql.SQLPool;
import team.bangbang.common.utility.LogicUtility;

/**
 * 系统架构共通外接调用的方法集合
 *
 * @author 帮帮组
 * @version 1.0 2006-07-16
 */
public final class CommonMPI {
	/* 日志对象 */
	private  final static Logger logger = LoggerFactory.getLogger(CommonMPI.class);

	/* 生成关键字段的序列号值 */
	private static Sequence sequenceInstance = null;
	/* 性别备选项 */
	private final static String[] sexFlags = { "男", "女" };

	/** 针对数据库的ISNULL函数名称 */
	public static final String ISNULL = "Oracle".equalsIgnoreCase(SQLPool
			.getDatabaseName()) ? "NVL" : ("MySQL".equalsIgnoreCase(SQLPool
			.getDatabaseName()) ? "IFNULL" : "ISNULL");
	/** */
	public static final String NOW = "Oracle".equalsIgnoreCase(SQLPool
			.getDatabaseName()) ? "SYSDATE" : ("MySQL".equalsIgnoreCase(SQLPool
			.getDatabaseName()) ? "NOW()" : "getDate()");

	/**
	 * 取得性别列表，男：1；女：2
	 *
	 * @return 性别列表List&lt;KeyValue&lt;Integer, String&gt;&gt;
	 */
	public static List<KeyValue> getSexList() {
		return getDictionaryList(sexFlags);
	}

	/**
	 * 取得性别名称
	 *
	 * @param sexFlag 性别标识，男：1；女：2
	 *
	 * @return 性别名称
	 */
	public static String getSexName(Integer sexFlag) {
		return getDictionaryName(sexFlags, sexFlag);
	}

	/**
	 * 取得判断列表,是:true ; 否: false
	 *
	 * @return 列表List&lt;KeyValue&lt;Boolean, String&gt;&gt;
	 */
	public static List<KeyValue> getBooleanList() {
		List<KeyValue> lst = new ArrayList<KeyValue>();

		KeyValue kv1 = new KeyValue(new Boolean(true), "是");
		lst.add(kv1);

		KeyValue kv2 = new KeyValue(new Boolean(false), "否");
		lst.add(kv2);

		return lst;
	}

	/**
	 * 取得有效标志列表，有效：true；无效：false
	 *
	 * @return 有效标志列表List&lt;KeyValue&lt;Boolean, String&gt;&gt;
	 */
	public static List<KeyValue> getActiveList() {
		List<KeyValue> lst = new ArrayList<KeyValue>();

		KeyValue kv1 = new KeyValue(new Boolean(true), "有效");
		lst.add(kv1);

		KeyValue kv2 = new KeyValue(new Boolean(false), "无效");
		lst.add(kv2);

		return lst;
	}

	/**
	 * 将给定的字符串，按下标顺序变为KeyValue列表，用于获得固定数据字典：<br>
	 * Key: Integer(index + 1)<br>
	 * Value: String[index]<br>
	 *
	 * @param arr 给定的字符串数组
	 *
	 * @return 固定数据字典
	 */
	public static List<KeyValue> getDictionaryList(String[] arr) {

		List<KeyValue> lst = new ArrayList<KeyValue>();
		// 将配置的组织机构类型变为KeyValue对象
		for (int i = 0; arr != null && i < arr.length; i++) {
			KeyValue kv = new KeyValue(new Integer(i + 1), arr[i]);
			lst.add(kv);
		}

		return lst;
	}

	/**
	 * 根据给定的编号，从固定数据字典原始数据（字符串数组）中找到相应的名称。
	 *
	 * @param arr
	 *            固定数据字典原始数据
	 * @param flag
	 *            固定数据字典的编号
	 *
	 * @return 编号对应的名称
	 */
	public static String getDictionaryName(String[] arr, Integer flag) {
		if (arr == null) {
			return null;
		}

		int nIndex = (flag == null) ? 0 : flag.intValue();
		// see getCategoryList()
		nIndex--;
		if (nIndex < 0 || nIndex >= arr.length) {
			return null;
		}

		return arr[nIndex];
	}

	/**
	 * 以{file.attachment.directory}/[tableName]/[pkValue].dat路径文件存储记录附属文件
	 *
	 * @param tableName
	 *            数据库表名
	 * @param dt
	 *            对应的数据库表记录创建时间，如果不为null，则获取yyyyMM作为一个目录层次
	 * @param pkValue
	 *            关键字的值
	 * @param content
	 *            附属文件的内容
	 * @throws IOException IO异常
	 */
	public static void writeAttachFile(String tableName, Date dt,
			String pkValue, byte[] content) throws IOException {
		// 获得存放文件的目录
		String menu = Config.getProperty("file.attachment.directory");
		if (menu == null) {
			throw new IOException("为了存放记录关联文件，请在"
					+ "application.properties文件中配置{file.attachment.directory}目录！");
		}

		String dir = menu + "/" + tableName + "/";
		if (dt != null) {
			dir += LogicUtility.getYearMonth(dt) + "/";
		}
		File f = new File(dir);
		if (!f.exists()) {
			f.mkdirs();
		}

		try {
			FileWriter fw = new FileWriter(dir + pkValue + ".dat");
			fw.writeBytes(content);
		} catch (IOException e) {
			throw e;
		}
	}

	/**
	 * 以{file.attachment.directory}/[tableName]/[pkValue].dat构建文件路径，读取该文件。
	 *
	 * 该文件为存储记录附属文件
	 *
	 * @param tableName
	 *            数据库表名
	 * @param dt
	 *            对应的数据库表记录创建时间，如果不为null，则获取yyyyMM作为一个目录层次
	 * @param pkValue
	 *            关键字的值
	 * @return 记录附属文件的内容
	 * @throws IOException IO异常
	 */
	public static byte[] readAttachFile(String tableName, Date dt,
			String pkValue) throws IOException {
		// 获得存放文件的目录
		String menu = Config.getProperty("file.attachment.directory");
		if (menu == null) {
			throw new IOException("在application.properties文件中没有配置"
					+ "{file.attachment.directory}目录！");
		}

		String file = menu + "/" + tableName + "/";
		if (dt != null) {
			file += LogicUtility.getYearMonth(dt) + "/";
		}

		file += pkValue + ".dat";
		File f = new File(file);
		if (!f.exists()) {
			return null;
		}

		byte[] by = null;
		try {
			FileReader reader = new FileReader(file);
			by = reader.readBytes();
		} catch (IOException e) {
			throw e;
		}

		return by;
	}

	/**
	 * 以{file.attachment.directory}/[tableName]/[pkValue].dat构建文件路径，读取该文件。
	 *
	 * 该文件为存储记录附属文件
	 *
	 * @param tableName
	 *            数据库表名
	 * @param dt
	 *            对应的数据库表记录创建时间，如果不为null，则获取yyyyMM作为一个目录层次
	 * @param pkValue
	 *            关键字的值
	 *
	 * @return 记录附属文件的内容
	 * @throws IOException IO异常
	 */
	public static boolean deleteAttachFile(String tableName, Date dt,
			String pkValue) throws IOException {
		// 获得存放文件的目录
		String menu = Config.getProperty("file.attachment.directory");
		if (menu == null) {
			throw new IOException("在application.properties文件中没有配置"
					+ "{file.attachment.directory}目录！");
		}

		String file = menu + "/" + tableName + "/";
		if (dt != null) {
			file += LogicUtility.getYearMonth(dt) + "/";
		}

		file += pkValue + ".dat";

		File f = new File(file);
		if (!f.exists()) {
			return false;
		}

		f = new File(file);

		return f.delete();
	}

	/**
	 * 获得随机文字字符串，可以包含数字和字母
	 *
	 * @param length
	 *            随机文字长度
	 *
	 * @return 随机文字字符串，可以包含数字和字母
	 */
	public static String getRandomString(int length) {
		// 随机码字符串序列
		String codeSerial = "123456789abcdefghkmnpqrstwxyABCDEFGHKLMNPQRSTWXY";
		// 生成随机类
		Random random = new Random();
		int n = codeSerial.length();
		StringBuffer code = new StringBuffer();
		for (int i = 0; i < length; i++) {
			code.append(codeSerial.charAt(random.nextInt(n)));
		}

		return code.toString();
	}

	/**
	 * 获得随机数字字符串，只能包含数字
	 *
	 * @param length
	 *            随机数字长度
	 *
	 * @return 随机数字字符串，只能包含数字
	 */
	public static String getRandomNumber(int length) {
		String serial = "0123456789";
		// 生成随机类
		Random random = new Random();
		int n = serial.length();
		StringBuffer code = new StringBuffer();
		for (int i = 0; i < length; i++) {
			code.append(serial.charAt(random.nextInt(n)));
		}

		return code.toString();
	}

	/**
	 * 得到当前应用的地址，包含context，但不包含context后面的uri部分。
	 * 
	 * 假设请求地址为：
	 * http://www.bangbang.team/admin/ab/cd.do
	 * 
	 * context为：/admin
	 * 
	 * 则当前方法返回：
	 * http://www.bangbang.team/admin/
	 *
     * @param request HTTP请求
	 *
	 * @return 当前应用的地址，以"/"符号结尾
	 */
	public static String getApplicationUrl(HttpServletRequest request) {
		// 可能是http或者https协议
		String protocol = request.getProtocol().toLowerCase();
		if (protocol.startsWith("https")) {
			protocol = "https";
		} else {
			protocol = "http";
		}

		String host = request.getServerName();
		int port = request.getServerPort();
		// 获得Context名字
		String context = request.getContextPath();
		String url = protocol + "://" + host;
		// 添加端口
		if (protocol.equals("http") && port == 80) {
			// http协议默认端口80
		} else if (protocol.equals("https") && port == 443) {
			// https协议默认端口443
		} else {
			url += ":" + port;
		}

		if (context != null && context.length() > 0) {
			if (!context.startsWith("/")) {
				url += "/";
			}

			url += context + "/";
		} else {
			url += "/";
		}

		return url;
	}

	/**
	 * @return 关键字段的序列号值
	 */
	public synchronized static long generateSequenceId() {
		if(sequenceInstance == null) {
			sequenceInstance = new Sequence();
		}

		return sequenceInstance.nextId();
	}

	/**
	 * 获取请求发起者的IP地址
	 *
	 * @param request HTTP请求
	 * @return 请求发起者的IP地址
	 */
	public static String getRequestIP(HttpServletRequest request) {
		String ip = request.getHeader("x-forwarded-for");
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getHeader("WL-Proxy-Client-IP");
		}
		if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
			ip = request.getRemoteAddr();
		}
		return ip;
	}

	/**
	 * 获得uri参数值，如果该参数没有值，则返回当前访问的URI（contextPath之后的字符串部分，以"/"符号开头）
	 *
     * @param request HTTP请求
	 *
	 * @return uri，以"/"符号开头
	 */
	public static String getURI(HttpServletRequest request) {
		String uri = request.getParameter("uri");
		if (uri == null) {
			// 当前访问的地址（contextPath之后的字符串部分）
			uri = request.getRequestURI().trim();
			// 替换地址中的重复 / 符号
			uri = uri.replaceAll("/+", "/");
			String cp = request.getContextPath();
			uri = uri.substring(cp.length());
		}
		
		if (!uri.startsWith("/")) {
			uri = "/" + uri;
		}
		
		return uri;
	}

	/* 权限管理系统的根地址，包括context-path，以 / 符号结尾 */
	private static String CONSOLE_ROOT = null;
	/**
	 * @return 权限管理系统的根地址，包括context-path，以 / 符号结尾
	 */
	public static String getConsoleRoot() {
		if(CONSOLE_ROOT != null) {
			return CONSOLE_ROOT;
		}

		// 管理后台的根地址
		String consoleRoot = Config.getProperty("console.root");
		if(consoleRoot == null || consoleRoot.trim().length() == 0) {
			logger.error("当前应用未配置 console.root 参数，无法定位权限管理系统的根地址");
			return "/";
		}

		if (!consoleRoot.endsWith("/")) {
			// 保证管理后台的根地址以 / 结尾
			consoleRoot += "/";
		}

		CONSOLE_ROOT = consoleRoot;

		return CONSOLE_ROOT;
	}

	/**
	 * 获得指定账户的临时文件完整路径。
	 *
	 * 文件路径组成：[file.attachment.directory]\Temp\[accountId].dat
	 *
	 * @param accountId 指定账户编号，必须包含所在机构的编号和用户编号
	 * @return 临时文件，文件路径组成：[file.attachment.directory]\Temp\[accountId].dat
	 * @throws BizException 异常
	 */
	public static String getAccountTemporaryFile(Object accountId) throws BizException {
		// 检查参数
		if (accountId == null) {
			throw new BizException("指定的账户编号为空！");
		}

		// 附件文件根目录
		String root = Config.getProperty("file.attachment.directory");
		if (root == null || root.trim().length() == 0) {
			throw new BizException("在 application.properties 文件中没有配置 "
					+ "附件文件根目录（KEY: file.attachment.directory）！");
		}

		if (!root.endsWith("/") && !root.endsWith("\\")) {
			root += File.separator;
		}

		// 临时文件目录
		String tempFolder = root + "Temp" + File.separator;
		File folder = new File(tempFolder);

		if (!folder.exists()) {
			folder.mkdirs();
		}

		return tempFolder + accountId + ".dat";
	}
}
