package team.bangbang.common.config;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.Map;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;
import team.bangbang.common.utility.LogicUtility;

/**
 * 共通类库，读取application.properties文件中的配置信息
 * 
 * 读取配置文件时，按照spring boot的配置文件优先级进行读取：
 * 
 * <pre>
 * 1. 执行启动指令的目录下的config目录
 * 2. 执行启动指令的目录
 * 3. classpath(即resources目录)的config目录
 * 4. classpath(即resources目录)目录
 * </pre>
 *
 * @author 帮帮组
 * @version 1.0
 * @version 1.1 按照spring boot的优先级顺序（4个不同的位置）读取配置文件
 */
public final class Config {
	/* 日志对象 */
	private final static Logger logger = LoggerFactory.getLogger(Config.class);
	/* 配置文件的名称 */
	private static String config_file = "application";
	/* 配置信息对象 */
	private static Properties prop = null;
	/* 是否加载了文件 */
	private static boolean loaded = false;
	/* 当前环境名称（可能是dev、test、prod） */
	private static String activeProfile = null;

	/**
	 * 禁用构造方法的外部调用
	 */
	private Config() {
	}

	/**
	 * 读取配置文件中的信息
	 */
	private static void initialize() {
		prop = new Properties();
		// 读引导配置文件：bootstrap
		read("bootstrap");
		// 读入口配置文件：application
		read(config_file);

		// 系统环境变量是否配置了SPRING_PROFILES_ACTIVE？
		String profile = System.getenv("SPRING_PROFILES_ACTIVE");
		if (profile == null || profile.trim().length() == 0) {
			// ide是否指定了指定了spring.profiles.active
			profile = System.getProperty("spring.profiles.active");
		}

		if (profile == null || profile.trim().length() == 0) {
			// application.propertperties 是否配置了{spring.profiles.active}？
			profile = prop.getProperty("spring.profiles.active");
		}

		if (profile != null && profile.trim().length() > 0) {
			activeProfile = profile.trim();
			read(config_file + "-" + profile.trim());
		}

		// 是否配置了nacos作为配置中心？
		// nacos请求地址
		String addrs = prop.getProperty("spring.cloud.nacos.config.server-addr");
		if (addrs != null && addrs.trim().length() > 0) {
			NacosConfigReader ncr = new NacosConfigReader(prop);
			Map<String, String> mpConfig = ncr.readConfig();
			if (mpConfig != null && !mpConfig.isEmpty()) {
				prop.putAll(mpConfig);
			}
		}

		loaded = true;
	}

	/**
	 * 重新加载配置信息
	 */
	public static void refresh() {
		prop = null;
		loaded = false;
	}

	/**
	 * 获取指定配置参数的value值
	 *
	 * @param key 配置参数的KEY值
	 * @return 配置参数的value值
	 */
	public static String getProperty(String key) {
		if (!loaded) {
			initialize();
		}
		return prop.getProperty(key);
	}

	/**
	 * 返回所有配置参数的KEY集合
	 *
	 * @return 所有配置参数的KEY集合
	 */
	public static Enumeration<String> keys() {
		if (!loaded) {
			initialize();
		}

		return new Enumeration<String>() {
			Enumeration<Object> keys = prop.keys();

			/**
			 * @return 是否有下一个Key元素
			 */
			public boolean hasMoreElements() {
				return keys.hasMoreElements();
			}

			/**
			 * @return 下一个Key元素
			 */
			public String nextElement() {
				return (String) keys.nextElement();
			}
		};
	}

	/**
	 * 获得当前环境名称（可能是dev、test、prod）
	 *
	 * @return 当前环境名称（可能是dev、test、prod）
	 */
	public static String getActiveProfile() {
		if (!loaded) {
			initialize();
		}

		if (activeProfile == null) {
			activeProfile = "";
		}

		return activeProfile;
	}

	/**
	 * 读取配置文件
	 * 
	 * @param fileName 配置文件名称，不包含扩展名
	 */
	private static void read(String fileName) {
		// 读properties文件
		boolean bl = readProperties(fileName + ".properties");
		if (bl)
			return;

		// 读yml文件
		bl = readYml(fileName + ".yml");
		if (bl)
			return;

		// 读yaml文件
		bl = readYml(fileName + ".yaml");
		if (bl)
			return;
	}

	/**
	 * 读取properties配置文件
	 * 
	 * @param fileName properties配置文件名，包含扩展名
	 * @return 读取是否成功
	 */
	private static boolean readProperties(String fileName) {
		InputStream is = null;

		// 获得外置配置文件，包括：
		// 1. 执行启动指令目录/config/[fileName]
		// 2. 执行启动指令目录/[fileName]
		File f = getExternalFile(fileName);
		if (f != null) {
			try {
				is = new FileInputStream(f);
				prop.load(is);

				return true;
			} catch (Exception ex) {
				logger.info("读取配置文件 " + fileName + " 失败，忽略读取");
			} finally {
				try {
					if (is != null)
						is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		// 3. classpath(即resources目录)目录下的config目录
		try {
			is = Thread.currentThread().getContextClassLoader().getResourceAsStream("config/" + fileName);
			prop.load(is);

			return true;
		} catch (Exception ex) {
			logger.info("未发现配置文件 classpath://config/" + fileName + "，忽略读取");
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		// 4. classpath(即resources目录)目录
		try {
			is = Thread.currentThread().getContextClassLoader().getResourceAsStream(fileName);
			prop.load(is);

			return true;
		} catch (Exception ex) {
			logger.info("未发现配置文件 classpath://" + fileName + "，忽略读取");
		} finally {
			try {
				if (is != null)
					is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		return false;
	}

	/**
	 * 获得外置配置文件，优先级为：
	 * 
	 * 1. 执行启动指令目录/config/[fileName]
	 * 
	 * 2. 执行启动指令目录/[fileName]
	 * 
	 * @param fileName 文件名称
	 * @return 外置配置文件
	 */
	private static File getExternalFile(String fileName) {
		// 执行启动指令目录
		String path = System.getProperty("user.dir");
		if (!path.endsWith(File.separator)) {
			path += File.separator;
		}
		String path1 = path + "config" + File.separator + fileName;

		File f = new File(path1);

		// 如果文件不存在，则返回null
		if (f.exists()) {
			// 1. 执行启动指令目录/config/[fileName]
			return f;
		}

		// 检查
		// 2. 执行启动指令目录/[fileName]
		String path2 = path + fileName;
		f = new File(path2);

		// 如果文件不存在，则返回null
		if (f.exists()) {
			// 2. 执行启动指令目录/[fileName]
			return f;
		}

		return (f.exists() ? f : null);
	}

	/**
	 * 读取yml配置文件
	 * 
	 * @param fileName yml配置文件名，包含扩展名
	 * @return 读取是否成功
	 */
	@SuppressWarnings("rawtypes")
	private static boolean readYml(String fileName) {
		InputStream is = null;
		Map m = null;
		Yaml yaml = new Yaml();
		boolean loaded = false;

		// 获得外置配置文件，包括：
		// 1. 执行启动指令目录/config/[fileName]
		// 2. 执行启动指令目录/[fileName]
		File f = getExternalFile(fileName);
		if (f != null && f.exists()) {
			try {
				is = new FileInputStream(f);
				m = yaml.load(is);
				loaded = true;
			} catch (Exception ex) {
				logger.info("读取配置文件 " + fileName + " 失败，忽略读取");
			} finally {
				try {
					if (is != null)
						is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		// 3. classpath(即resources目录)目录下的config目录
		if (!loaded) {
			try {
				is = Thread.currentThread().getContextClassLoader().getResourceAsStream("config/" + fileName);
				if (is != null) {
					m = yaml.load(is);
					loaded = true;
				}
			} catch (Exception ex) {
				logger.info("未发现配置文件 classpath://config/" + fileName + "，忽略读取");
			} finally {
				try {
					if (is != null)
						is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		if (!loaded) {
			try {
				is = Thread.currentThread().getContextClassLoader().getResourceAsStream(fileName);
				if (is != null) {
					m = yaml.load(is);
					loaded = true;
				}
			} catch (Exception ex) {
				ex.printStackTrace();
				logger.info("未发现配置文件 classpath://" + fileName + "，忽略读取");
				return false;
			} finally {
				try {
					if (is != null)
						is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}

		loadFromMap(m, "");

		return true;
	}

	/**
	 * 把Map中的数据读取到Properties中
	 * 
	 * @param m      Map中的数据，有嵌套
	 * @param prefix KEY值前缀
	 */
	@SuppressWarnings("rawtypes")
	static void loadFromMap(Map m, String prefix) {
		if (m == null || m.isEmpty()) {
			return;
		}

		for (Object key : m.keySet()) {
			Object value = m.get(key);

			String sk = String.valueOf(key).trim();

			sk = (prefix == null || prefix.trim().length() == 0) ? sk : prefix.trim() + "." + sk;

			if (value == null)
				continue;

			// 处理子数据
			if (value instanceof Map) {
				loadFromMap((Map) value, sk);
				continue;
			}

			// 正常数据
			prop.put(sk, String.valueOf(value).trim());
		}
	}
}
