package team.bangbang.common.config;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;
import team.bangbang.common.net.http.HttpClient;

import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

/**
 * 从Nacos配置中心获取配置参数
 *
 * @author 帮帮组
 * @version 1.0
 */
public class NacosConfigReader {
    /* 日志对象 */
    private final static Logger logger = LoggerFactory.getLogger(NacosConfigReader.class);
    /* HttpClient */
    private static HttpClient hc = new HttpClient();
    /* 请求地址，多个请求地址以半角逗号间隔 */
    private String addrs = null;
    /* 命名空间 */
    private String namespace = null;
    /* 组 */
    private String group = null;
    /* dataId */
    private String dataId = null;
    /* 配置文件格式：properties、yml */
    private String extension = null;

    /**
     * 构造方法
     *
     * @param prop nacos配置中心参数
     */
    public NacosConfigReader(Properties prop) {
        // nacos请求地址，多个请求地址以半角逗号间隔
        this.addrs = prop.getProperty("spring.cloud.nacos.config.server-addr");

        // 命名空间
        this.namespace = prop.getProperty("spring.cloud.nacos.config.namespace");
        // group
        this.group = prop.getProperty("spring.cloud.nacos.config.group");
        if (this.group == null || this.group.trim().length() == 0) {
            this.group = "DEFAULT_GROUP";
        }
        // dataId
        this.dataId = prop.getProperty("spring.application.name");
        // 配置文件格式：properties、yml
        this.extension = prop.getProperty("spring.cloud.nacos.config.file-extension");
    }

    /**
     * 读取配置中心的配置
     *
     * @return 配置中心的配置
     */
    public Map<String, String> readConfig() {
        Map<String, String> mp = new HashMap<String, String>();
        if (addrs == null || addrs.trim().length() == 0) {
            return mp;
        }

        // 获取nacos中的配置信息的API地址
        String url = getApiUrl();
        if (url == null || url.trim().length() == 0) {
            return mp;
        }

        // 获取配置
        Map<String, String> config = readConfigByApi(url);
        return config;
    }

    /**
     * @return 获取配置中心配置的API地址
     */
    private String getApiUrl() {
        // 地址示例：
        // http://localhost:8848/nacos/v1/cs/configs?tenant=25488bb9-913b-4b38-9e2f-05f7f47c5ab0&dataId=component-user&group=DEFAULT_GROUP
        int index = (addrs != null ? addrs.indexOf(",") : -1);
        String addr = (index >= 0 ? addrs.substring(0, index).trim() : addrs);
        if (addr == null || addr.length() == 0) return null;

        String url = "http://" + addr + "/nacos/v1/cs/configs?";
        url += "tenant=" + namespace;
        url += "&dataId=" + dataId + "." + extension;
        url += "&group=" + group;

        return url;
    }

    private Map<String, String> readConfigByApi(String url) {
        String content = null;
        try {
            content = hc.get(url).toString();
        } catch (Exception e) {
            logger.error("请求nacos配置中心的API发生错误，请求地址为：" + url);
            logger.error(e.getMessage());
        }

        if (content == null || content.trim().length() == 0) {
            return null;
        }

        try {
            if ("yml".equalsIgnoreCase(extension) || "yaml".equalsIgnoreCase(extension)) {
                Yaml yaml = new Yaml();
                Map<String, String> m = yaml.load(content);
                Config.loadFromMap(m, "");

                return m;
            } else if("properties".equalsIgnoreCase(extension)) {
                return loadPropertiesFromString(content);
            } else {
                logger.info("未指定${spring.cloud.nacos.config.file-extension}，使用properties格式加载配置...");
                return loadPropertiesFromString(content);
            }
        } catch (Exception e) {
            logger.error("解析nacos配置中心的配置发生错误：" + e.getMessage());
        }

        return null;
    }

    private Map<String, String> loadPropertiesFromString(String content) {
        Map<String, String> m = new HashMap<String, String>();
        // 尝试使用properties解析
        Properties prop = new Properties();
        try {
            prop.load(new StringReader(content));

            for(Object key : prop.keySet()) {
                String value = (String)prop.getProperty((String)key);
                m.put((String)key, value);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return m;
    }
}
