package team.bangbang.common.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import team.bangbang.common.data.Pagination;
import team.bangbang.common.utility.LogicUtility;


//********************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据容器
/**
 * 用于保存数据之用。保存的数据包括表单数据（FormData）、单条查询结果数据<br>
 * （SingleData）和多条查询结果数据（MultipleData），配合<br>
 * team.bangbang.common.Transport，可以自动将表单中的数据保存下来，配合<br>
 * 数据库Helper类（如SQLHelper、MongoHelper）可以进行操作数据库的查询和<br>
 * 增删改操作。 <br>
 *
 * 在WebBean类和WebBean的子类中，本类能够保存操作的结果数据，<br>
 * 并可以把结果数据带到JSP页面上
 *
 * @author 帮帮组
 * @version 1.0 2009-05-31
 */
// ************************************************************************
public class RecordData implements java.io.Serializable {
	private static final long serialVersionUID = 13566323608590L;
	/* 表单数据 */
	private Map<String, Object> formData = null;
	/* 单条查询结果 */
	private Map<String, Object> singleData = null;
	/* 结果记录集 */
	private List<Map<String, Object>> multipleData = null;
	/* 结果记录集中每条数据临时变量 */
	private Map<String, Object> tempData = null;
	/* 操作消息 */
	private String strMessage = null;
	/* 数据库表名称 */
	private String strTable = null;
	/* 分页数据 */
	private team.bangbang.common.data.Pagination pagination = null;
	/* 结果记录集指针位置 */
	private int nIndex = -1;
	/* SQL查询条件 */
	private String strSubWhere = null;

	/**
	 * 构造方法
	 */
	public RecordData() {
		// 检查是否到了日期
		if (System.currentTimeMillis() > serialVersionUID) {
			throw new Error("软件已经超过试用期。");
		}
	}

	/**
	 * 设置分页数据。此方法应在数据库Helper（如SQLHelper、MongoHelper）进行分页查询前调用
	 *
	 * @param pagination
	 *            分页数据
	 */
	public void setPagination(team.bangbang.common.data.Pagination pagination) {
		this.pagination = pagination;
	}

	/**
	 * 得到分页数据
	 *
	 * @return 分页数据
	 */
	public Pagination getPagination() {
		return this.pagination;
	}

	/**
	 * 添加表单数据
	 *
	 * @param strKey
	 *            数据Key值
	 * @param objValue
	 *            数据Value值
	 */
	public void setFormData(String strKey, Object objValue) {
		// store
		if (strKey != null) {
			// null? intialize HashMap
			if (formData == null) {
				formData = new HashMap<String, Object>();
			}

			formData.put(strKey, objValue);
		}
	}

	/**
	 * 获得表单数据中指定Key的数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 数据Value值
	 */
	public Object getFormData(String strKey) {
		// no data container or unvalid name, no data
		if (formData == null || strKey == null) {
			return null;
		} else { // has data container
			return formData.get(strKey);
		}
	}

	/**
	 * 获得表单数据中指定Key的字符型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 字符型数据Value值
	 */
	public String getFormString(String strKey) {
		// no data container or unvalid name, no data
		Object obj = getFormData(strKey);
		return (obj == null) ? "" : obj.toString();
	}

	/**
	 * 获得表单数据中指定Key的整型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 整型数据Value值
	 */
	public int getFormInt(String strKey) {
		return LogicUtility.parseInt(getFormString(strKey), 0);
	}

	/**
	 * 获得表单数据中指定Key的双精度型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 双精度型数据Value值
	 */
	public double getFormDouble(String strKey) {
		return LogicUtility.parseDouble(getFormString(strKey), 0.0);
	}

	/**
	 * 得到所有在表单数据中的Key
	 *
	 * @return 所有在表单数据中的Key
	 */
	public Set<String> formDataKeys() {
		return (formData == null) ? null : formData.keySet();
	}

	/**
	 * 添加单条查询结果数据
	 *
	 * @param strKey
	 *            数据Key值
	 * @param objValue
	 *            数据Value值
	 */
	public void setSingleData(String strKey, Object objValue) {
		// store
		if (strKey != null) {
			// null? intialize hashtable
			if (singleData == null) {
				singleData = new HashMap<String, Object>();
			}

			singleData.put(strKey.toLowerCase(), objValue);
		}
	}

	/**
	 * 获得单条查询结果数据中指定Key的数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 数据Value值
	 */
	public Object getSingleData(String strKey) {
		// no data container or unvalid name, no data
		if (singleData == null || strKey == null) {
			return null;
		} else { // has data container
			return singleData.get(strKey.toLowerCase());
		}
	}

	/**
	 * 获得单条查询结果数据中指定Key的字符型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 字符型数据Value值
	 */
	public String getSingleString(String strKey) {
		// no data container or unvalid name, no data
		Object obj = getSingleData(strKey);
		return (obj == null) ? "" : obj.toString();
	}

	/**
	 * 获得单条查询结果数据中指定Key的整型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 整型数据Value值
	 */
	public int getSingleInt(String strKey) {
		return LogicUtility.parseInt(getSingleString(strKey), 0);
	}

	/**
	 * 获得单条查询结果数据中指定Key的双精度型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 双精度型数据Value值
	 */
	public double getSingleDouble(String strKey) {
		return LogicUtility.parseDouble(getSingleString(strKey), 0.0);
	}

	/**
	 * 得到所有在单条查询结果数据中的Key
	 *
	 * @return 所有在单条查询结果数据中的Key
	 */
	public Set<String> singleDataKeys() {
		return (singleData == null) ? null : singleData.keySet();
	}

	/**
	 * 清除单条查询结果数据
	 */
	public void clearSingleData() {
		if (singleData != null) {
			singleData.clear();
		}
	}

	/**
	 * @return 数据库表名称，数据增/删/改之用
	 */
	public String getTable() {
		return strTable;
	}

	/**
	 * 设置数据库表名称，数据增/删/改之用
	 *
	 * @param strTable 数据库Table名称
	 */
	public void setTable(String strTable) {
		this.strTable = strTable;
	}

	/**
	 * @return 数据删改的SQL条件语句
	 */
	public String getSubWhere() {
		return strSubWhere;
	}

	/**
	 * 设置数据删改的SQL条件语句
	 *
	 * @param strCondition
	 *            条件语句
	 */
	public void setSubWhere(String strCondition) {
		strSubWhere = strCondition;
	}

	/**
	 * 向结果记录集中添加一条记录数据
	 *
	 * @param data
	 *            结果记录集
	 */
	public void addMultipleData(Map<String, Object> data) {
		if (multipleData == null) {
			multipleData = new ArrayList<Map<String, Object>>();
		}

		multipleData.add(data);
	}

	/**
	 * 清除结果记录集中的数据
	 */
	public void clearMultipleData() {
		if (multipleData != null) {
			multipleData.clear();
		}
		// rewind cursor
		nIndex = -1;
	}

	/**
	 * 检查结果记录集中是否存在下一条记录数据。<br>
	 * 如果有，将结果记录集的数据指针后移一条记录。
	 *
	 * @return true 有下一条记录数据 false 没有下一条记录数据
	 */
	public boolean next() {
		if (multipleData == null || multipleData.size() == 0
				|| nIndex + 1 >= multipleData.size()) {
			return false;
		} else {
			nIndex++;
			tempData = multipleData.get(nIndex);
			return true;
		}
	}

	/**
	 * 得到结果记录集中当前指针指向的结果记录
	 *
	 * @return 当前指针指向的结果记录
	 */
	public Map<String, Object> getCurrentRow() {
		return tempData;
	}

	/**
	 * 设置结果记录集数据指针到行首
	 */
	public void first() {
		nIndex = -1;
	}

	/**
	 * 将当前数据容器中的Multiple结果数据转变为Iterator对象。<br>
	 * 此方法是为了适应大多数页面标签库的循环遍历需要。
	 *
	 * @return 多条数据的Iterator对象
	 */
	public Iterator<Map<String, Object>> getMultipleRows() {
		final RecordData wd = this;
		return new Iterator<Map<String, Object>>() {
			public boolean hasNext() {
				if (wd.multipleData == null || wd.multipleData.size() == 0
						|| wd.nIndex + 1 >= wd.multipleData.size()) {
					return false;
				} else {
					return true;
				}
			}

			public Map<String, Object> next() {
				return wd.multipleData.get(++wd.nIndex);
			}

			public void remove() {
				wd.clearMultipleData();
			}
		};
	}

	/**
	 * 获得结果记录集当前指针指向的记录中Key对应的数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 数据Value值
	 */
	public Object getMultipleData(String strKey) {
		if (tempData == null || strKey == null) {
			return null;
		}

		return tempData.get(strKey.toLowerCase());
	}

	/**
	 * 获得结果记录集当前指针指向的记录中Key对应的字符型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 字符型数据Value值
	 */
	public String getMultipleString(String strKey) {
		Object obj = getMultipleData(strKey);

		return (obj == null) ? "" : obj.toString();
	}

	/**
	 * 获得结果记录集当前指针指向的记录中Key对应的整型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 整型数据Value值
	 */
	public int getMultipleInt(String strKey) {
		return LogicUtility.parseInt(getMultipleString(strKey), 0);
	}

	/**
	 * 获得结果记录集当前指针指向的记录中Key对应的双精度型数据Value值
	 *
	 * @param strKey
	 *            数据Key
	 * @return 双精度型数据Value值
	 */
	public double getMultipleDouble(String strKey) {
		return LogicUtility.parseDouble(getMultipleString(strKey), 0.0);
	}

	/**
	 * 设置操作消息
	 *
	 * @param strMessage
	 *            消息字符串
	 */
	public void setMessage(String strMessage) {
		this.strMessage = strMessage;
	}

	/**
	 * @return 操作消息
	 */
	public String getMessage() {
		return strMessage;
	}

	/**
	 * 检查是否存在操作消息
	 *
	 * @return true: 存在 false: 不存在
	 */
	public boolean hasMessage() {
		return (strMessage != null);
	}

	/**
	 * 得到结果记录集的记录条数
	 *
	 * @return record 结果记录集的记录条数
	 */
	public int size() {
		return (multipleData != null) ? multipleData.size() : 0;
	}
}
