package team.bangbang.common.data.util;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import team.bangbang.common.utility.LogicUtility;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据成型器
/**
 * 把map中的数据按ognl表达式的规则，将请求的数据经类型转换后放入指定的业务对象中
 *
 * @author 帮帮组
 * @version 1.0
 */
// ************************************************************************
public class DataShaper {
	/**
	 * 将数据写入某个宿主对象的属性对象中
	 *
	 * @param obj
	 *            宿主对象，包含属性对象
	 * @param exp
	 *            表达式，含有.号调用，以属性对象的名称开头
	 * @param objValue
	 *            表达式对应的值
	 */
	public static void writeField(Object obj, String exp, Object objValue) {
		int nIndex = exp.indexOf(".");
		// Bean中对应的属性名称
		String objectName = (nIndex <= 0) ? exp : exp.substring(0, nIndex);

		// 宿主对象的类型
		Class<?> cls = obj.getClass();
		// 查找对应的Field
		Field f = null;
		try {
			f = cls.getField(objectName);
		} catch (Exception e) {
		}

		if (f == null) {
			// Bean中对应的属性不存在
			return;
		}

		if (nIndex <= 0) {
			// 直接给Bean中对应的属性赋值
			try {
				f.set(obj, toObject(f.getType(), objValue));
			} catch (Exception e) {
			}

			return;
		}

		// 查找属性对象
		Object objField = null;
		try {
			objField = f.get(obj);
		} catch (Exception e) {
		}

		if (objField == null) {
			return;
		}

		try {
			executeExpression(objField, exp.substring(nIndex + 1), objValue);
		} catch (Exception e) {
		}
	}

	/**
	 * 把数据按ognl表达式的规则，将请求的数据经类型转换后放入指定的业务对象中
	 *
	 * @param objField
	 *            pojo实例，比如名称叫order，order下有sales对象，sales下有userName
	 * @param strKey
	 *            表达式，含有.号调用，其值例如sales.userName
	 * @param objValue
	 *            表达式对应的值，字符串数据或者文件上传的File对象
	 * @throws Exception 异常
	 */
	public static void executeExpression(Object objField, String strKey,
			Object objValue) throws Exception {
		int nIndex = strKey.indexOf(".");
		if (nIndex <= 0) {
			// 1. 使用Field映射
			Field f = null;
			try {
				f = objField.getClass().getField(strKey);
			} catch (Exception e) {
			}

			if (f != null) {
				// 填入field值
				objValue = toObject(f.getDeclaringClass(), objValue);
				f.set(objField, objValue);
				return;
			}

			// 2. 使用方法映射
			String mtdSetName = "set" + upperFirst(strKey);
			// 所有方法
			Method[] mtds = objField.getClass().getMethods();
			for (Method m : mtds) {
				Class<?>[] csP = m.getParameterTypes();
				if (mtdSetName.equals(m.getName()) && csP != null
						&& csP.length == 1) {
					// 方法名称相同，并且只有一个参数的set方法
					try {
						m.invoke(objField, toObject(csP[0], objValue));
					} catch (Exception e) {
					}
				}
			}

			return;
		}

		// 1. 使用Field映射
		Field f = null;
		try {
			f = objField.getClass().getField(strKey);
		} catch (Exception e) {
		}

		if (f != null) {
			// 获得field值
			Object returnValued = f.get(objField);
			if (returnValued == null) {
				// 构造一个实例填入
				Class<?> cls = f.getDeclaringClass();
				// 返回值类型
				returnValued = cls.newInstance();
				// 将此对象值填入set方法
				if (returnValued != null) {
					f.set(objField, returnValued);
				}
			}

			if (returnValued != null) {
				// 进一步推进表达式
				strKey = strKey.substring(nIndex + 1);
				executeExpression(returnValued, strKey, objValue);
				return;
			}
		}

		// 2. 使用方法映射
		// 获得方法名称
		String mtdGetName = "get" + upperFirst(strKey.substring(0, nIndex));
		Method mtdGet = objField.getClass().getMethod(mtdGetName);
		// 获得方法的返回对象
		Object returnValued = mtdGet.invoke(objField);
		if (returnValued == null) {
			// 构造一个实例填入
			Class<?> cls = mtdGet.getReturnType();
			if (cls == null)
				return;
			// 返回值类型
			returnValued = cls.newInstance();
			// 将此对象值填入set方法
			if (returnValued == null)
				return;

			String mtdSetName = "set" + upperFirst(strKey.substring(0, nIndex));
			Method mtdSet = objField.getClass().getMethod(mtdSetName, cls);
			// 调用set方法填入objField
			mtdSet.invoke(objField, returnValued);
		}
		// 进一步推进表达式
		strKey = strKey.substring(nIndex + 1);
		executeExpression(returnValued, strKey, objValue);
	}

	/**
	 * 将传入的字符串进行首字母大写转换
	 *
	 * @param str
	 *            传入的字符串
	 * @return 首字母大写转换后的字符串
	 */
	private static String upperFirst(String str) {
		if (str == null || str.length() == 0) {
			return str;
		}

		return str.substring(0, 1).toUpperCase() + str.substring(1);
	}

	/**
	 * 将字符串数据转换为相应类型的对象数据
	 *
	 * @param clsF
	 *            指定的数据类型，目前仅仅支持String、Integer、Boolean、 Double、Date五种数据类型
	 * @param objValue
	 *            字符串数据或者文件上传的File对象
	 * @return 字符串数据转换后的相应类型对象数据
	 * @throws ParseException 异常
	 */
	public static Object toObject(Class<?> clsF, Object objValue)
			throws ParseException {
		if (clsF.equals(String.class)) {
			toStringObject(objValue);
		}

		if (clsF.equals(Integer.class)) {
			return toIntegerObject(objValue);
		}

		if (clsF.equals(int.class)) {
			try {
				int n = Integer.parseInt(String.valueOf(objValue));
				return n;
			} catch (Exception ex) {
				return null;
			}
		}

		if (clsF.equals(Long.class)) {
			return toLongObject(objValue);
		}

		if (clsF.equals(long.class)) {
			try {
				long l = Long.parseLong(String.valueOf(objValue));
				return l;
			} catch (Exception ex) {
				return null;
			}
		}

		if (clsF.equals(Boolean.class)) {
			return toBooleanObject(objValue);
		}

		if (clsF.equals(boolean.class)) {
			try {
				boolean bl = Boolean.parseBoolean(String.valueOf(objValue));
				return bl;
			} catch (Exception ex) {
				return null;
			}
		}

		if (clsF.equals(Double.class)) {
			try {
				if (objValue == null)
					return null;

				String s = String.valueOf(objValue).trim();
				if (s.length() == 0)
					return null;

				double d = Double.parseDouble(s);
				return new Double(d);
			} catch (Exception ex) {
				return null;
			}
		}

		if (clsF.equals(double.class)) {
			try {
				double d = Double.parseDouble(String.valueOf(objValue));
				return d;
			} catch (Exception ex) {
				return null;
			}
		}

		if (clsF.equals(Date.class)) {
			try {
				if (objValue == null)
					return null;

				String s = String.valueOf(objValue).trim();
				if (s.length() == 0)
					return null;

				return parseDate(s);
			} catch (Exception ex) {
				return null;
			}
		}

		// 当作Object类型返回
		return objValue;
	}

	private static Object toBooleanObject(Object objValue) {
		try {
			if (objValue == null)
				return null;

			String s = String.valueOf(objValue).trim();
			if (s.length() == 0)
				return null;

			s = s.toLowerCase();
			if ("true".equals(s)) {
				return true;
			}
			if ("false".equals(s)) {
				return false;
			}

			return null;
		} catch (Exception ex) {
			return null;
		}
	}

	private static Object toLongObject(Object objValue) {
		try {
			if (objValue == null)
				return null;

			String s = String.valueOf(objValue).trim();
			if (s.length() == 0)
				return null;

			long l = Long.parseLong(s);
			return new Long(l);
		} catch (Exception ex) {
			return null;
		}
	}

	private static Object toIntegerObject(Object objValue) {
		try {
			if (objValue == null)
				return null;

			String s = String.valueOf(objValue).trim();
			if (s.length() == 0)
				return null;

			int n = Integer.parseInt(s);
			return new Integer(n);
		} catch (Exception ex) {
			return null;
		}
	}

	private static String toStringObject(Object objValue) {
		if (objValue == null)
			return null;

		if (objValue.getClass().isArray()) {
			// 是数组
			// 数组长度
			int len = Array.getLength(objValue);

			StringBuffer sb = new StringBuffer();
			for (int i = 0; i < len; i++) {
				if (sb.length() > 0)
					sb.append(",");
				sb.append(Array.get(objValue, i));
			}
			return sb.toString();
		}

		return objValue.toString();
	}

	/**
	 * 根据一段日期时间字符串，转换得到日期时间对象
	 *
	 * @param str
	 *            日期时间字符串
	 * @return 日期时间对象
	 */
	private static Date parseDate(String str) {
		// 没有输入值
		if (str == null)
			return null;
		str = str.trim();
		if (str.length() == 0)
			return null;

		str = str.replace('/', '-');
		str = str.replace('\\', '-');

		// yyyy-MM-dd HH:mm:ss
		String[] sa = str.split("\\s+");
		if (sa == null || sa.length == 0)
			return null;

		// 年月日
		String s1 = null;
		// 时分秒
		String s2 = null;

		if (sa.length == 1) {
			// 可能是 年月日，也可能是 时分秒
			if (sa[0].indexOf(":") > 0) {
				s2 = sa[0];
			} else {
				s1 = sa[0];
			}
		}

		if (sa.length >= 2) {
			for (int i = 0; sa != null && i < 2 && i < sa.length; i++) {
				if (sa[i].indexOf(":") > 0 || s1 != null) {
					s2 = sa[i];
				} else {
					s1 = sa[i];
				}
			}
		}

		if (s1 == null)
			s1 = "1900-01-01";
		if (s2 == null)
			s2 = "00:00:00";

		int[] ymd = getYmd(s1);
		int[] hms = getHms(s2);

		Calendar cal = Calendar.getInstance();
		cal.set(ymd[0], ymd[1] - 1, ymd[2], hms[0], hms[1], hms[2]);
		return cal.getTime();
	}

	private static int[] getHms(String s2) {
		int[] hms = { 0, 0, 0 };
		int nEnd = s2.indexOf(":");
		if (nEnd < 0) {
			// 只有小时
			// 第1个数字
			hms[0] = LogicUtility.parseInt(s2, hms[0]);
			return hms;
		}

		int nStart = 0;
		// 时分秒
		// 第1个数字
		hms[0] = LogicUtility.parseInt(s2.substring(nStart, nEnd), hms[0]);
		// 第2个数字
		nStart = nEnd + 1;
		if (nStart >= s2.length())
			return hms;
		nEnd = s2.indexOf(":", nStart);
		if (nEnd < 0) {
			// 只有1个 -
			hms[1] = LogicUtility.parseInt(s2.substring(nStart), hms[1]);
			nEnd = s2.length();
		} else {
			hms[1] = LogicUtility.parseInt(s2.substring(nStart, nEnd), hms[1]);
		}
		// 第3个数字
		nStart = nEnd + 1;
		if (nStart >= s2.length())
			return hms;
		hms[2] = LogicUtility.parseInt(s2.substring(nStart), hms[2]);

		return hms;
	}

	private static int[] getYmd(String s1) {
		int[] ymd = { 1900, 1, 1 };
		int nEnd = s1.indexOf("-");
		if (nEnd < 0) {
			// 只有小时
			// 第1个数字
			ymd[0] = LogicUtility.parseInt(s1, ymd[0]);
			return ymd;
		}

		int nStart = 0;
		// 年月日
		// 第1个数字
		ymd[0] = LogicUtility.parseInt(s1.substring(nStart, nEnd), ymd[0]);
		// 第2个数字
		nStart = nEnd + 1;
		if (nStart >= s1.length())
			return ymd;
		nEnd = s1.indexOf("-", nStart);
		if (nEnd < 0) {
			// 只有1个 -
			ymd[1] = LogicUtility.parseInt(s1.substring(nStart), ymd[1]);
			nEnd = s1.length();
		} else {
			ymd[1] = LogicUtility.parseInt(s1.substring(nStart, nEnd), ymd[1]);
		}

		// 第3个数字
		nStart = nEnd + 1;
		if (nStart >= s1.length())
			return ymd;
		ymd[2] = LogicUtility.parseInt(s1.substring(nStart), ymd[2]);

		return ymd;
	}
}
