package team.bangbang.common.data.util;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import team.bangbang.common.config.Config;
import team.bangbang.common.exception.BizException;
import team.bangbang.common.file.FileUtil;
import team.bangbang.common.utility.LogicUtility;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.File;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：页面参数传输器

/**
 * 初始化页面请求，包括上传文件、转存Request中的Parameter、初始化数据操作<br>
 * 对象和数据容器
 *
 * @author 帮帮组
 * @version 1.0
 ************************************************************************/
public final class Transport {
    /* 服务端文件临时存储路径 */
    private static String tempSaveDir = System.getProperty("java.io.tmpdir");

    /* Spring配置文件中的最大文件size（支持MB、KB单位） */
    private static String spring_max_file_size = (Config.getProperty("spring.servlet.multipart.max-file-size") != null ?
            Config.getProperty("spring.servlet.multipart.max-file-size").toUpperCase().replaceAll("\\s+", "") : null);
    /* 兼容以前的最大文件size（支持MB、KB单位，默认为512KB） */
    private static String multipart_maxSize = (Config.getProperty("multipart.maxSize") != null ?
            Config.getProperty("multipart.maxSize").toUpperCase().trim().replaceAll("\\s+", "") : "512000");

    /* 上传文件的size限制(默认为512KB) */
    private static int maxSize = LogicUtility.parseInt((spring_max_file_size != null) ?
            spring_max_file_size.replaceAll("MB", "000000").replaceAll("KB", "000") :
            multipart_maxSize.replaceAll("MB", "000000").replaceAll("KB", "000"), 512000);

    /* 脚本开始标记（XSS攻击过滤） */
    private static Pattern scriptStart = Pattern.compile("<\\s*script[^>|.]*>",
            Pattern.CASE_INSENSITIVE);
    /* 脚本结束标记（XSS攻击过滤） */
    private static Pattern scriptEnd = Pattern.compile(
            "<\\s*/\\s*script[^>|.]*>", Pattern.CASE_INSENSITIVE);

    /* 文件上传进度数据保存的KEY前缀 */
    private static final String UPLOAD_PROGRESS_PREFIX = "UPLOAD_PROGRESS_PREFIX";

    /**
     * 把页面请求的参数传递到数据容器中，如果有文件上传，则进行相应的上传处理
     *
     * @param request 页面请求
     * @param data    数据容器，存储字符串参数或者文件上传的File对象
     */
    public static void transport(HttpServletRequest request,
                                 Map<String, Object> data) {
        String strEnctype = request.getContentType();

        // 使用multipart/form-data编码方式
        if (strEnctype != null && strEnctype.startsWith("multipart/form-data")
                && strEnctype.indexOf("boundary=") != -1) {
            // 有文件上传
            readRequestInStream(request, data);

            // 清除文件上传的进度
            HttpSession session = request.getSession();
            Enumeration<?> er = session.getAttributeNames();
            while (er != null && er.hasMoreElements()) {
                String key = (String) er.nextElement();
                if (key.startsWith(UPLOAD_PROGRESS_PREFIX)) {
                    session.removeAttribute(key);
                }
            }
        }

        // 读取QueryString参数
        readRequestURLParameter(request, data);
    }

    /**
     * 单纯文本参数表单提交的处理
     *
     * @param request 页面请求
     * @param data    数据容器，存储字符串参数或者文件上传的File对象
     */
    private static void readRequestURLParameter(HttpServletRequest request,
                                                Map<String, Object> data) {
        Enumeration<?> er = request.getParameterNames();

        int i = 0, nLen = 0;
        String strKey = null;
        String strValues[] = null;
        while (er != null && er.hasMoreElements()) {
            strKey = (String) er.nextElement();
            strValues = request.getParameterValues(strKey);
            if (strValues != null && strValues.length > 0) {
                nLen = strValues.length;
                for (i = 0; i < nLen; i++) {
                    // 保存参数
                    setParameter(data, strKey, strValues[i]);
                }
            }
        } // end while
    }

    /**
     * 含有符合内容的表单提交处理
     *
     * @param request 页面请求
     * @param data    数据容器，存储字符串参数或者文件上传的File对象
     * @throws BizException
     * @throws BizException
     */
    private static void readRequestInStream(final HttpServletRequest request,
                                            Map<String, Object> data) {
        DiskFileItemFactory factory = new DiskFileItemFactory();
        // 设置临时文件目录
        factory.setRepository(new File(tempSaveDir));
        // 设置最多只允许在内存中存储的数据,单位:字节
        factory.setSizeThreshold(2048);

        ServletFileUpload fu = new ServletFileUpload(factory);
        // 设置允许用户上传文件大小,单位:字节
        fu.setSizeMax(maxSize);
        // 设置字符集为UTF-8
        fu.setHeaderEncoding("UTF-8");
        // 开始读取上传信息 得到所有文件
        try {
            List<?> fileItems = fu.parseRequest(request);
            // 依次处理每个上传的文件和表单信息
            int nCount = (fileItems == null) ? 0 : fileItems.size();

            for (int i = 0; i < nCount; i++) {
                FileItem item = (FileItem) fileItems.get(i);

                // 控件名称
                String eleName = item.getFieldName();

                // 忽略其他不是文件域的所有表单信息
                if (item.isFormField()) {
                    // 处理表单数据
                    setParameter(data, eleName, item.getString());
                } else {
                    // 客户端文件名称
                    String clientFileName = item.getName();
                    if (clientFileName == null
                            || clientFileName.trim().length() == 0) {
                        continue;
                    }

                    // 保存文件字符串名称
                    setParameter(data, eleName, clientFileName);

                    // 上传文件
                    long size = item.getSize();
                    if (size == 0L) {
                        continue;
                    }

                    // 此时name为file选择框内的字符值
                    File dir = new File(tempSaveDir);
                    if (!dir.exists()) {
                        dir.mkdirs();
                    }

                    File savedFile = new File(tempSaveDir, System
                            .currentTimeMillis()
                            + "."
                            + FileUtil.getFileExtension(clientFileName));
                    item.write(savedFile);

                    // 保存文件实例
                    setParameter(data, eleName, savedFile);
                } // end if
            } // end for
        } catch (FileUploadException e) {
            // 这里异常产生的原因可能是用户上传文件超过限制、不明类型的文件等
            e.printStackTrace();
            // 自己处理的代码
            //throw new BizException("文件上传失败，原因可能是网络故障或者文件大小超过 " + maxSize
            //+ " 字节！");
        } catch (Exception e) {
            e.printStackTrace();
            // 自己处理的代码
            // throw new BizException(e);
        }
    }

    /**
     * 把name-value对保存到数据容器中。如果已经存在同名的name，<br>
     * 则value以数组形式追加在原保存值之后。
     *
     * @param data  数据容器，存储字符串参数或者文件上传的File对象
     * @param name  name值
     * @param value value值
     */
    private static void setParameter(Map<String, Object> data, String name,
                                     Object value) {
        if (value == null) {
            value = "";
        }

        // 添加XSS过滤
        if (value instanceof String && hasScriptTag(name, (String) value)) {
            return;
        }

        Object obj = data.get(name);

        if (obj == null) {
            data.put(name, value);
            return;
        }

        Object[] valueArray = null;
        Object[] temp = null;

        // 有多个同名数据
        if (obj instanceof Object[]) {
            temp = (Object[]) obj;
        } else {
            // 上次只有一个数据
            temp = new Object[1];
            temp[0] = obj;
        }

        int count = temp.length;
        valueArray = new Object[count + 1];
        // read old values
        for (int i = 0; i < count; i++) {
            valueArray[i] = temp[i];
        }
        // add new value
        valueArray[count] = value;
        data.put(name, valueArray);
    } // end setParameter

    /**
     * 检查传入的值是否有脚本标记
     *
     * @param name  传入的参数KEY
     * @param value 传入的参数Value
     * @return true：有 false：没有
     */
    private static boolean hasScriptTag(String name, String value) {
        if (name == null || value == null) {
            return false;
        }

        // 如果Name以Id或者Flag结尾，长度超过32视为有XSS攻击
        // 按照我们的开发规范，以Id或者Flag结尾的参数名称，其值不会超过36个字符。
        // 使用GUID的形式的Id，长度为36个字符
        // 允许添加前缀后缀，但不可超过50个字符
        name = name.toUpperCase();
        if (name.endsWith("ID") || name.endsWith("FLAG")) {
            if (value.length() > 50) {
                return true;
            }
        }

        boolean blA = scriptStart.matcher(value).find();
        if (blA) {
            // 有脚本开始标记
            return true;
        }

        boolean blB = scriptEnd.matcher(value).find();

        return blB;
    }
}
