package team.bangbang.common.exp;

import java.awt.Dimension;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;

import team.bangbang.common.CommonMPI;
import team.bangbang.common.exception.BizException;
import team.bangbang.common.file.ExcelWriter;

/**
 * 数据记录导出器
 *
 * @author 帮帮组
 * @version 1.0 2019年8月17日
 *
 * @param <T> 数据记录中的类型
 */
public abstract class RecordExporter<T> extends Exporter {
	/* 操作人编号 */
	private Object accountId = null;
	/* 数据记录 */
	private List<T> recordList = null;
	/* 导出的Excel文件名称，不需要带扩展名 */
	private String excelFileName = null;

	/**
	 * @param accountId     操作人编号
	 * @param recordList    数据记录
	 * @param excelFileName 导出的Excel文件名称，不需要带扩展名
	 */
	public RecordExporter(Object accountId, List<T> recordList, String excelFileName) {
		this.accountId = accountId;
		this.recordList = recordList;
		this.excelFileName = excelFileName;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see team.bangbang.common.exp.IExporter#export()
	 */
	@Override
	public String[] export() throws BizException {
		if (recordList == null || titles == null || fields == null) {
			throw new BizException("请首先执行查询，然后再执行导出！");
		}

		ExcelWriter ew = null;
		// 数据库链接
		try {
			if (titles.length != fields.length) {
				throw new BizException("标题数量（" + titles.length +
						"）与对应的字段数量（" + fields.length + "）不一致！");
			}

			// 临时文件
			String tempFile = CommonMPI.getAccountTemporaryFile(accountId);
			ew = new ExcelWriter(tempFile);

			// data
			// 行数量
			int count = 0;
			for (int i = 0; recordList != null && i < recordList.size(); i++) {
				if (count % 60000 == 0) {
					int nIndex = (count / 60000);
					// 创建工作簿
					ew.createSheet("数据 " + (nIndex + 1));
					// 从第2行第2列开始写
					ew.toRow(1);
					ew.toColumn(1);
					// 记录位置1
					Dimension off1 = ew.getOffset();
					for (int j = 0; j < titles.length; j++) {
						if (!selectedFields.contains(fields[j])) {
							// 忽略
							continue;
						}

						ew.writeCell(titles[j]);
					}

					// 记录位置2
					Dimension off2 = ew.getOffset();
					// 设置表头样式
					ew.setHeader(off1, off2);
				}

				// 换行
				ew.nextRow();
				// 第2列开始
				ew.toColumn(1);

				// 数据
				T t = recordList.get(i);

				for (int j = 0; j < fields.length; j++) {
					if (!selectedFields.contains(fields[j])) {
						// 忽略
						continue;
					}

					Object s = (fields[j] == null || fields[j].trim().length() == 0) ? null : getObject(t, fields[j]);

					ew.writeCell(s);
				}

				count++;
			}
		} catch (Exception e) {
			throw new BizException(e);
		} finally {
			try {
				if (ew != null)
					ew.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
		} // end try

		// 下载到客户端的文件信息。
		// String[0]：文件地址，{file.attachment.directory}/Temp/{UserId}.dat <br>
		// String[1]：文件名称，文件名称进行URL转码
		String s = excelFileName + ".xls";
		try {
			s = URLEncoder.encode(s, "UTF-8");
			s = URLEncoder.encode(s, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}
		String[] result = {"Temp/" + accountId + ".dat", s};
		return result;
	}

	/**
	 * 从数据对象中获取字段对应的值
	 *
	 * @param t         数据对象
	 * @param fieldName 字段名称
	 * @return 数据对象中获取的字段对应的值
	 */
	protected abstract Object getObject(T t, String fieldName);
}
