package team.bangbang.common.file;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import org.apache.poi.ss.usermodel.CellType;


//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：Excel文件数据检查
/**
 * 读取Excel文件数据，对照给定的类注解进行数据类型、范围（最大值/最小值）检查
 *
 * @author 帮帮组
 * @version 1.0 2012-12-12
 */
// ************************************************************************
public class ExcelDataChecker {
	/* Excel文件 */
	private String excelFile = null;
	/* 需要检查的工作簿序号，从0开始 */
	private int sheetIndex = 0;
	/* 数据规则定义类，每个属性包含使用ExcelColumn注解定义检查规则，属性名不作要求 */
	private Class<?> clazz = null;
	/* 检查开始行，从0开始 */
	private int fromRow = 0;
	/* Excel读 */
	private ExcelReader er = null;
	/* Excel数据检查规则 */
	private List<ExcelColumn> ecList = new ArrayList<ExcelColumn>();
	
	/**
	 * 构造一个Excel文件数据检查器
	 * 
	 * @param excelFile Excel文件
	 * @param sheetIndex 需要检查的工作簿序号，从0开始
	 * @param clazz 数据规则定义类，每个属性包含使用ExcelColumn注解定义检查规则，属性名不作要求
	 * @param fromRow 检查开始行，从0开始
	 */
	public ExcelDataChecker(String excelFile, int sheetIndex, Class<?> clazz, int fromRow) {
		this.excelFile = excelFile;
		this.sheetIndex = sheetIndex;
		this.clazz = clazz;
		this.fromRow = fromRow;
	}

	/**
	 * 对Excel进行格式检查
	 *
	 * @return 错误结果列表，如果列表empty，则表示检查通过
	 */
	public List<String> check() {
		List<String> msgs = new ArrayList<String>();
		try {
			// 检查输入参数
			checkParameter(msgs);
			
			if (!msgs.isEmpty()) {
				return msgs;
			}
			
			// 检查数据
			checkData(msgs);
		} catch (Exception ex) {
			msgs.add(ex.getMessage());
		} finally {
			if (this.er != null) er.close();
		}
		
		return msgs;
	}

	/**
	 * 检查输入参数：Excel文件、需要检查的工作簿序号、数据规则定义类、检查开始行
	 * 
	 * @param msgs 检查结果消息列表
	 */
	private void checkParameter(List<String> msgs) {
		// Excel文件
		try {
			er = new ExcelReader(this.excelFile);
		} catch (Exception e) {
			msgs.add("读Excel文件遇到问题，请确认Excel文件是否存在（" + this.excelFile + "）");
			// 如果Excel文件读遇到问题，则直接返回
			return;
		}
		
		// 需要检查的工作簿序号
		try {
			er.setSheetIndex(sheetIndex);
		} catch (Exception e) {
			msgs.add("指定的工作簿不存在（序号：" + this.sheetIndex + "）");
			// 如果工作簿不存在，则直接返回
			return;
		}
		
		// 检查开始行
		try {
			boolean bl = er.toRow(fromRow);
			if (!bl) {
				msgs.add("指定的数据行不存在（序号：" + this.fromRow + "）");
				// 如果数据行不存在，则直接返回
				return;
			}
		} catch (Exception e) {
			msgs.add("指定的数据行不存在（序号：" + this.fromRow + "）");
			// 如果数据行不存在，则直接返回
			return;
		}
		
		// 数据规则定义类，是否有属性
		Field[] fs = this.clazz.getDeclaredFields();
		if (fs == null || fs.length == 0) {
			msgs.add("数据规则定义类未包含任何属性（序号：" + this.clazz.getName() + "）");
			// 如果数据行不存在，则直接返回
			return;
		}
		
		ecList.clear();
		Class<ExcelColumn> c2 = ExcelColumn.class;
		for(Field f : fs) {
			// 检查
			ExcelColumn ec = f.getAnnotation(c2);
			if (ec == null) continue;
			ecList.add(ec);
		}
		
		if (ecList.isEmpty()) {
			msgs.add("数据规则定义类未包含ExcelColumn注解（序号：" + this.clazz.getName() + "）");
		}
		
		// ecList按照注解的index值排序
		ecList.sort(new Comparator<ExcelColumn>() {
			@Override
			public int compare(ExcelColumn o1, ExcelColumn o2) {
				if (o1 == null && o2 == null) return 0;
				if (o1 == null) return -1;
				if (o2 == null) return 1;
				
				if (o1.index() < o2.index()) {
					return -1;
				}
				if (o1.index() == o2.index()) {
					return 0;
				}
				
				return 1;
			}			
		});
	}

	/**
	 * 检查Excel数据
	 * 
	 * @param msgs 检查结果消息列表
	 */
	private void checkData(List<String> msgs) {
		// 定位开始行
		er.toRow(fromRow);
		
		int lineNo = fromRow + 1;
		while(true) {
			// 检查指定行数据
			checkDataLine(lineNo, msgs);
			
			// 检查下一行
			boolean bl = er.nextRow();
			if (!bl) {
				// 已经到了最后一行
				break;
			}
			lineNo++;
		}
	}

	/**
	 * 检查指定行数据
	 * 
	 * @param lineNo Excel行，从1开始
	 * @param msgs 检查结果消息列表
	 */
	private void checkDataLine(int lineNo, List<String> msgs) {
		for(int i = 0; i < this.ecList.size(); i++) {
			ExcelColumn ec = ecList.get(i);
			
			er.toColumn(ec.index());
			
			// 数据类型
			CellType ct = ec.type();
			String typeName = getCellTypeName(ct);
			Object value = er.readCell();
			String s = (value == null ? null : String.valueOf(value));
			
			if (s == null || s.trim().length() == 0) {
				// 是否必填
				if (ec.required()) {
					msgs.add("（行：" +  lineNo + "，列：" + (ec.index() + 1) + "）要求必填");
				}
				// 空数据，后面的检查都忽略
				continue;
			}
			
			boolean okType = checkType(ct, value);
			if (!okType) {
				// 类型不对
				msgs.add("（行：" +  lineNo + "，列：" + (ec.index() + 1) + "）类型必须是" + typeName);
			}
			
			// 数值检查
			if (value instanceof Double) {
				double d = ((Double) value).doubleValue();
				if (d < ec.min() || d > ec.max()) {
					msgs.add("（行：" +  lineNo + "，列：" + (ec.index() + 1) + "）数值必须在 [" + ec.min() + ", " + ec.max() + "] 范围内");
				}
			}
		} // end for
	}

	/**
	 * 检查指定的数据是否是指定的Excel类型
	 * 
	 * @param ct 指定的Excel类型
	 * @param value 指定的数据
	 * @return 指定的数据是否是指定的Excel类型
	 */
	private boolean checkType(CellType ct, Object value) {
		if (ct == CellType.BOOLEAN) {
			return value instanceof Boolean;
		}		
		if (ct == CellType.NUMERIC) {
			return value instanceof Double;
		}
		if (ct == CellType.STRING) {
			return value instanceof String;
		}
		
		return true;
	}

	/**
	 * @param ct Excel单元格数据类型
	 * @return Excel单元格数据类型名称
	 */
	private String getCellTypeName(CellType ct) {
		if (ct == CellType.BLANK) {
			return "空白";
		}
		if (ct == CellType.BOOLEAN) {
			return "布尔型";
		}		
		if (ct == CellType.ERROR) {
			return "错误类型";
		}		
		if (ct == CellType.FORMULA) {
			return "公式";
		}
		if (ct == CellType.NUMERIC) {
			return "数值";
		}
		if (ct == CellType.STRING) {
			return "字符";
		}
		
		return "不明类型";
	}
}
