package team.bangbang.common.file;

import java.awt.Dimension;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.FillPatternType;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;


//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：Excel写入
/**
 * 使用Java Excel API写入Excel文档信息
 *
 * @author 帮帮组
 * @version 1.0 2012-05-06
 * 已经废弃，推荐使用alibaba的easyexcel。  2021-04-06
 */
// ************************************************************************
@Deprecated
public class ExcelWriter {
	/* 数据输出流 */
	private OutputStream os = null;
	/* 工作簿 */
	private Workbook workbook = null;
	/* 工作表 */
	private Sheet sheet = null;
	/* 行号 */
	private int rowNo = 0;
	/* 当前行 */
	private Row currentRow = null;
	/* 列号 */
	private int columnNo = 0;
	/* 当前Cell是否已经写内容 */
	private boolean hasWrite = false;

	/**
	 * 构造一个Excel写入对象，如果已经存在文件，则覆盖。
	 *
	 * @param excelFile Excel文件
	 */
	public ExcelWriter(String excelFile) {
		// 新建Workbook
		if (excelFile.trim().toLowerCase().endsWith(".xlsx")) {
			this.workbook = new XSSFWorkbook();
		} else {
			this.workbook = new HSSFWorkbook();
		}
		
		try {
			this.os = new FileOutputStream(excelFile);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 构造一个Excel写入对象，并预先从Excel模板中读取格式。
	 *
	 * @param templateFile Excel模板文件文件
	 * @param excelFile Excel文件
	 */
	public ExcelWriter(String templateFile, String excelFile) {		
		if (templateFile != null && templateFile.trim().length() > 0) {			
			// 读取模板文件
			InputStream is = null;
			try {
				// 扩展名
				String ext1 = FileUtil.getFileExtension(templateFile);
				String ext2 = FileUtil.getFileExtension(templateFile);
				if (!ext1.equalsIgnoreCase(ext2)) {
					throw new Exception("模板Excel文件扩展名和目标Excel扩展名必须一致！");
				}
				
				is = new FileInputStream(templateFile);
				
				if (excelFile.trim().toLowerCase().endsWith(".xlsx")) {
					this.workbook = new XSSFWorkbook(is);
				} else {
					this.workbook = new HSSFWorkbook(is);
				}
			} catch (Exception ex) {
				ex.printStackTrace();
			} finally {
				try {
					if (is != null)
						is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		
		if (this.workbook == null) {
			// 新建Workbook
			if (excelFile.trim().toLowerCase().endsWith(".xlsx")) {
				this.workbook = new XSSFWorkbook();
			} else {
				this.workbook = new HSSFWorkbook();
			}
		}

		try {
			this.os = new FileOutputStream(excelFile);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 建立工作表
	 *
	 * @param strName 工作表名称
	 */
	public void createSheet(String strName) {
		this.sheet = workbook.createSheet(strName);
		workbook.setActiveSheet(workbook.getSheetIndex(strName));

		// 行号、列号归零
		rowNo = 0;
		columnNo = 0;

		boolean hasRow = (rowNo < sheet.getLastRowNum());
		if (hasRow) {
			currentRow = sheet.getRow(rowNo);
		} else {
			currentRow = sheet.createRow(rowNo);
		}

		// 当前Cell未写内容
		hasWrite = false;
	}

	/**
	 * 建立工作表
	 *
	 * @param strName 工作表名称
	 * @param index 工作表序号，从0开始
	 */
	public void createSheet(String strName, int index) {
		createSheet(strName);
		workbook.setSheetOrder(strName, index);
	}

	/**
	 * 设定当前工作簿
	 *
	 * @param nIndex 工作簿序号，以0开始
	 */
	public void setSheetIndex(int nIndex) {
		this.sheet = workbook.getSheetAt(nIndex);

		// 行号、列号归零
		rowNo = 0;
		columnNo = 0;

		boolean hasRow = (rowNo < sheet.getLastRowNum());
		if (hasRow) {
			currentRow = sheet.getRow(rowNo);
		} else {
			currentRow = sheet.createRow(rowNo);
		}

		// 当前Cell未写内容
		hasWrite = false;
	}

	/**
	 * 游标转移到下一行
	 *
	 * @return 此返回值无意义，仅与ExcelReader中方法保持相同规格。
	 */
	public boolean nextRow() {
		// 行号加1
		rowNo++;
		// 列号归零
		columnNo = 0;

		// 已经到最后一行
		boolean hasRow = (rowNo < sheet.getLastRowNum());
		if (hasRow) {
			currentRow = sheet.getRow(rowNo);
		} else {
			currentRow = sheet.createRow(rowNo);
		}

		// 当前Cell未写内容
		hasWrite = false;

		return true;
	}

	/**
	 * 跳到指定的行
	 *
	 * @param nRow 指定的行，based zero
	 */
	public void toRow(int nRow) {
		rowNo = nRow;
		// 列号归零
		columnNo = 0;

		// 已经到最后一行
		boolean hasRow = (rowNo < sheet.getLastRowNum());
		if (hasRow) {
			currentRow = sheet.getRow(rowNo);
		} else {
			currentRow = sheet.createRow(rowNo);
		}

		// 当前Cell未写内容
		hasWrite = false;
	}

	/**
	 * 跳到指定的列
	 *
	 * @param nColumn 指定的列，based zero
	 */
	public void toColumn(int nColumn) {
		columnNo = nColumn;

		boolean hasRow = (rowNo < sheet.getLastRowNum());
		if (hasRow) {
			currentRow = sheet.getRow(rowNo);
		} else {
			currentRow = sheet.createRow(rowNo);
		}

		// 当前Cell未写内容
		hasWrite = false;
	}

	/**
	 * 获得当前游标的偏移量：&lt;N列, N行&gt;
	 *
	 * @return &lt;N列, N行&gt;
	 */
	public Dimension getOffset() {
		return new Dimension(hasWrite ? columnNo - 1 : columnNo, rowNo);
	}

	/**
	 * 合并单元格。先merge()，次writeCell()，再setRowHeight()/setColumnWidth()。
	 *
	 * @param offset1 起始偏移量：列、行 based zero
	 * @param offset2 终止偏移量：列、行 based zero
	 */
	public void merge(Dimension offset1, Dimension offset2) {
		// 4个参数：起始行，结束行，起始列，结束列
		CellRangeAddress region = new CellRangeAddress(offset1.height, offset2.height, offset1.width, offset2.width);
		sheet.addMergedRegion(region);
	}

	/**
	 * 设定当前Cell的文本内容。。
	 *
	 * @param content 内容
	 */
	public void writeCell(Object content) {
		if (content == null)
			content = "";
		currentRow.createCell(columnNo++).setCellValue(content.toString());

		// 当前Cell已写内容
		hasWrite = true;
	}

	/**
	 * 设置标题样式
	 *
	 * @param offset1 起始偏移量：列、行 based zero
	 * @param offset2 终止偏移量：列、行 based zero
	 */
	public void setHeader(Dimension offset1, Dimension offset2) {
		int x1 = offset1.width;
		int y1 = offset1.height;
		int x2 = offset2.width;
		int y2 = offset2.height;

		// 取出所有的cell，设置单元格样式
		Font f = this.workbook.createFont();
		// 加粗
		f.setBold(true);
		f.setColor(IndexedColors.WHITE.index);

		for (int i = y1; i <= y2; i++) {
			for (int j = x1; j <= x2; j++) {
				CellStyle cs = this.workbook.createCellStyle();
				cs.setFont(f);
				// cell
				Cell c = sheet.getRow(i).getCell(j);
				if (c == null) {
					sheet.getRow(i).createCell(j);
					c = sheet.getRow(i).getCell(j);
				}
				c.setCellStyle(cs);
				// 背景颜色
				cs.setFillForegroundColor(IndexedColors.BLUE_GREY.index);
				cs.setFillPattern(FillPatternType.SOLID_FOREGROUND);

				// 表格线
				// 靠上？
				if (i == y1) {
					cs.setBorderTop(BorderStyle.THIN);
					cs.setTopBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderTop(BorderStyle.DOTTED);
					cs.setTopBorderColor(IndexedColors.DARK_BLUE.index);
				}
				// 靠右？
				if (j == x2) {
					cs.setBorderRight(BorderStyle.THIN);
					cs.setRightBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderRight(BorderStyle.DOTTED);
					cs.setRightBorderColor(IndexedColors.DARK_BLUE.index);
				}
				// 靠下？
				if (i == y2) {
					cs.setBorderBottom(BorderStyle.THIN);
					cs.setBottomBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderBottom(BorderStyle.DOTTED);
					cs.setBottomBorderColor(IndexedColors.DARK_BLUE.index);
				}
				// 靠左？
				if (j == x1) {
					cs.setBorderLeft(BorderStyle.THIN);
					cs.setLeftBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderLeft(BorderStyle.DOTTED);
					cs.setLeftBorderColor(IndexedColors.DARK_BLUE.index);
				}

				// 居中显示
				cs.setAlignment(HorizontalAlignment.CENTER);
				cs.setVerticalAlignment(VerticalAlignment.CENTER);
			} // end for
		}
	}

	/**
	 * 设置页尾样式
	 *
	 * @param offset1 起始偏移量：列、行 based zero
	 * @param offset2 终止偏移量：列、行 based zero
	 */
	public void setFooter(Dimension offset1, Dimension offset2) {
		int x1 = offset1.width;
		int y1 = offset1.height;
		int x2 = offset2.width;
		int y2 = offset2.height;

		// 取出所有的cell，设置单元格样式
		Font f = this.workbook.createFont();
		// 加粗
		f.setBold(true);
		f.setColor(IndexedColors.DARK_RED.index);

		// 取出所有的cell，设置单元格样式
		for (int i = y1; i <= y2; i++) {
			for (int j = x1; j <= x2; j++) {
				CellStyle cs = this.workbook.createCellStyle();
				cs.setFont(f);
				// cell
				Cell c = sheet.getRow(i).getCell(j);
				if (c == null) {
					sheet.getRow(i).createCell(j);
					c = sheet.getRow(i).getCell(j);
				}
				c.setCellStyle(cs);

				// 背景颜色
				cs.setFillForegroundColor(IndexedColors.GREY_25_PERCENT.index);
				cs.setFillPattern(FillPatternType.SOLID_FOREGROUND);

				// 表格线
				// 靠上？
				if (i == y1) {
					cs.setBorderTop(BorderStyle.THIN);
					cs.setTopBorderColor(IndexedColors.GREY_50_PERCENT.index);
				} else {
					cs.setBorderTop(BorderStyle.DOTTED);
					cs.setTopBorderColor(IndexedColors.GREY_80_PERCENT.index);
				}
				// 靠右？
				if (j == x2) {
					cs.setBorderRight(BorderStyle.THIN);
					cs.setRightBorderColor(IndexedColors.GREY_50_PERCENT.index);
				} else {
					cs.setBorderRight(BorderStyle.DOTTED);
					cs.setRightBorderColor(IndexedColors.GREY_80_PERCENT.index);
				}
				// 靠下？
				if (i == y2) {
					cs.setBorderBottom(BorderStyle.THIN);
					cs.setBottomBorderColor(IndexedColors.GREY_50_PERCENT.index);
				} else {
					cs.setBorderBottom(BorderStyle.DOTTED);
					cs.setBottomBorderColor(IndexedColors.GREY_80_PERCENT.index);
				}
				// 靠左？
				if (j == x1) {
					cs.setBorderLeft(BorderStyle.THIN);
					cs.setLeftBorderColor(IndexedColors.GREY_50_PERCENT.index);
				} else {
					cs.setBorderLeft(BorderStyle.DOTTED);
					cs.setLeftBorderColor(IndexedColors.GREY_80_PERCENT.index);
				}

				// 居中显示
				cs.setAlignment(HorizontalAlignment.CENTER);
				cs.setVerticalAlignment(VerticalAlignment.CENTER);
			} // end for
		}
	}

	/**
	 * 设置边框样式
	 *
	 * @param offset1 起始偏移量：列、行 based zero
	 * @param offset2 终止偏移量：列、行 based zero
	 */
	public void setBorder(Dimension offset1, Dimension offset2) {
		int x1 = offset1.width;
		int y1 = offset1.height;
		int x2 = offset2.width;
		int y2 = offset2.height;

		// 取出所有的cell，设置单元格样式
		for (int i = y1; i <= y2; i++) {
			for (int j = x1; j <= x2; j++) {
				CellStyle cs = this.workbook.createCellStyle();
				// cell
				Cell c = sheet.getRow(i).getCell(j);
				if (c == null) {
					sheet.getRow(i).createCell(j);
					c = sheet.getRow(i).getCell(j);
				}
				c.setCellStyle(cs);

				// 靠上？
				if (i == y1) {
					cs.setBorderTop(BorderStyle.THIN);
					cs.setTopBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderTop(BorderStyle.DOTTED);
					cs.setTopBorderColor(IndexedColors.LIGHT_BLUE.index);
				}
				// 靠右？
				if (j == x2) {
					cs.setBorderRight(BorderStyle.THIN);
					cs.setRightBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderRight(BorderStyle.DOTTED);
					cs.setRightBorderColor(IndexedColors.LIGHT_BLUE.index);
				}
				// 靠下？
				if (i == y2) {
					cs.setBorderBottom(BorderStyle.THIN);
					cs.setBottomBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderBottom(BorderStyle.DOTTED);
					cs.setBottomBorderColor(IndexedColors.LIGHT_BLUE.index);
				}
				// 靠左？
				if (j == x1) {
					cs.setBorderLeft(BorderStyle.THIN);
					cs.setLeftBorderColor(IndexedColors.LIGHT_BLUE.index);
				} else {
					cs.setBorderLeft(BorderStyle.DOTTED);
					cs.setLeftBorderColor(IndexedColors.LIGHT_BLUE.index);
				}
			}
		}
	}

	/**
	 * 设置对齐方式，对其方式可选：left、center、right、justify、top、middle、bottom
	 *
	 * @param offset1 起始偏移量：列、行 based zero
	 * @param offset2 终止偏移量：列、行 based zero
	 * @param align   对其方式，可选：left、center、right、justify、top、middle、bottom
	 */
	public void setAlignment(Dimension offset1, Dimension offset2, String align) {
		if (align == null)
			return;
		align = align.trim().toLowerCase();

		int x1 = offset1.width;
		int y1 = offset1.height;
		int x2 = offset2.width;
		int y2 = offset2.height;

		// 水平对齐方式 {"left", "center", "centre", "right", "justify"};
		HorizontalAlignment ali = null;
		if ("left".equals(align)) {
			ali = HorizontalAlignment.LEFT;
		} else if ("center".equals(align)) {
			ali = HorizontalAlignment.CENTER;
		} else if ("centre".equals(align)) {
			ali = HorizontalAlignment.CENTER;
		} else if ("right".equals(align)) {
			ali = HorizontalAlignment.RIGHT;
		} else if ("justify".equals(align)) {
			ali = HorizontalAlignment.JUSTIFY;
		}

		// 垂直对齐方式 {"top", "middle", "bottom"};
		VerticalAlignment val = null;
		if (ali == null) {
			if ("top".equals(align)) {
				val = VerticalAlignment.TOP;
			} else if ("middle".equals(align)) {
				val = VerticalAlignment.CENTER;
			} else if ("bottom".equals(align)) {
				val = VerticalAlignment.BOTTOM;
			}

			if (val == null) {
				return;
			}
		}

		CellStyle cs = workbook.createCellStyle();
		cs.setAlignment(ali);
		cs.setVerticalAlignment(val);

		// 取出所有的cell，设置单元格样式
		for (int i = x1; i < x2; i++) {
			for (int j = y1; j < y2; j++) {
				// 叠加单元格样式
				CellStyle cs2 = sheet.getRow(i).getCell(j).getCellStyle();
				if (cs2 != null) {
					cs2.setAlignment(ali);
					cs2.setVerticalAlignment(val);
					continue;
				}

				sheet.getRow(i).getCell(j).setCellStyle(cs);
			}
		}
	}

	/**
	 * 设置列宽度。先merge()，次writeCell()，再setRowHeight()/setColumnWidth()。
	 *
	 * @param columnIndex 列号，based zero
	 * @param width       列宽度，单位：一个字符宽度
	 */
	public void setColumnWidth(int columnIndex, int width) {
		sheet.setColumnWidth(columnIndex, width * 256);
	}

	/**
	 * 设置行高。先merge()，次writeCell()，再setRowHeight()/setColumnWidth()。
	 *
	 * @param rowIndex 行号，based zero
	 * @param height   行高，单位：像素
	 */
	public void setRowHeight(int rowIndex, int height) {
		sheet.getRow(rowIndex).setHeightInPoints((float) height);
	}

	/**
	 * 写入并关闭工作簿
	 */
	public void close() {
		try {
			this.workbook.write(this.os);
		} catch (Exception ex) {
			try {
				this.workbook.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
			try {
				this.os.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}
}
