package team.bangbang.common.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * 文件工具类
 *
 * @author Bangbang
 * @version 1.0  2021年1月8日
 */
public class FileUtil {
	/**
	 * 将文件转移到指定的位置，如果目标文件所在的目录不存在，则自动创建。<br>
	 * 移动成功后，原文件将从原位置删除。
	 *
	 * 该移动没有调用File.renameTo()函数，而是使用了文件copy，
	 * 避免在Windows、Linux环境下File.renameTo()功能差异。
	 *
	 * @param src
	 *            原文件
	 * @param dest
	 *            目标文件
	 * @return true:移动成功 false:移动失败
	 */
	public static boolean moveFile(File src, File dest) {
		if(src == null || dest == null) return false;

		// 检查目标文件所在目录是否存在
		File dir = dest.getParentFile();
		if (dir != null && !dir.exists()) {
			dir.mkdirs();
		}

		// 文件复制
		byte[] buff = new byte[512];
		int len = 0;
		// 输入输出流
		InputStream is = null;
		OutputStream os = null;
		try {
			is = new FileInputStream(src);
			os = new FileOutputStream(dest, false);

			int loopTimes = 0;
			while((len = is.read(buff)) > 0) {
				os.write(buff, 0, len);
				loopTimes++;

				if(loopTimes > 0 && loopTimes / 1024 == 0) {
					os.flush();
				}
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		} finally {
			try {
				if(is != null) is.close();
			} catch (Exception ex1) {}

			try {
				if(os != null) os.close();
			} catch (Exception ex2) {}
		}

		return src.delete();
	}

	/**
	 * 得到文件扩展名，该扩展名不带有"."
	 *
	 * @param fileName
	 *            文件名
	 * @return 文件扩展名，不包括.号
	 */
	public static String getFileExtension(String fileName) {
		if (fileName == null) {
			return "";
		}

		int nIndex = fileName.lastIndexOf(".");
		return (nIndex >= 0) ? fileName.substring(nIndex + 1) : "";
	}
}
