package team.bangbang.common.file;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import team.bangbang.common.exception.BizException;
import team.bangbang.common.utility.LogicUtility;

/**
 * 加载资源文件，传入调用者的类型和资源文件，获得资源文件输入流。
 *
 * 注意此处的资源文件必须在调用者所在的包下面。
 *
 * @author 帮帮组
 * @version 1.0 Apr 10, 2012
 */
public class ResourceLoader {
	/**
	 * 将WEB应用中相对路径的文件变为输入流。
	 *
	 * @param resource
	 *            相对路径
	 * @return 文件变输入流
	 * @throws IOException IO异常
	 */
	public static InputStream getResourceAsStream(String resource)
			throws IOException {
		String stripped = resource.startsWith("/") ? resource.substring(1)
				: resource;

		InputStream stream = null;
		ClassLoader classLoader = Thread.currentThread()
				.getContextClassLoader();
		if (classLoader != null) {
			stream = classLoader.getResourceAsStream(stripped);
		}
		if (stream == null) {
			stream = LogicUtility.class.getResourceAsStream(resource);
		}
		if (stream == null) {
			stream = LogicUtility.class.getClassLoader().getResourceAsStream(
					stripped);
		}
		if (stream == null) {
			throw new IOException(resource + " not found");
		}
		return stream;
	}
	
	/**
	 * 传入调用者的类型和资源文件，获得资源文件输入流
	 *
	 * @param cls          调用者的类型
	 * @param resourceFile 资源配置文件
	 * @return 资源文件输入流
	 * @throws BizException IO异常
	 */
	public static InputStream getInputStream(Class<?> cls, String resourceFile) throws BizException {
		// 调用者所在的包
		String pkg = (cls != null ? cls.getPackage().getName() : null);

		// 注意：不管是windows下，还是linux下，在jar包中资源的目录层次必须全部用分隔符“/”
		if (pkg != null && pkg.length() > 0) {
			// 添加包路径
			resourceFile = pkg.replace('.', '/') + '/' + resourceFile;
		}

		resourceFile = resourceFile.replaceAll("^/+", "");

		InputStream is = Thread.currentThread().getContextClassLoader().getResourceAsStream(resourceFile);

		if (is == null) {
			throw new BizException("File (" + resourceFile + ") is not exist.");
		}

		return is;
	}

	/**
	 * 传入调用者的类型和资源文件，获得资源文件全路径
	 *
	 * @param cls          调用者的类型
	 * @param resourceFile 资源配置文件
	 * @return 得资源文件全路径
	 *
	 * @throws BizException 文件未找到
	 */
	public static String getPath(Class<?> cls, String resourceFile) throws BizException {
		// 调用者所在的包
		String pkg = cls.getPackage().getName();
		if (pkg != null && pkg.length() > 0) {
			// 添加包路径
			resourceFile = pkg.replace('.', '/') + '/' + resourceFile;
		}

		URL url = Thread.currentThread().getContextClassLoader().getResource(resourceFile);

		if (url == null) {
			throw new BizException("File (" + resourceFile + ") is not exist.");
		}

		return url.getFile();
	}
}
