package team.bangbang.common.file;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

/**
 * 序列化、反向序列化
 *
 * @author 帮帮组
 * @version 1.0 2012-7-26
 */
public final class Serialization {
    /**
	 * 将实例对象序列化为字节数组
	 *
	 * @param obj
	 *            实例对象
	 * @return 实例对象序列化后的字节数组
	 * @throws IOException IO异常
	 */
	public static byte[] serialize(Serializable obj)
			throws IOException {
		// 参数校验
		if (obj == null) {
			return null;
		}

		// 对象输出流
		ObjectOutputStream oos = null;
		try {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			oos = new ObjectOutputStream(baos);
			// 写入实例对象
			oos.writeObject(obj);
			oos.flush();

			byte[] datas = baos.toByteArray();
			return datas;
		} catch (IOException ioex) {
			throw ioex;
		} finally {
			if (oos != null) {
				oos.close();
			}
		} // end try
	}

	/**
	 * 将实例对象序列化保存到指定的文件中
	 *
	 * @param obj
	 *            实例对象
	 * @param file
	 *            存储文件
	 * @throws IOException IO异常
	 */
	public static void serialize(Serializable obj, String file)
			throws IOException {
		// 参数校验
		if (obj == null || file == null || file.length() == 0) {
			return;
		}

		// 检查文件夹是否存在
		File f = new File(file);
		// 文件夹
		File folder = f.getParentFile();
		if (folder != null && !folder.exists()) {
			folder.mkdirs();
		}

		// 对象输出流
		ObjectOutputStream oos = null;
		try {
			oos = new ObjectOutputStream(new FileOutputStream(f));
			// 写入实例对象
			oos.writeObject(obj);
			oos.flush();
		} catch (IOException ioex) {
			throw ioex;
		} finally {
			if (oos != null)
				oos.close();
		} // end try
	}

	/**
	 * 指定的字节数组反序列化产生实例对象
	 *
	 * @param datas
	 *            指定的字节数组
	 * @return 实例对象
	 * @throws IOException IO异常
	 */
	public static Serializable unserialize(byte[] datas) throws IOException {
		// 参数校验
		if (datas == null || datas.length == 0) {
			return null;
		}

		// 对象输入流
		ObjectInputStream ois = null;
		// 结果对象
		Serializable obj = null;
		try {
			ByteArrayInputStream bais = new ByteArrayInputStream(datas);
			ois = new ObjectInputStream(bais);
			// 写入实例对象
			obj = (Serializable) ois.readObject();
		} catch (IOException ioex) {
			throw ioex;
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} finally {
			if (ois != null)
				ois.close();
		} // end try

		return obj;
	}
}
