package team.bangbang.common.filter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import team.bangbang.common.config.Config;
import team.bangbang.common.data.KeyValue;
import team.bangbang.common.utility.LogicUtility;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：权限Filter
/**
 * 防DDOS过滤器
 *
 * 本过滤器应该覆盖应用入口、应用服务。
 *
 * 参数名称：
 *
 * <pre>
 * enter-uri	(必选，不含contextPath)
 * 入口地址，如/sample/request.jsp，多个入口地址使用换行间隔
 *
 * referer-prefix	(可选，设定则校验入口地址以外的请求referer)
 * referer前缀，多个referer前缀使用换行间隔。除了入口地址外，其它请求referer必须以[referer_prefix]内的某个值开头
 *
 * second-limit	(可选，[second_limit]、[visit_limit]必须全部大于0才能生效)
 * 限制的秒数
 *
 * visit-limit	(可选，[second_limit]、[visit_limit]必须全部大于0才能生效)
 * 指定时间[second_limit]内限制的访问次数
 *
 * deny-message (可选)
 * 拒绝的返回消息，缺省为“{errcode:174000, errcode:'Your are welcome!'}”
 * </pre>
 *
 * 工作机理如下：
 *
 * <pre>
 * 1. 在程序入口设置Session，Session中保存2 KEY：SessionId、浏览器的user-agent；
 * 2. 在程序入口之外的应用服务中，检查2 KEY是否与程序入口保存的2 KEY相同；
 * 3. 针对本过滤器覆盖的访问，单个Session请求在指定的[second_limit]秒时间内，只允许执行[visit_limit]次访问
 * </pre>
 *
 * @author 帮帮组
 * @version 1.0 2017-04-02 紧张了这么长时间，清明假期第一天，春光明媚！！
 * @version 1.1 2021-01-28 在南瑞路8号整理培训材料，顺便添加一下filter的启用参数
 */
// ************************************************************************
@WebFilter(filterName="DDosFilter",urlPatterns="/ddos/*")
public class DDosFilter implements Filter {
	/* Session中的DDOS身份KEY */
	private final static String KEY_DDOS_IDENTITY = "KEY_DDOS_IDENTITY";
	/* Session中近期的访问记录 */
	private final static String KEY_DDOS_VISIT = "KEY_DDOS_VISIT";
	/* 当前filter是否启用，默认为false  */
	private final static boolean enable = Config.getProperty("filter.DDosFilter.enable") != null && Config.getProperty("filter.DDosFilter.enable").trim().equalsIgnoreCase("true");
	/* 入口地址，如/sample/request.jsp，多个入口地址使用换行间隔 */
	private String[] enter_uris = null;
	/* referer前缀，多个referer前缀使用换行间隔 */
	private String[] referer_prefixs = null;
	/* 限制的秒数 */
	private int second_limit = 0;
	/* 指定时间[second_limit]内限制的访问次数 */
	private int visit_limit = 0;
	/* 拒绝的返回消息，缺省为“{errcode:174000, errcode:'Your are not welcome!'}” */
	private String deny_message = Config.getProperty("filter.DDosFilter.parameter.deny-message");
	private String s1 = Config.getProperty("filter.DDosFilter.parameter.enter-uri");
	private String s2 = Config.getProperty("filter.DDosFilter.parameter.second-limit");
	private String s3 = Config.getProperty("filter.DDosFilter.parameter.visit-limit");
	private String s4 = Config.getProperty("filter.DDosFilter.parameter.referer-prefix");

	// ***************************************************************************
	/**
	 * 初始化方法
	 *
	 * @param argFilterConfig
	 *            过滤配置信息
	 * @exception ServletException
	 *                Servlet 异常
	 *
	 * @see Filter#init(FilterConfig)
	 */
	// ***************************************************************************
	public void init(FilterConfig argFilterConfig) throws ServletException {
		// 读取权限Filter的配置参数
		if (s1 != null && s1.trim().length() > 0) {
			s1 = s1.replace(',', ' ');
			// 过滤掉多余的空格
			enter_uris = s1.trim().split("\\s+");
		}

		if (s4 != null && s4.trim().length() > 0) {
			s4 = s4.replace(',', ' ');
			// 过滤掉多余的空格
			referer_prefixs = s4.trim().split("\\s+");
		}

		second_limit = LogicUtility.parseInt(s2, 0);

		visit_limit = LogicUtility.parseInt(s3, 0);

		// 拒绝的返回消息，缺省为“{errcode:174000, errcode:'Your are not welcome!'}”
		if(deny_message == null || deny_message.trim().length() == 0) {
			deny_message = "{errcode:174000, errcode:'Your are not welcome!'}";
		} else {
			deny_message = deny_message.trim();
		}
	}

	// ***************************************************************************
	/**
	 * HTTP 请求字符编码过滤处理
	 *
	 * @param argRequest
	 *            ServletRequest
	 * @param argResponse
	 *            ServletResponse
	 * @param argChain
	 *            FilterChain
	 * @exception IOException
	 *                数据流异常
	 * @exception ServletException
	 *                servlet异常
	 *
	 * @see Filter#doFilter(ServletRequest,
	 *      ServletResponse, FilterChain)
	 */
	// ***************************************************************************
	public void doFilter(ServletRequest argRequest, ServletResponse argResponse, FilterChain argChain)
			throws IOException, ServletException {
		if (!enable) {
			// 未启用filter，直接忽略
			argChain.doFilter(argRequest, argResponse);
			return;
		}
		
		HttpServletRequest request = (HttpServletRequest) argRequest;
		// Session信息
		HttpSession session = request.getSession(true);

		// 在同一个Session内
		// 3：检查访问次数限制：单个Session请求在指定的[second_limit]秒时间内，只允许执行[visit_limit]次访问
		// 近期的访问记录
		@SuppressWarnings("unchecked")
		List<KeyValue> vRecords = (List<KeyValue>)session.getAttribute("KEY_DDOS_VISIT");
		if(vRecords == null) {
			vRecords = new ArrayList<KeyValue>();
			session.setAttribute(KEY_DDOS_VISIT, vRecords);
		}

		// 访问地址，如request.getRemoteHost() : 58.212.249.67
		String ip = request.getRemoteHost();

		// 当前访问的地址（contextPath之后的字符串部分），如/sample/request.jsp
		String strURI = request.getRequestURI().trim();
		strURI = strURI.substring(request.getContextPath().length());

		if (isEnterUri(strURI)) {
			if(!canVisit(vRecords, ip)) {
				argResponse.getWriter().println(deny_message);
				return;
			}

			// 属于入口地址
			// 0：初始化
			// 在Session中保存2 KEY：SessionId、浏览器的user-agent；
			String v = session.getId() + ":" + request.getHeader("user-agent");
			session.setAttribute(KEY_DDOS_IDENTITY, v);
			argChain.doFilter(argRequest, argResponse);
			return;
		}

		// 1：检查referer
		String referer = request.getHeader("referer");
		if(!isValidReferer(referer)) {
			argResponse.getWriter().println(deny_message);
			return;
		}

		// 2：检查2 KEY是否一致，保证在同一个session中
		String v1 = session.getId() + ":" + request.getHeader("user-agent");
		String v2 = (String) session.getAttribute(KEY_DDOS_IDENTITY);
		if (!v1.equals(v2)) {
			// 不在同一个Session内
			argResponse.getWriter().println(deny_message);
			return;
		}

		if(!canVisit(vRecords, ip)) {
			argResponse.getWriter().println(deny_message);
			return;
		}

		argChain.doFilter(argRequest, argResponse);
	}

	/**
	 * 检查访问的地址是否输入入口地址
	 *
	 * @param strURI
	 *            访问的地址（context之后的部分，以/开始）
	 * @return 是否允许进入
	 */
	private boolean isEnterUri(String strURI) {
		if (strURI == null || strURI.length() == 0 || enter_uris == null || enter_uris.length == 0) {
			// 未配置入口地址
			return false;
		}

		// 该地址是否在入口地址中
		for (String s : enter_uris) {
			if (strURI.equals(s)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * 当前页面的Referer是否属于合法的Referer
	 * @param referer 引用地址
	 * @return 是有效的引用地址
	 */
	private boolean isValidReferer(String referer) {
		// 如果没有设置referer_prefix，则不作校验，全部返回true
		if(referer_prefixs == null || referer_prefixs.length == 0) {
			return true;
		}

		if(referer == null || referer.trim().length() == 0) {
			return false;
		}

		referer = referer.trim();

		for (String s : referer_prefixs) {
			if (referer.startsWith(s)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * 单个Session请求在指定的[second_limit]秒时间内，只允许执行[visit_limit]次访问。检查是否在允许范围内
	 *
	 * @param vRecords 指定的[second_limit]秒时间内访问的记录，内中元素Key为访问时间，Value为访问的IP
	 * @param ip 当前访问的IP
	 * @return 是否在允许访问的范围内
	 */
	private boolean canVisit(List<KeyValue> vRecords, String ip) {
		// [second_limit]、[visit_limit]必须全部大于0才能生效
		if(second_limit <= 0 || visit_limit <= 0) {
			return true;
		}

		if(vRecords == null) {
			return false;
		}

		// 当前访问
		// 内中元素Key为访问时间，Value为访问的IP
		KeyValue kv = new KeyValue(System.currentTimeMillis(), ip);

		vRecords.add(0, kv);

		// 当前时间
		long time = System.currentTimeMillis();
		// [second_limit]时间点
		time -= second_limit * 1000;

		// 清除[second_limit]时间点以前的访问记录
		while(vRecords.size() > 0) {
			// 尾部元素序号
			int end = vRecords.size() - 1;
			kv = vRecords.get(end);

			if(kv == null) {
				vRecords.remove(end);
				continue;
			} else if((Long)kv.getKey() < time) {
				vRecords.remove(end);
				continue;
			} else {
				// 没有清除了，跳出
				break;
			}
		}

		return (vRecords.size() <= visit_limit);
	}

	// ***************************************************************************
	/**
	 * 销毁处理方法
	 *
	 * @see Filter#destroy()
	 */
	// ***************************************************************************
	public void destroy() {
	}
}
