package team.bangbang.common.filter;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.CommonMPI;
import team.bangbang.common.config.Config;
import team.bangbang.common.config.Constants;
import team.bangbang.common.data.StatusCode;
import team.bangbang.common.data.ThreadVariable;
import team.bangbang.common.data.response.ResponseBase;
import team.bangbang.sso.IFunctionLimitSSO;
import team.bangbang.sso.SSOContext;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：单点登录Filter
/**
 * 1. 根据HTTP请求对象生成单点登录处理对象，并将单点登录处理对象放入线程上下文中。
 *
 * 2. 检查当前访问是否有权限。
 *
 * 该Filter通过web.xml加载生效
 *
 * 参数名称：login-url<br>
 * 没有登录的账户信息，需要转向的目标登录页面，可以是sso登录页面
 *
 * 参数名称：no-validation-modules<br>
 * 免除路径访问校验的模块(以斜线开始)，可以使用通配符
 *
 * 参数名称：no-validation-urls<br>
 * 免除路径访问校验的地址(以斜线开始)，可以使用通配符
 *
 * @author Bangbang
 * @version 1.0 2020-12-16
 * @version 1.1 2020-12-22  修改SSOClientFilter为SSOFilter
 * @version 1.2 2021-01-28 在南瑞路8号整理培训材料，顺便添加一下filter的启用参数
 */
// ************************************************************************
@WebFilter(filterName = "SSOFilter", urlPatterns = "/*")
public class SSOFilter implements Filter {
	/* 当前filter是否启用，默认为false  */
	private final static boolean enable = Config.getProperty("filter.SSOFilter.enable") != null && Config.getProperty("filter.SSOFilter.enable").trim().equalsIgnoreCase("true");
	/* 免校验模块，可以使用通配符 */
	private static String[] no_validation_modules = null;
	/* 免校验地址 */
	private static String[] no_validation_urls = null;
	/* 没有登录的账户信息，需要转向的目标登录页面，可以是sso登录页面 */
	private static String login_url = null;

	// ***************************************************************************
	/**
	 * 初始化方法
	 *
	 * @param argFilterConfig 过滤配置信息
	 * @exception ServletException Servlet 异常
	 *
	 * @see Filter#init(FilterConfig)
	 */
	// ***************************************************************************
	public void init(FilterConfig argFilterConfig) throws ServletException {
		// 免校验模块，可以使用通配符
		String modules = argFilterConfig.getInitParameter("no-validation-modules");
		if (modules == null || modules.trim().length() == 0) {
			// 以web.xml中的配置参数优先
			modules = Config.getProperty("sso.function-limit.no-validation-modules");
		}
		
		// 免校验地址
		String urls = argFilterConfig.getInitParameter("no-validation-urls");
		if (urls == null || urls.trim().length() == 0) {
			// 以web.xml中的配置参数优先
			urls = Config.getProperty("sso.function-limit.no-validation-urls");
		}

		// 没有登录的账户信息，需要转向的目标登录页面
		String slogin_url = argFilterConfig.getInitParameter("login-url");
		if (slogin_url  == null || slogin_url.trim().length() == 0) {
			login_url = Config.getProperty("sso.function-limit.login-url");
		} else {
			// 以web.xml中的配置参数优先
			login_url = slogin_url;
		}

		if (modules != null && modules.trim().length() > 0) {
			no_validation_modules = modules.replaceAll("\\s+",  "").split(",");
		}

		if (no_validation_modules == null) {
			no_validation_modules = new String[0];
		}

		if (urls != null && urls.trim().length() > 0) {
			no_validation_urls = urls.replaceAll("\\s+",  "").split(",");
		}

		if (no_validation_urls == null) {
			no_validation_urls = new String[0];
		}
	}

	// ***************************************************************************
	/**
	 * 生成SSO对象，并放入上下文中，以备业务系统使用
	 *
	 * @param argRequest  ServletRequest
	 * @param argResponse ServletResponse
	 * @param argChain    FilterChain
	 * @exception IOException      数据流异常
	 * @exception ServletException servlet异常
	 *
	 * @see Filter#doFilter(ServletRequest,
	 *      ServletResponse, FilterChain)
	 */
	// ***************************************************************************
	public void doFilter(ServletRequest argRequest, ServletResponse argResponse, FilterChain argChain)
			throws IOException, ServletException {
		// 清除原有数据
		ThreadVariable.removeVariable(Constants.KEY_SSO_ACCOUNT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_APPLICATION);
		ThreadVariable.removeVariable(Constants.KEY_SSO_DATA_LIMIT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_FUNCTION_LIMIT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_TOKEN);
		ThreadVariable.removeVariable(SSOContext.KEY_HTTP_SERVLET_REQUEST);
		ThreadVariable.removeVariable(SSOContext.KEY_HTTP_SERVLET_RESPONSE);
		
		if (!enable) {
			// 未启用filter，直接忽略
			argChain.doFilter(argRequest, argResponse);
			return;
		}
		
		// HTTP请求
		HttpServletRequest request = (HttpServletRequest) argRequest;
		// HTTP响应
		HttpServletResponse response = (HttpServletResponse) argResponse;
		// 允许通过header向客户端传递token
		response.addHeader("Access-Control-Allow-Headers", "token");

		// 设置响应字符集
		response.setCharacterEncoding("UTF-8");

		// 创建单点登录上下文
		new SSOContext(request, response);

		// HTTP方法
		String method = request.getMethod();
		if (method.equalsIgnoreCase("OPTIONS")) {
			// 所有预请求，均放行
			// 添加跨域
			String s = response.getHeader("Access-Control-Allow-Origin");
			if (s == null) {
				response.getHeader("access-control-allow-origin");
			}
			if (s == null) {
				response.addHeader("Access-Control-Allow-Origin", "*");
			}
			// 对预请求通知放行
			response.setStatus(HttpServletResponse.SC_OK);
			return;
		}

		// 当前访问的地址（contextPath之后的字符串部分）
		String strURI = CommonMPI.getURI(request);

		// 地址是免除登录校验，直接放行
		if ("/".equals(strURI) || ignoreValidation(strURI)) {
			// 不需要在Filter中校验菜单项的访问权限
			argChain.doFilter(argRequest, argResponse);
			return;
		}

		ResponseBase rb = new ResponseBase();
		// 是否传入了applicationId
		String appId = SSOContext.getApplicationId();
		if (appId == null || appId.trim().length() == 0) {
			rb.setStatusCode(StatusCode.REQUEST_DATA_EXPECTED);
			rb.setMessage("未设置applicationId参数，可以通过HTTP请求传递applicationId参数，也可以在配置文件中添加 sso.application.id 设置。");
			response.getWriter().print(JSONObject.toJSONString(rb));
			return;
		}

		// 其它
		// 检查  功能权限
		IFunctionLimitSSO flClt = SSOContext.getFunctionLimitSSO();
		if (flClt != null && canVisit(flClt, appId)) {
			// 检查通过
			argChain.doFilter(argRequest, argResponse);
			return;
		}

		// 检查不通过
		response.setContentType("application/json");
		addCrossHeader(response);

		rb.setStatusCode(StatusCode.DATA_STATUS_ERROR);
		rb.setMessage("访问URI无权限：" + strURI + "，或者当前登录信息已经失效，请重新登录 " + login_url);
		response.getWriter().print(JSONObject.toJSONString(rb));
	}

	/**
	 * 检查当前请求是否允许访问
	 *
	 * @param flClt 功能权限检查器
	 * @param applicationId 系统/资源编号
	 * @return 当前请求是否允许访问
	 */
	private boolean canVisit(IFunctionLimitSSO flClt, String applicationId) {
		// 是否传递了权限编码
		String code = SSOContext.getHttpRequest().getParameter("code");
		String uri = null;
		if (code != null && code.trim().length() > 0) {
			code = code.trim();
		} else {
			// 获得访问的URI
			uri = SSOContext.getHttpRequest().getParameter("uri");
			if (uri == null || uri.trim().length() == 0) {
				uri = CommonMPI.getURI(SSOContext.getHttpRequest());
			}

			uri = uri.trim();
		}

		return flClt.canVisit(applicationId, code, uri);
	}

	/**
	 * 检查访问的地址是否要在Filter中忽略权限校验
	 *
	 * @param strURI 访问的地址（context之后的部分，以/开始）
	 * @return 是否要忽略权限校验
	 */
	private boolean ignoreValidation(String strURI) {
		// 所有*Select.ext、*Frame.ext结尾的请求都不需要验证
		int nIndex = strURI.lastIndexOf(".");
		String strTemp = strURI;
		if (nIndex > 0) {
			strTemp = strTemp.substring(0, nIndex);
		}
		if (strTemp.endsWith("Select") || strTemp.endsWith("Frame")) {
			return true;
		}

		// 该模块是否免校验的模块中
		for (String m : no_validation_modules) {
			if (m.indexOf("*") < 0) {
				// 没有使用通配符
				if (strURI.startsWith(m)) {
					// 在免校验的模块中
					return true;
				}
			} else {
				// 使用了通配符
				String regx = m.replaceAll("\\.", "\\\\.").replaceAll("\\*", ".+");
				boolean bl = strURI.matches("^" + regx + "$");
				if (bl) {
					return true;
				}
			}
		}

		// 该地址是否在免除校验的地址列表中
		for (String e : no_validation_urls) {
			if (e.indexOf("*") < 0) {
				// 没有使用通配符
				if (strURI.equals(e)) {
					// 在免除校验的地址列表中
					return true;
				}
			} else {
				// 使用了通配符
				String regx = e.replaceAll("\\.", "\\\\.").replaceAll("\\*", ".+");
				boolean bl = strURI.matches("^" + regx + "$");
				if (bl) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * 添加跨域处理
	 *
	 * @param response HTTP响应
	 */
	private void addCrossHeader(HttpServletResponse response) {
		// 添加跨域
		String s = response.getHeader("Access-Control-Allow-Origin");
		if (s == null) {
			response.getHeader("access-control-allow-origin");
		}
		if (s == null) {
			response.addHeader("Access-Control-Allow-Origin", "*");
		}
	}

	// ***************************************************************************
	/**
	 * 销毁处理方法
	 *
	 * @see Filter#destroy()
	 */
	// ***************************************************************************
	public void destroy() {
	}
}
