package team.bangbang.common.net.ftp;

import team.bangbang.common.net.ftp.FtpInformation;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.Socket;
import java.util.StringTokenizer;

/**
 * 处理FTP上传
 *
 * @author 帮帮组
 */
public class FtpClient {
	/* 服务器 */
	private String host = null;
	/* 端口 */
	private int port = 21;
	/* 用户名 */
	private String user = "Anonymous";
	/* 密码 */
	private String password = "";
	/* FTP客户端 */
	private Socket socket = null;
	/* 输入 */
	private BufferedWriter writer = null;
	/* 输出 */
	private BufferedReader reader = null;
	/* 用于向宿主实例发送的上传数据 */
	FtpInformation info = null;

	/**
	 * 构造方法
	 *
	 * @param host
	 *            服务器
	 */
	public FtpClient(String host) {
		this.host = host;
	}

	/**
	 * @param password
	 *            密码
	 */
	public void setPassword(String password) {
		this.password = password;
	}

	/**
	 * @param port
	 *            端口
	 */
	public void setPort(int port) {
		this.port = port;
	}

	/**
	 * @param user
	 *            用户名
	 */
	public void setUser(String user) {
		this.user = user;
	}

	/**
	 * 连接FTP服务器
	 *
	 * @throws IOException IO异常
	 */
	public void connect() throws IOException {
		if (socket != null) {
			throw new IOException(
					"FtpClient is already connected. Disconnect first.");
		}
		socket = new Socket(host, port);

		reader = new BufferedReader(new InputStreamReader(socket
				.getInputStream()));
		writer = new BufferedWriter(new OutputStreamWriter(socket
				.getOutputStream()));

		String response = readLine();
		if (!response.startsWith("220 ")) {
			throw new IOException(
					"FtpClient received an unknown response when "
							+ "connecting to the FTP server: " + response);
		}

		sendLine("USER " + user);

		response = readLine();
		if (!response.startsWith("331 ")) {
			throw new IOException("FtpClient received an unknown response "
					+ "after sending the user: " + response);
		}

		sendLine("PASS " + password);

		response = readLine();
		if (!response.startsWith("230 ")) {
			throw new IOException("FtpClient was unable to log in with "
					+ "the supplied password: " + response);
		}

		// 设定为二进制传输
	}

	/**
	 * 向FTP服务器发送命令行
	 *
	 * @param line
	 *            命令行
	 * @throws IOException
	 */
	private void sendLine(String line) throws IOException {
		if (socket == null) {
			throw new IOException("FtpClient not connected.");
		}

		try {
			writer.write(line + "\r\n");
			writer.flush();
		} catch (IOException e) {
			closeSocket();
			throw e;
		}
	}

	/**
	 * 从服务器读取返回的一行数据
	 *
	 * @return 服务器返回的一行数据
	 * @throws IOException
	 */
	private String readLine() throws IOException {
		return reader.readLine();
	}

	/**
	 * 关闭FTP连接
	 *
	 * @throws IOException IO异常
	 */
	public void disconnect() throws IOException {
		// 发送关闭命令
		try {
			sendLine("QUIT");
		} catch (IOException ioex1) {
			ioex1.printStackTrace();
		}

		// 关闭socket
		closeSocket();
	}

	/**
	 * 关闭Socket
	 *
	 * @throws IOException
	 */
	private void closeSocket() {
		if (socket != null) {
			try {
				socket.close();
			} catch (IOException ioex) {
				ioex.printStackTrace();
			}

			socket = null;
		}
	}

	/**
	 * 返回当前工作目录
	 *
	 * @return 当前工作目录
	 * @throws IOException IO异常
	 */
	public synchronized String pwd() throws IOException {
		sendLine("PWD");
		String dir = null;
		String response = readLine();
		if (response.startsWith("257 ")) {
			int firstQuote = response.indexOf('\"');
			int secondQuote = response.indexOf('\"', firstQuote + 1);
			if (secondQuote > 0) {
				dir = response.substring(firstQuote + 1, secondQuote);
			}
		}

		return dir;
	}

	/**
	 * 改变当前的工作目录
	 *
	 * @param dir
	 *            目标目录
	 * @return true: 改变成功 false: 改变失败
	 * @throws IOException IO异常
	 */
	public synchronized boolean cwd(String dir) throws IOException {
		sendLine("CWD " + dir);
		String response = readLine();
		return (response.startsWith("250 "));
	}

	/**
	 * 改变文件传输方式为二进制方式
	 *
	 * @return true: 改变成功 false: 改变失败
	 * @throws IOException IO异常
	 */
	public synchronized boolean bin() throws IOException {
		sendLine("TYPE I");
		String response = readLine();
		return (response.startsWith("200 "));
	}

	/**
	 * 改变文件传输方式为ASCII文本方式
	 *
	 * @return true: 改变成功 false: 改变失败
	 * @throws IOException IO异常
	 */
	public synchronized boolean ascii() throws IOException {
		sendLine("TYPE A");
		String response = readLine();
		return (response.startsWith("200 "));
	}

	/**
	 * 以passive mode上传文件，此方法上传到FTP服务器上的时候文件名与客户端文件相同。
	 *
	 * @param localFile
	 *            需要上传的本地文件名。
	 * @return true: 上传成功 false: 上传失败
	 * @throws IOException IO异常
	 */
	public synchronized boolean upload(String localFile) throws IOException {
		File file = new File(localFile);
		if (file.isDirectory()) {
			throw new IOException("FtpClient cannot upload a directory.");
		}

		return upload(file, file.getName());
	}

	/**
	 * 把输入流中的数据写到服务器服务器上，使用指定的文件名保存为文件
	 *
	 * @param localFile
	 *            本地文件
	 * @param remoteFileName
	 *            上传到服务器上的目标文件名
	 *
	 * @return true: 上传成功 false: 上传失败
	 * @throws IOException IO异常
	 */
	public synchronized boolean upload(File localFile, String remoteFileName)
			throws IOException {
		info = new FtpInformation();
		info.setFlag(FtpInformation.UPLOAD);
		info.setFileSize(localFile.length());

		// in passive mode to avoid NAT or firewall problems at the client end
		sendLine("PASV");
		String response = readLine();
		if (!response.startsWith("227 ")) {
			throw new IOException("FtpClient could not request passive mode: "
					+ response);
		}

		String dataHost = host;
		int dataPort = port;
		int opening = response.indexOf('(');
		int closing = response.indexOf(')', opening + 1);
		if (closing > 0) {
			String dataLink = response.substring(opening + 1, closing);
			StringTokenizer tokenizer = new StringTokenizer(dataLink, ",");
			try {
				dataHost = tokenizer.nextToken() + "." + tokenizer.nextToken() + "."
						+ tokenizer.nextToken() + "." + tokenizer.nextToken();
				dataPort = Integer.parseInt(tokenizer.nextToken()) * 256
						+ Integer.parseInt(tokenizer.nextToken());
			} catch (Exception e) {
				throw new IOException(
						"FtpClient received bad data link information: "
								+ response);
			}
		}

		// 在服务器上建立文件
		sendLine("STOR " + remoteFileName);
		Socket dataSocket = new Socket(dataHost, dataPort);

		response = readLine();
		if (!response.startsWith("125 ") && !response.startsWith("150 ")) {
			dataSocket.close();
			throw new IOException(
					"FtpClient was not allowed to send the file: " + response);
		}

		// 建立数据通道
		BufferedInputStream input = null;
		BufferedOutputStream output = null;
		try {
			input = new BufferedInputStream(new FileInputStream(localFile));
			output = new BufferedOutputStream(dataSocket.getOutputStream());
			byte[] buffer = new byte[4096];
			int bytesRead = 0;
			while ((bytesRead = input.read(buffer)) != -1) {
				output.write(buffer, 0, bytesRead);
				output.flush();
				// 记录读取的字节数
				info.setCompleteSize(info.getCompleteSize() + bytesRead);
			}
		} catch (Exception ex) {
			throw new IOException(ex.getMessage());
		} finally {
			// 关闭文件输入流
			if (input != null) {
				try {
					input.close();
				} catch (Exception ex1) {
				}
			}
			// 关闭数据输出流
			if (output != null) {
				try {
					output.close();
				} catch (Exception ex2) {
				}
			}
			// 关闭数据通道
			if (dataSocket != null) {
				try {
					dataSocket.close();
				} catch (Exception ex3) {
				}
			}
		} // end try

		response = readLine();
	    return response.startsWith("226 ");
	} // end upload

	/**
	 * FTP过程中Ftp实例向宿主实例汇报的内容，被宿主实例调用。 如果返回值为null，表示当前为断开状态
	 *
	 * @return FTP汇报内容
	 */
	public FtpInformation getFtpInformation() {
		return info;
	}
}
