package team.bangbang.common.net.http;

import java.io.IOException;
import java.nio.charset.Charset;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.ConnectionPool;
import okhttp3.Dispatcher;
import okhttp3.FormBody;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

/**
 * 定义HTTP请求管理相关方法
 *
 * @author 帮帮组
 * @version 1.0 2017年9月26日
 * @version 1.1.2
 * 将客户端builder对象改为单例模式，保证不同实例使用同一个连接池
 * 4月2日了，明天清明节放假，可是近期南京是阴雨天，未来一周气温不超过18°C
 * 
 * 2021年4月2日
 */
public final class HttpClient {
	/* 日志对象 */
	private  final static Logger logger = LoggerFactory.getLogger(HttpClient.class);

	public static final String KEY_CONTENT_TYPE = "Content-Type";
	public static final String DefaultMime = "application/octet-stream";
	public static final String JsonMime = "application/json";
	public static final String XmlMime = "application/xml";
	public static final String FormMime = "application/x-www-form-urlencoded";
	private OkHttpClient httpClient = null;
	/* Header对象 */
	private Map<String, String> headers = new HashMap<String, String>();
	/* 使用的字符集 */
	private Charset charset = Charset.forName("UTF-8");
	/* 客户端builder对象 */
	private static OkHttpClient.Builder builder = null;
	
	/**
	 * 构造一个HttpClient对象
	 */
	public HttpClient() {
		// 获取一个客户端对象
		httpClient = getClient();
	}

	/**
	 * @return 获取一个客户端对象
	 */
	private synchronized OkHttpClient getClient() {
		if (builder != null) {
			// 已经创建了连接池
			return builder.build();
		}

		builder = new OkHttpClient.Builder();

		Dispatcher dispatcher = new Dispatcher();
		dispatcher.setMaxRequests(16);
		dispatcher.setMaxRequestsPerHost(4);

		builder.dispatcher(dispatcher);

		X509TrustManager tm = systemDefaultTrustManager();
		SSLSocketFactory ssf = systemDefaultSslSocketFactory(tm);
		builder.sslSocketFactory(ssf, tm);
		builder.hostnameVerifier(new TrustAllHostnameVerifier());

		// 连接超时时间 单位秒(默认60s)
		builder.connectTimeout(60, TimeUnit.SECONDS);
		// 回复超时时间 单位秒(默认60s)
		builder.readTimeout(60, TimeUnit.SECONDS);
		// 写超时时间 单位秒(默认 60 , 不超时)
		builder.writeTimeout(60, TimeUnit.SECONDS);

		// 5分钟
		ConnectionPool connectionPool = new ConnectionPool(8, 5 * 60, TimeUnit.MINUTES);
		builder.connectionPool(connectionPool);

		builder.networkInterceptors().add(new Interceptor() {
			@Override
			public Response intercept(Chain chain) throws IOException {
				Request request = chain.request();

				Response response = chain.proceed(request);
				MetaTag tag = (MetaTag) request.tag();
				// 响应截止时间
				tag.endTime = System.currentTimeMillis();

				String ip = chain.connection().socket().getRemoteSocketAddress().toString();
				tag.ip = ip;

				ResponseBody rb = response.body();
				MediaType mt = (rb != null ? rb.contentType() : null);
				long contentLength = (rb != null ? rb.contentLength() : 0);

				String s = String.format("URI[{}] ContentType:{} ContentLength:{} Duration:{}ms From:{}",
						response.request().url().uri().toString(), (mt != null ? mt.toString() : null),
						contentLength, tag.getDuration(), tag.ip);

				logger.debug(s);

				return response;
			}
		});			

		return builder.build();
	}

	private String userAgent() {
		String javaVersion = "Java/" + System.getProperty("java.version");
		String os = System.getProperty("os.name") + " " + System.getProperty("os.arch") + " "
				+ System.getProperty("os.version");
		String sdk = "Bangbang_Java";
		return sdk + " (" + os + ") " + javaVersion;
	}

	/**
	 * 设置字符集
	 *
	 * @param charset 字符集
	 */
	public void setCharset(Charset charset) {
		this.charset = charset;
	}

	/**
	 * @return header对象，该对象非空，使用该对象增加、删除http header。
	 */
	public Map<String, String> getHeaders() {
		return headers;
	}

	/**
	 * 发起GET方式请求
	 *
	 * @param url
	 *            请求地址
	 * @return 请求结果
	 * @throws IOException 网络IO异常
	 */
	public ResponseHandler get(String url) throws IOException {
		Request.Builder requestBuilder = new Request.Builder().get().url(url);
		return send(requestBuilder);
	}

	/**
	 * 发起POST方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param body
	 *            请求数据，可以是form data、json或者xml
	 * @param mimeType
	 *            MIME类型，缺省使用application/octet-stream
	 *
	 * @return 请求结果
	 * @throws IOException  网络IO异常
	 */
	public ResponseHandler post(String url, String body, String mimeType) throws IOException {
		byte[] bts = (body != null && body.length() > 0) ? body.getBytes(charset) : new byte[0];

		MediaType t = MediaType.parse(mimeType);
		RequestBody rbody = RequestBody.create(t, bts);

		Request.Builder requestBuilder = new Request.Builder().url(url).post(rbody);
		return send(requestBuilder);
	}

	/**
	 * 发起POST方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param params
	 *            请求参数，form data，使用mime类型为：application/x-www-form-urlencoded
	 *
	 * @return 请求结果
	 * @throws IOException  网络IO异常
	 */
	public ResponseHandler post(String url, Map<String, String> params)
			throws IOException {
		FormBody.Builder f = new FormBody.Builder(charset);
		if (params != null) {
			for (String key : params.keySet()) {
				String value = params.get(key);
				if (value == null) continue;
				f.add(key, value);
			}
		}

		Request.Builder requestBuilder = new Request.Builder().url(url).post(f.build());
		return send(requestBuilder);
	}
	
	/**
	 * 发起PUT方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param body
	 *            请求数据，可以是form data、json或者xml
	 * @param mimeType
	 *            MIME类型，缺省使用application/octet-stream
	 *
	 * @return 请求结果
	 * @throws IOException  网络IO异常
	 */
	public ResponseHandler put(String url, String body, String mimeType) throws IOException {
		byte[] bts = (body != null && body.length() > 0) ? body.getBytes(charset) : new byte[0];

		MediaType t = MediaType.parse(mimeType);
		RequestBody rbody = RequestBody.create(t, bts);

		Request.Builder requestBuilder = new Request.Builder().url(url).put(rbody);
		return send(requestBuilder);
	}
	
	/**
	 * 发起DELETE方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param body
	 *            请求数据，可以是form data、json或者xml
	 * @param mimeType
	 *            MIME类型，缺省使用application/octet-stream
	 *
	 * @return 请求结果
	 * @throws IOException  网络IO异常
	 */
	public ResponseHandler delete(String url, String body, String mimeType) throws IOException {
		byte[] bts = (body != null && body.length() > 0) ? body.getBytes(charset) : new byte[0];

		MediaType t = MediaType.parse(mimeType);
		RequestBody rbody = RequestBody.create(t, bts);

		Request.Builder requestBuilder = new Request.Builder().url(url).delete(rbody);
		return send(requestBuilder);
	}

	private ResponseHandler send(Request.Builder requestBuilder)
			throws IOException {
		if (headers != null) {
			for (String key : headers.keySet()) {
				requestBuilder.header(key, headers.get(key));
			}
		}

		if (!headers.containsKey("User-Agent") && !headers.containsKey("user-agent")) {
			requestBuilder.header("User-Agent", userAgent());
		}

		MetaTag tag = new MetaTag();
		tag.startTime = System.currentTimeMillis();

		Request req = null;
		Response res = null;
		try {
			req = requestBuilder.tag(tag).build();
			res = httpClient.newCall(req).execute();
		} catch (IOException e) {
			logger.error(e.getMessage());
			throw e;
		}

		int httpStatus = res.code();
		if(!res.isSuccessful()) {
			logger.error("请求 " + req.url().encodedPath() + " 失败，HTTP Status：" + httpStatus);
		}

		ResponseBody rb = res.body();
		ResponseHandler rh = new ResponseHandler(httpStatus, res.headers(), rb);

		return rh;
	}

	/**
	 * 异步发起GET方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param cb 回调函数
	 *
	 * @throws IOException  网络IO异常
	 */
	public void asyncGet(String url, AsyncCallback cb) throws IOException {
		Request.Builder requestBuilder = new Request.Builder().get().url(url);
		asyncSend(requestBuilder, cb);
	}

	/**
	 * 异步发起POST方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param body
	 *            请求数据，可以是form data、json或者xml
	 * @param mimeType
	 *            MIME类型，缺省使用application/octet-stream
	 * @param cb 回调函数
	 *
	 * @throws IOException  网络IO异常
	 */
	public void asyncPost(String url, String body, String mimeType, AsyncCallback cb) throws IOException {
		byte[] bts = (body != null && body.length() > 0) ? body.getBytes(charset) : new byte[0];

		MediaType t = MediaType.parse(mimeType);
		RequestBody rbody = RequestBody.create(t, bts);

		Request.Builder requestBuilder = new Request.Builder().url(url).post(rbody);
		asyncSend(requestBuilder, cb);
	}

	/**
	 * 异步发起POST方式请求
	 *
	 * @param url
	 *            请求地址
	 * @param params
	 *            请求参数，form data，使用mime类型为：application/x-www-form-urlencoded
	 * @param cb 回调函数
	 *
	 * @throws IOException  网络IO异常
	 */
	public void asyncPost(String url, Map<String, String> params, AsyncCallback cb)
			throws IOException {
		FormBody.Builder f = new FormBody.Builder(charset);
		if (params != null) {
			for (String key : params.keySet()) {
				f.add(key, params.get(key));
			}
		}

		Request.Builder requestBuilder = new Request.Builder().url(url).post(f.build());
		asyncSend(requestBuilder, cb);
	}

	/**
	 * 异步发起HTTP请求
	 *
	 * @param requestBuilder 请求builder
	 * @param cb 回调函数
	 */
	private void asyncSend(Request.Builder requestBuilder, final AsyncCallback cb) {
		if (headers != null) {
			for (String key : headers.keySet()) {
				requestBuilder.header(key, headers.get(key));
			}
		}

		requestBuilder.header("User-Agent", userAgent());

		final MetaTag tag = new MetaTag();
		tag.startTime = System.currentTimeMillis();

		httpClient.newCall(requestBuilder.tag(tag).build()).enqueue(new Callback() {
			@Override
			public void onFailure(Call call, IOException e) {
				e.printStackTrace();
			}

			@Override
			public void onResponse(Call call, Response response) throws IOException {
				int httpStatus = response.code();
				if(!response.isSuccessful()) {
					logger.error("请求 " + call.request().url().encodedPath() + " 失败，HTTP Status：" + httpStatus);
				}

				ResponseHandler rh = new ResponseHandler(httpStatus, response.headers(), response.body());

				cb.complete(rh);
			}
		});
	}

	private static SSLSocketFactory systemDefaultSslSocketFactory(X509TrustManager xtm) {
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("SSL");

            sslContext.init(null, new TrustManager[]{xtm}, new SecureRandom());

        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (KeyManagementException e) {
            e.printStackTrace();
        }

        return sslContext.getSocketFactory();
	}

	private static X509TrustManager systemDefaultTrustManager() {
		X509TrustManager xtm = new X509TrustManager() {
			@Override
			public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
			}

			@Override
			public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
			}

			@Override
			public X509Certificate[] getAcceptedIssuers() {
                X509Certificate[] x509Certificates = new X509Certificate[0];
                return x509Certificates;
			}
        };

        return xtm;
	}

	private static class TrustAllHostnameVerifier implements HostnameVerifier {
		@Override
		public boolean verify(String hostname, SSLSession session) {
			return true;
		}
	}

	private static class MetaTag {
		/** 开始时间，毫秒 */
		public long startTime = 0L;
		/** 结束时间，毫秒 */
		public long endTime = 0L;
		public String ip = null;

		/**
		 * @return 请求持续的毫秒数
		 */
		public int getDuration() {
			return (int) (endTime - startTime);
		}
	}
}
