package team.bangbang.common.net.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.alibaba.fastjson.JSONObject;

import okhttp3.Headers;
import okhttp3.ResponseBody;
import team.bangbang.common.utility.LogicUtility;

/**
 * Http响应处理
 *
 * @author 帮帮组
 * @version 1.0 2017年9月26日
 */
public class ResponseHandler {
	/* Http响应状态 */
	private int httpStatus = 0;
	/* HTTP响应数据 */
	private ResponseBody rb = null;
	private byte[] data = null;
	private InputStream is = null;
	private String string = null;
	/* 响应头数据 */
	private Map<String, String> headers = new HashMap<String, String>();

	/**
	 * @param httpStatus Http响应状态
	 * @param headers  Http响应头
	 * @param responseBody HTTP响应数据
	 */
	public ResponseHandler(int httpStatus, Headers headers, ResponseBody responseBody) {
		this.httpStatus = httpStatus;
		// 响应头数据
		Set<String> names = (headers != null ? headers.names() : null);
		if (names != null) {
			for (String key : names) {
				String value = headers.get(key);
				if (value == null) continue;
				
				this.headers.put(key, value);
			}
		}
		
		this.rb = responseBody;
	}

	/**
	 * @return HTTP响应数据（输入流）
	 */
	public InputStream toInputStream() {
		if(is == null) {
			is = rb.byteStream();
		}

		return is;
	}

	/**
	 * @return HTTP响应数据（字节数组）
	 */
	public byte[] toBytes() {
		if(data == null) {
			// 使用输入流读取数据，避免响应数据量大的情况下出现unexpect end stream的异常
			InputStream is = null;
			try {
				is = toInputStream();
				data = LogicUtility.readInputStream(is);
			} catch (IOException e) {
				e.printStackTrace();
			} finally {				
				try {
					if (is != null) is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
				if (rb != null) {
					rb.close();
				}
			}
		}

		return data;
	}

	/**
	 * @return HTTP响应数据（字符串）
	 */
	public String toString() {
		if(string == null) {
			// 字节
			byte[] bts = toBytes();
			try {
				string = new String(bts, "UTF-8");
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		return string;
	}

	/**
	 * @return HTTP响应数据（JSON对象）
	 */
	public JSONObject toJSONObject() {
		if(string == null) {
			try {
				string = rb.string();
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				if (rb != null) rb.close();
			}
		}

		return JSONObject.parseObject(string);
	}

	/**
	 * @param <T> 目标Java对象的类
	 * @param clazz Java对象的class类型
	 *
	 * @return HTTP响应数据（Java对象）
	 */
	public <T> T toJavaObject(Class<T> clazz) {
		JSONObject json = toJSONObject();

		return JSONObject.toJavaObject(json, clazz);
	}

	/**
	 * @return Http响应状态
	 */
	public int getHttpStatus() {
		return httpStatus;
	}

	/**
	 * @return 响应头数据
	 */
	public Map<String, String> getHeaders() {
		return headers;
	}
}
