package team.bangbang.common.queue;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import team.bangbang.common.config.Config;
import team.bangbang.common.queue.IQueueManager;
import team.bangbang.common.queue.Publisher;
import team.bangbang.common.queue.Subscriber;
import team.bangbang.common.queue.kafka.KafkaManager;
import team.bangbang.common.queue.rabbit.RabbitManager;
import team.bangbang.common.redis.RedisUtil;

/**
 * 队列代理者，主要有2个作用：
 *
 * <ol>
 * <li>登记订阅者</li>
 * <li>安排生产者</li>
 * </ol>
 *
 * @author 帮帮组
 * @version 1.0  2018年9月30日
 */
public class QueueProxy {
	/* 保存topic下面的订阅者id */
	private static Map<String, Set<String>> subscribers = new HashMap<String, Set<String>>();
	/* topic作为key值的前缀 */
	private static final String KEY_PREFIX = "SUBSCRIBER_UNDER_TOPIC:";
	/* 队列管理者 */
	private IQueueManager manager = null;

	/**
	 * 设置队列管理者
	 */
	public QueueProxy() {
		// 配置文件中，使用了什么队列？
		// 可选项：rabbit、rocket、kafka
		String mq = Config.getProperty("mq.product");
		// 默认使用rabbit
		if(mq == null || mq.trim().length() == 0) {
			mq = "rabbit";
		}
		mq = mq.trim();

		if(mq.equalsIgnoreCase("rabbit")) {
			manager = new RabbitManager();
		}

		if(mq.equalsIgnoreCase("kafka")) {
			manager = new KafkaManager();
		}
	}

	/**
	 * 登记一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 一个队列消费者
	 */
	public boolean subscribe(Subscriber subscriber) {
		if(manager == null) return false;
		// 1. 设置订阅者
		manager.subscribe(subscriber);

		// 2. 保存订阅者
		// 订阅者KEY
		String key = KEY_PREFIX + subscriber.getTopic();
		if(RedisUtil.HAS_REDIS) {
			// 存储在Redis中
			RedisUtil.addIntoSet(key, String.valueOf(subscriber.getIndex()));
		} else {
			Set<String> s = subscribers.get(key);
			if(s == null || s.isEmpty()) {
				s = new HashSet<String>();
				subscribers.put(key, s);
			}
			s.add(String.valueOf(subscriber.getIndex()));
		}

		return false;
	}

	/**
	 * 删除一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 删除是否成功
	 */
	public boolean remove(Subscriber subscriber) {
		if(manager == null) return false;
		// 1. 删除订阅者
		manager.remove(subscriber);

		// 2. 保存订阅者
		// 订阅者KEY
		String key = KEY_PREFIX + subscriber.getTopic();
		if(RedisUtil.HAS_REDIS) {
			// 存储在Redis中
			RedisUtil.removeFromSet(key, String.valueOf(subscriber.getIndex()));
		} else {
			Set<String> s = subscribers.get(key);
			if(s != null && !s.isEmpty()) {
				s.remove(String.valueOf(subscriber.getIndex()));
			}
		}
		return false;
	}

	/**
	 * 创建一个发布者
	 *
	 * @param topic 订阅主题
	 *
	 * @return 一个发布者
	 */
	public Publisher createPublisher(String topic) {
		if(manager == null) return null;

		return manager.createPublisher(topic);
	}

	/**
	 * 获得当前指定topic下订阅者的数量
	 *
	 * @param topic 订阅主题
	 *
	 * @return 当前指定topic下订阅者的数量
	 */
	public static int getSubscriberCount(String topic) {
		// 订阅者KEY
		String key = KEY_PREFIX + topic;
		if(RedisUtil.HAS_REDIS) {
			// 存储在Redis中
			Set<String> s = RedisUtil.getSet(key);
			return (s == null ? 0 : s.size());
		}

		Set<String> s = subscribers.get(key);
		return (s == null ? 0 : s.size());
	}
}
