package team.bangbang.common.queue.kafka;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.config.Config;
import team.bangbang.common.queue.IQueueManager;
import team.bangbang.common.queue.Publisher;
import team.bangbang.common.queue.Subscriber;

/**
 * Kafka队列管理者
 *
 * 今天是国庆节，没有买到回南京的票，在如家酒店码代码呢，外面阳光灿烂、万里无云
 *
 * @author 帮帮组
 * @version 1.0 2018年10月1日
 */
public class KafkaManager implements IQueueManager {
	/* 日志对象 */
	private  final static Logger logger = LoggerFactory.getLogger(KafkaManager.class);
	/* 发布者配置参数 */
	private static Map<String, Object> props_p = new HashMap<String, Object>();
	/* 订阅者配置参数 */
	private static Map<String, Object> props_s = new HashMap<String, Object>();
	/* 管理客户端 */
	private static AdminClient adminClient = null;

	// 使用配置参数构造队列管理者
	static {
		// 1. 发布者配置参数
		// 用于建立与 kafka 集群连接的 host/port 组。
		props_p.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, Config.getProperty("spring.kafka.bootstrap-servers"));
		props_p.put(ProducerConfig.ACKS_CONFIG, "all");
		props_p.put(ProducerConfig.RETRIES_CONFIG, 3);
		props_p.put(ProducerConfig.BATCH_SIZE_CONFIG, 16384);
        props_p.put(ProducerConfig.LINGER_MS_CONFIG, 1);
        props_p.put(ProducerConfig.BUFFER_MEMORY_CONFIG, 33554432);
        props_p.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringSerializer");
        props_p.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringSerializer");

		// 2. 订阅者配置参数
		props_s.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, Config.getProperty("mq.kafka.bootstrap-servers"));

		// Consumer 的 offset 是否自动提交
		props_s.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, "true");
		// 自动提交 offset 到 zookeeper 的时间间隔，时间是毫秒
        props_s.put(ConsumerConfig.AUTO_COMMIT_INTERVAL_MS_CONFIG, "1000");
        // auto.offset.reset: latest, earliest, none
        props_s.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, "latest");
        props_s.put(ConsumerConfig.SESSION_TIMEOUT_MS_CONFIG, "30000");
        props_s.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringDeserializer");
        props_s.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, "org.apache.kafka.common.serialization.StringDeserializer");

		adminClient = AdminClient.create(props_s);
	}

	/**
	 * 获得订阅者所在的订阅组编号
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 订阅者所在的订阅组编号
	 */
	private static String getGroupId(Subscriber subscriber) {
		return subscriber.getTopic() + "_group_" + subscriber.getIndex();
	}

	/**s
	 * 创建一个发布者
	 *
	 * @param topic 订阅主题
	 *
	 * @return 一个发布者
	 */
	public Publisher createPublisher(String topic) {
		Producer<String, String> producer = new KafkaProducer<String, String>(props_p);
		// 创建发布者
		return new KafkaPublisher(producer, topic);
	}

	/**
	 * 登记一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 一个队列消费者
	 */
	public boolean subscribe(final Subscriber subscriber) {
		String topic = subscriber.getTopic();
		// 1. 获得订阅者的订阅组编号
		// 每个订阅组中如果有多个worker，则只会有一个worker收到消息
		String groupId = getGroupId(subscriber);
		props_s.put("group.id", groupId);

		Collection<String> gIds = new HashSet<String>();
		gIds.add(groupId);

		try {
			List<String> topics = new ArrayList<String>();
			topics.add(topic);

			@SuppressWarnings("resource")
			final Consumer<String, String> consumer = new KafkaConsumer<String, String>(props_s);
			consumer.subscribe(topics);

			Duration d = Duration.ofMillis(1000);

			Runnable r = new Runnable() {
				public void run() {

					// 循环消费消息
					while (true) {
						ConsumerRecords<String, String> records = consumer.poll(d);

						// 必须在下次 poll 之前消费完这些数据, 且总耗时不得超过 SESSION_TIMEOUT_MS_CONFIG
						// 建议开一个单独的线程池来消费消息，然后异步返回结果
						for (ConsumerRecord<String, String> record : records) {
							String value = record.value();
							JSONObject json = JSONObject.parseObject(value);

							try {
								subscriber.consume(json);
							} catch (Exception ex) {
								ex.printStackTrace();
								logger.error(ex.getMessage());
								logger.error("队列消费者（" + subscriber.getTopic() + ":" + subscriber.getIndex() + "）因前述异常暂停");
								break;
							}
						}

						try {
							Thread.sleep(10);
						} catch (InterruptedException e) {
							e.printStackTrace();
						}
					}
				}
			};

			Thread t = new Thread(r);
			t.start();
		} catch (Exception ex) {
			ex.printStackTrace();

			return false;
		}

		return true;
	}

	/**
	 * 删除一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 删除是否成功
	 */
	public boolean remove(Subscriber subscriber) {
		if (subscriber == null || adminClient == null) {
			return false;
		}

		// 1. 获得订阅者的订阅组编号
		String groupId = getGroupId(subscriber);

		List<String> groupIds = new ArrayList<String>();
		groupIds.add(groupId);

		adminClient.deleteConsumerGroups(groupIds);

		return true;
	}
}
