package team.bangbang.common.queue.rabbit;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONObject;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Consumer;
import com.rabbitmq.client.DefaultConsumer;
import com.rabbitmq.client.Envelope;

import team.bangbang.common.queue.IQueueManager;
import team.bangbang.common.queue.Publisher;
import team.bangbang.common.queue.Subscriber;
import team.bangbang.common.queue.rabbit.pool.RabbitPool;
import team.bangbang.common.redis.RedisUtil;

/**
 * Rabbit队列管理者
 *
 * @author 帮帮组
 * @version 1.0 2018年9月30日
 */
public class RabbitManager implements IQueueManager {
	/* 日志对象 */
	private  final static Logger logger = LoggerFactory.getLogger(RabbitManager.class);
	/* 保存Exchange的名称集合 */
	private static Set<String> exchanges = new HashSet<String>();
	/* Exchange的名称集合在Redis中的KEY */
	private static final String KEY = "RABBIT_EXCHANGES";

	/**
	 * 获得指定topic对应的路由器名称
	 *
	 * @param topic 消息topic
	 *
	 * @return 路由名称
	 */
	public static String getExchangeName(String topic) {
		return topic + "_Exchange";
	}

	/**
	 * 获得队列名称
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 队列名称
	 */
	private static String getQueueName(Subscriber subscriber) {
		return subscriber.getTopic() + "_" + subscriber.getIndex();
	}

	/**
	 * 检查指定消息topic的路由器是否存在，如果不存在，则创建
	 *
	 * @param topic 消息topic
	 */
	private static void checkExchange(String topic) {
		String exchangeName = getExchangeName(topic);
		if (RedisUtil.HAS_REDIS) {
			Set<String> s = RedisUtil.getSet(KEY);
			if (s != null && s.contains(exchangeName)) {
				// 路由器已经存在
				return;
			}
		} else if (exchanges.contains(exchangeName)) {
			// 路由器已经存在
			return;
		}

		// 路由器不存在，则创建
		RabbitPool rp = new RabbitPool();
		// 创建渠道
		Channel channel = rp.getChannel();
		// 创建路由
		try {
			channel.exchangeDeclare(exchangeName, "fanout");

			// 记录路由器
			if (RedisUtil.HAS_REDIS) {
				RedisUtil.addIntoSet(KEY, exchangeName);
			} else {
				exchanges.add(exchangeName);
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 创建一个发布者
	 *
	 * @param topic 订阅主题
	 *
	 * @return 一个发布者
	 */
	public Publisher createPublisher(String topic) {
		checkExchange(topic);

		// 路由器不存在，则创建
		RabbitPool rp = new RabbitPool();
		// 创建渠道
		Channel channel = rp.getChannel();

		// 创建发布者
		return new RabbitPublisher(channel, topic);
	}

	/**
	 * 登记一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 一个队列消费者
	 */
	public boolean subscribe(final Subscriber subscriber) {
		String topic = subscriber.getTopic();

		checkExchange(topic);

		// 1. 订阅者queue名称
		String queueName = getQueueName(subscriber);

		RabbitPool rp = new RabbitPool();
		Channel channel = null;
		try {
			// 创建渠道
			channel = rp.getChannel();

			// 创建并绑定消息队列
			setExchangeProperty(channel, queueName, getExchangeName(topic), "fanout");

			Consumer consumer = new DefaultConsumer(channel) {
				@Override
				public void handleDelivery(String consumerTag, Envelope envelope, AMQP.BasicProperties properties,
						byte[] body) throws IOException {
					if (envelope.getRoutingKey().equals("fanout")) {
						String message = new String(body, "UTF-8");
						JSONObject json = JSONObject.parseObject(message);

						try {
							subscriber.consume(json);
						} catch (Exception ex) {
							ex.printStackTrace();
							logger.error(ex.getMessage());
						}
					}
				}
			};
			// 设置为自动应答
			channel.basicConsume(queueName, true, consumer);
		} catch (Exception ex) {
			ex.printStackTrace();

			return false;
		}

		return true;
	}

	/**
	 * 创建对列，并绑定到路由上
	 *
	 * @param channel      队列channel
	 * @param queueName    队列名称
	 * @param exchangeName 路由名称
	 * @param routingKey   路由类型
	 */
	private void setExchangeProperty(Channel channel, String queueName, String exchangeName, String routingKey) {
		try {
			channel.queueDeclare(queueName, true, false, false, null);
			channel.queueBind(queueName, exchangeName, routingKey);
			channel.basicQos(1);
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/**
	 * 删除一个订阅者
	 *
	 * @param subscriber 订阅者
	 *
	 * @return 删除是否成功
	 */
	public boolean remove(Subscriber subscriber) {
		if (subscriber == null) {
			return false;
		}

		// 队列名称
		String quequeName = getQueueName(subscriber);

		RabbitPool rp = new RabbitPool();
		Channel channel = null;
		try {
			// 创建渠道
			channel = rp.getChannel();

			// 删除消息队列
			channel.queueDelete(quequeName);
		} catch (IOException e) {
			e.printStackTrace();

			return false;
		} finally {
			if (channel != null) {
				try {
					channel.close();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		return true;
	}
}
