package team.bangbang.common.redis;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;
import redis.clients.jedis.Protocol;
import team.bangbang.common.cipher.ICipher;
import team.bangbang.common.config.Config;
import team.bangbang.common.utility.LogicUtility;

import java.lang.reflect.Constructor;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据库连接获得器

/**
 * 通过配置文件config.properties中的相关配置，构造Redis连接池
 *
 * @author 帮帮组
 * @version 1.0
 */
// ************************************************************************
public final class RedisPool {
    /* 连接池 */
    private static JedisPool pool = null;

    static {
        // 构造连接池
        if (RedisUtil.HAS_REDIS) {
            configure();
        }
    }

    /**
     * 获得数据库连接
     *
     * @return 数据库连接
     */
    public static Jedis getJedis() {
        return (pool != null ? pool.getResource() : null);
    }

    /**
     * 根据配置文件参数配置Redis数据库连接池
     */
    private static void configure() {
        if (pool != null) {
            return;
        }

        JedisPoolConfig config = new JedisPoolConfig();
        // 控制一个pool可分配多少个jedis实例，通过pool.getResource()来获取；
        // 如果赋值为-1，则表示不限制；如果pool已经分配了maxTotal个jedis实例，则此时pool的状态为exhausted(耗尽)。
        int maxTotal = LogicUtility.parseInt(Config.getProperty("spring.redis.jedis.pool.max-active"), 100);
        config.setMaxTotal(maxTotal);

        // 控制一个pool最多有多少个状态为idle(空闲的)的jedis实例。
        int maxIdle = LogicUtility.parseInt(Config.getProperty("spring.redis.jedis.pool.max-idle"), 5);
        config.setMaxIdle(maxIdle);

        // 表示当borrow(引入)一个jedis实例时，最大的等待毫秒数，如果超过等待时间，则直接抛出JedisConnectionException；
        config.setMaxWaitMillis(100 * 1000);
        // 在borrow一个jedis实例时，是否提前进行validate操作；如果为true，则得到的jedis实例均是可用的；
        config.setTestOnBorrow(true);

        // 主机
        String host = Config.getProperty("spring.redis.host");
        // 端口
        int port = LogicUtility.parseInt(Config.getProperty("spring.redis.port"), 6379);

        // 数据库
        int database = LogicUtility.parseInt(Config.getProperty("spring.redis.database"), Protocol.DEFAULT_DATABASE);

        String pwd = getPassword();
        pool = new JedisPool(config, host, port, Protocol.DEFAULT_TIMEOUT, pwd, database);
    }

    private static String getPassword() {
        // 明文密码
        String pwd = Config.getProperty("spring.redis.password");
        // 密文密码
        String pwd2 = Config.getProperty("spring.redis.redispas");
        if (pwd2 == null || pwd2.length() == 0) {
            // 不存在密文密码，直接使用明文密码
            return pwd;
        }

        // 存在密文密码，则试图解密
        try {
            String clazz = Config.getProperty("spring.redis.password-decoder");
            Constructor<?> cst = Class.forName(clazz).getDeclaredConstructor();
            ICipher cipher = (ICipher)cst.newInstance();
            // 约定：ICipher具体实现类.decrypt(Object)方法返回值必须为字符串
            pwd = (String)cipher.decrypt(pwd2);
        } catch (Exception ex) {

        }

        return pwd;
    }
}
