package team.bangbang.common.redis;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import team.bangbang.common.config.Config;

import team.bangbang.common.redis.RedisPool;
import redis.clients.jedis.Jedis;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据库连接获得器
/**
 * Redis操作类
 *
 * 此處的操作均爲單個方法，方法調用前打開連接，方法調用后關閉連接。
 *
 * 如果需要批量調用多個Redis方法，請使用RedisPool獲取Redis連接后再行使用，但必須注意，使用完畢后手動關閉連接！！
 *
 * @author 帮帮组
 * @version 1.0
 */
// ************************************************************************
public final class RedisUtil {
	/** 是否配置了redis数据库 */
	public static final boolean HAS_REDIS = (Config.getProperty("spring.redis.host") != null);

	/**
	 * 将Java对象存储到Redis中
	 *
	 * @param key           数据KEY值
	 * @param value         字符串对象
	 * @param expireSeconds 过期时间，单位为秒。如果&lt;=0，表示永不过期
	 */
	public static void setString(String key, String value, int expireSeconds) {
		if (!HAS_REDIS)
			return;

		if (key == null || key.trim().length() == 0) {
			return;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return;
		}

		try {
			if (value == null) {
				jedis.del(key);
			} else {
				jedis.set(key, value);
				if (expireSeconds > 0)
					jedis.expire(key, expireSeconds);
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	/**
	 * 从Redis中获取字符串对象
	 *
	 * @param key 数据KEY值
	 *
	 * @return 字符串对象
	 */
	public static String getString(String key) {
		if (!HAS_REDIS)
			return null;

		if (key == null || key.trim().length() == 0) {
			return null;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return null;
		}

		String value = "";

		try {
			value = jedis.get(key);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return value;
	}

	/**
	 * 向集合中添加元素
	 *
	 * @param key    数据KEY值
	 * @param values 字符串对象集合
	 */
	public static void addIntoSet(String key, String... values) {
		if (!HAS_REDIS)
			return;

		if (key == null || key.trim().length() == 0 || values == null || values.length == 0) {
			return;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return;
		}

		try {
			jedis.sadd(key, values);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}

	/**
	 * 获取KEY下面的集合
	 *
	 * @param key 指定数据KEY值
	 * @return 指定KEY下面的集合
	 */
	public static Set<String> getSet(String key) {
		if (!HAS_REDIS)
			return null;

		if (key == null || key.trim().length() == 0) {
			return Collections.emptySet();
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return Collections.emptySet();
		}

		Set<String> set = new HashSet<String>();

		try {
			set = jedis.smembers(key);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return set;
	}

	/**
	 * 从集合中删除元素
	 *
	 * @param key    指定数据KEY值
	 * @param values 要删除的元素
	 *
	 * @return 删除的元素数量
	 */
	public static int removeFromSet(String key, String... values) {
		if (!HAS_REDIS)
			return 0;

		if (key == null || key.trim().length() == 0) {
			return 0;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return 0;
		}

		try {
			return (int) jedis.srem(key, values).longValue();
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return 0;
	}

	/**
	 * 从左侧添加到队列中
	 *
	 * @param key    队列名称
	 * @param values 值数组
	 * @return 当前列表长度
	 */
	public static long lpush(String key, String... values) {
		if (!HAS_REDIS)
			return 0L;

		if (key == null || values == null) {
			return 0L;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return 0L;
		}

		try {
			return jedis.lpush(key, values);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return 0L;
	}

	/**
	 * 從隊列右側取出一個元素
	 *
	 * @param key 隊列名稱
	 * @return 取出的元素
	 */
	public static String rpop(String key) {
		if (!HAS_REDIS)
			return null;

		if (key == null) {
			return null;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return null;
		}

		try {
			return jedis.rpop(key);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return null;
	}

	/**
	 * 获取队列中指定位置的元素列表
	 *
	 * @param key   隊列名稱
	 * @param start 开始位置，从0开始。/ 又或者用负数表示与右侧的距离，-1表示最右侧的数据，-2表示右侧倒数第2个数据
	 * @param end   结束位置。/ 又或者用负数表示与右侧的距离，-1表示最右侧的数据，-2表示右侧倒数第2个数据
	 *
	 * @return 取出的元素
	 */
	public static List<String> lrange(String key, long start, long end) {
		if (!HAS_REDIS)
			return null;

		if (key == null) {
			return null;
		}

		// 获取的结果
		List<String> lst = null;
		// 保存到Redis中
		Jedis jedis = null;
		try {
			jedis = team.bangbang.common.redis.RedisPool.getJedis();
			if (jedis == null) {
				return Collections.emptyList();
			}

			lst = jedis.lrange(key, start, end);
		} catch (Exception ex) {
			ex.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		if (lst == null || lst.size() == 0) {
			return Collections.emptyList();
		}

		return lst;
	}

	/**
	 * 獲得隊列的長度
	 *
	 * @param key 隊列名稱
	 * @return 隊列的長度
	 */
	public static long llen(String key) {
		if (!HAS_REDIS)
			return 0L;

		if (key == null) {
			return 0L;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return 0L;
		}

		try {
			return jedis.llen(key);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return 0L;
	}

	/**
	 * 獲得集合中元素的數量
	 *
	 * @param key 集合名稱
	 * @return 集合中元素的數量
	 */
	public static long scard(String key) {
		if (!HAS_REDIS)
			return 0L;

		if (key == null) {
			return 0L;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return 0L;
		}

		try {
			return jedis.scard(key);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return 0L;
	}

	/**
	 * 从Redis中获取KEY集合
	 *
	 * @param pattern KEY的模板
	 *
	 * @return Java对象
	 */
	public static Set<String> keys(String pattern) {
		if (!HAS_REDIS)
			return null;

		if (pattern == null || pattern.trim().length() == 0) {
			return null;
		}

		Jedis jedis = team.bangbang.common.redis.RedisPool.getJedis();
		if (jedis == null) {
			return null;
		}

		Set<String> ss = null;
		try {
			ss = jedis.keys(pattern);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}

		return ss;
	}

	/**
	 * 设置key的过期时间
	 *
	 * @param key     数据KEY值
	 * @param seconds 过期的秒数
	 */
	public static void expire(String key, int seconds) {
		if (!HAS_REDIS)
			return;

		if (key == null || key.trim().length() == 0 || seconds <= 0) {
			return;
		}

		Jedis jedis = RedisPool.getJedis();
		if (jedis == null) {
			return;
		}

		try {
			jedis.expire(key, seconds);
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (jedis != null) {
				jedis.close();
			}
		}
	}
}
