package team.bangbang.common.service.agent;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.service.agent.data.ResponseList;
import team.bangbang.common.service.agent.data.ResponseResult;
import team.bangbang.common.service.agent.data.RestfulAgent;
import team.bangbang.common.utility.LogicUtility;

/**
 * Restful服务代理 - 响应数据的转换
 *
 * @author 帮帮组
 * @version 1.0  2019-09-28
 */
public class ResponseTransform {
	/* 源响应数据 */
	private JSONObject fromData = null;

	/**
	 * 设置源响应数据
	 * @param fromData 源响应数据
	 */
	public void setFromData(JSONObject fromData) {
		this.fromData = fromData;
	}

	/**
	 * 按照Restful服务代理配置获得目标响应数据
	 *
	 * @param ra Restful服务代理配置
	 * @return 目标响应数据
	 */
	public JSONObject getToData(RestfulAgent ra) {
		if(ra == null || ra.getResponse() == null) return null;

		// 处理result标签
		List<ResponseResult> rrs = ra.getResponse().getResults();

		JSONObject toData = new JSONObject();

		// 处理
		doResult(fromData, rrs, toData);

		// 处理list标签
		List<ResponseList> rls = ra.getResponse().getLists();

		for(int i = 0; rls != null && i < rls.size(); i++) {
			ResponseList rl = rls.get(i);
			// 原数据
			JSONArray ja1 = getJSONArray(fromData, rl.getFrom());
			if(ja1 == null || ja1.isEmpty()) continue;

			// 目标数据
			JSONArray ja2 = new JSONArray();
			putJSONArray(toData, rl.getTo(), ja2);

			// list下未配置Result
			rrs = rl.getListResults();
			if(rrs.isEmpty()) continue;

			// 遍历list下面的每组result
			for(int j = 0; ja1 != null && j < ja1.size(); j++) {
				JSONObject jFrom = ja1.getJSONObject(j);
				// 处理list元素的字段
				JSONObject jTo = new JSONObject();

				doResult(jFrom, rrs, jTo);

				ja2.add(jTo);
			}
		}

		return toData;
	}

	/**
	 * 将fData按照Result映射转换后存储到tData中
	 *
	 * @param fData 原数据
	 * @param rrs Result映射
	 * @param tData 目标数据
	 */
	private void doResult(JSONObject fData, List<ResponseResult> rrs, JSONObject tData) {
		for(int i = 0; rrs != null && i < rrs.size(); i++) {
			ResponseResult rr = rrs.get(i);
			String value = getValue(fData, rr.getFrom());

			value = doFormat(value, rr.getFormat());
			value = doDefault(value, rr.getDefaultValue());

			setValue(tData, rr.getTo(), value, rr.getMap());
		}
	}

	/**
	 * 设置默认值
	 *
	 * @param value 值
	 * @param defaultValue 默认值
	 * @return 值
	 */
	private String doDefault(String value, String defaultValue) {
		return (value != null ? value : defaultValue);
	}

	/* 格式化日期 */
	private static Map<String, SimpleDateFormat> sdfMap = new HashMap<String, SimpleDateFormat>();

	/**
	 * 格式化日期
	 *
	 * @param value 日期数值
	 * @param format 格式
	 * @return 格式化之后的日期
	 */
	private String doFormat(String value, String format) {
		if(value == null || value.trim().length() == 0 ||
				format == null || format.trim().length() == 0) {
			return value;
		}

		format = format.trim();

		SimpleDateFormat sdf = sdfMap.get(format);
		if(sdf == null) {
			sdf = new SimpleDateFormat(format);
			sdfMap.put(format, sdf);
		}

		// 是长整型数字？
		long l = LogicUtility.parseLong(value, 0);
		if(l > 0) {
			Date d = new Date(l);
			return sdf.format(d);
		}

		// 是yyyy-MM-dd格式的字符串？
		Date d = LogicUtility.parseDate(value);

		return (d != null ? sdf.format(d) : value);
	}

	/**
	 * 将指定的数据ja按照指定的层级路径to放在toData中
	 *
	 * @param toData JSON数据
	 * @param to 表达式
	 * @param ja 指定的数据ja
	 */
	private void putJSONArray(JSONObject toData, String to, JSONArray ja) {
		if(toData == null || to == null || ja == null) return;

		int n = to.indexOf(".");
		try {
			if(n < 0) {
				toData.put(to, ja);
				return;
			}

			String key1 = to.substring(0, n);
			String key2 = to.substring(n + 1);

			JSONObject temp = new JSONObject();
			toData.put(key1, temp);

			putJSONArray(temp, key2, ja);
		} catch (Exception ex) {

		}
	}

	/**
	 * 根据表达式获得JSON数组
	 *
	 * @param data JSON数据
	 * @param key 表达式
	 * @return JSON数组
	 */
	private JSONArray getJSONArray(JSONObject data, String key) {
		if(data == null || key == null) return null;

		int n = key.indexOf(".");
		try {
			if(n < 0) {
				return data.getJSONArray(key);
			}

			String key1 = key.substring(0, n);
			String key2 = key.substring(n + 1);

			data = data.getJSONObject(key1);

			return getJSONArray(data, key2);
		} catch (Exception ex) {

		}

		return null;
	}

	/**
	 * 获取json中指定KEY的数值
	 *
	 * @param json json对象
	 * @param key KEY值
	 * @return 指定KEY对应的数值
	 */
	private String getValue(JSONObject json, String key) {
		if(json == null || json.isEmpty() || key == null) return null;
		// 是否有点
		int nIndex = key.indexOf(".");
		if(nIndex < 0) {
			// 是否有
			return json.getString(key);
		}

		// 带有.
		String key1 = key.substring(0, nIndex);
		String key2 = key.substring(nIndex + 1);

		JSONObject sonMap = json.getJSONObject(key1);

		return getValue(sonMap, key2);
	}

	/**
	 * 设置json中指定KEY的数值
	 *
	 * @param json json对象
	 * @param key KEY值
	 * @param value 指定KEY对应的数值
	 * @param map 数值映射
	 */
	private void setValue(JSONObject json, String key, String value, Map<String, String> map) {
		if(json == null || key == null) return;
		// 是否有点
		int nIndex = key.indexOf(".");
		if(nIndex < 0) {
			if(map != null && map.containsKey(value)) {
				// 进行数据转换
				value = map.get(value);
			}

			json.put(key, value);
			return;
		}

		// 带有.
		String key1 = key.substring(0, nIndex);
		String key2 = key.substring(nIndex + 1);

		// 下级Map
		JSONObject sonMap = json.getJSONObject(key1);
		if(sonMap == null) {
			// 下级不是map对象
			sonMap = new JSONObject();
			json.put(key1, sonMap);
		}

		setValue(sonMap, key2, value, map);
	}
}
