package team.bangbang.common.servlet;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import team.bangbang.common.config.Config;
import team.bangbang.common.utility.LogicUtility;

/**
 * 向客户端输出图片，可以将图片缩小到指定的方格内（maxWidth、maxHeight）<br>
 * （如果方格大于图片实际尺寸，则图片按原样显示）；<br>
 * 此外图片可以指定缩放的百分比。<br>
 * 如果没有指定图片方格大小和缩放的百分比，则使用默认大小。
 *
 * @author 帮帮组
 * @version 1.0 2009-06-24
 */
@WebServlet(urlPatterns = "/common/image")
public class ImageServlet extends HttpServlet {
	private static final long serialVersionUID = 5571876706562039274L;

	@Override
	protected void service(HttpServletRequest request,
			HttpServletResponse response) throws ServletException, IOException {
		// 获取文件名
		// 获取文件参数名称
		String file = request.getParameter("file");
		if (file == null) {
			return;
		}

		// 中文化处理
		file = LogicUtility.getCNString(file);

		// 拼合得到全路径
		file = Config.getProperty("file.attachment.directory") + "/" + file;

		// 获取图片的原始大小
		BufferedImage bufferedImage = ImageIO.read(new File(file));
		int originWidth = bufferedImage.getWidth();
		int originHeight = bufferedImage.getHeight();

		// 目标图片大小
		int width = 0, height = 0;
		// 获取百分比（不含百分号）
		int scale = LogicUtility.parseInt(request.getParameter("scale"), 0);
		if (scale > 0) {
			width = originWidth * scale / 100;
			height = originHeight * scale / 100;
		} else {
			// 检查最大maxWidth和maxHeight参数
			int maxWidth = LogicUtility.parseInt(request
					.getParameter("maxWidth"), 0);
			int maxHeight = LogicUtility.parseInt(request
					.getParameter("maxHeight"), 0);
			if (maxWidth >= originWidth && maxHeight >= originHeight) {
				// 不需要缩小，按原样显示
				width = originWidth;
				height = originHeight;
			} else {
				if (maxWidth > 0 && maxHeight > 0) {
					// 计算百分比
					double scale1 = ((double) maxWidth) / originWidth;
					double scale2 = ((double) maxHeight) / originHeight;
					if (scale1 < scale2) {
						// 以宽度maxWidth为准
						width = maxWidth;
						height = (int) (originHeight * scale1);
					} else {
						// 以高度maxHeight为准
						height = maxHeight;
						width = (int) (originWidth * scale2);
					}
				}
			}
		}

		if (width == 0 || height == 0) {
			// 以原尺寸输出
			ImageIO.write(bufferedImage, "JPEG", response.getOutputStream());
		} else {
			// 尺寸变换后输出
			ImageIO.write(resizeImage(bufferedImage, width, height), "JPEG",
					response.getOutputStream());
		} // end if
	}

	/**
	 * 进行图片的缩放
	 *
	 * @param originalImage
	 *            原始图片
	 * @param scaledWidth
	 *            目标图片的宽度
	 * @param scaledHeight
	 *            目标图片的高度
	 * @return 缩放后的图片
	 */
	private BufferedImage resizeImage(BufferedImage originalImage,
			int scaledWidth, int scaledHeight) {
		BufferedImage scaledBI = new BufferedImage(scaledWidth, scaledHeight,
				BufferedImage.TYPE_INT_RGB);
		Graphics2D g = scaledBI.createGraphics();
		g.setComposite(AlphaComposite.Src);
		g.drawImage(originalImage, 0, 0, scaledWidth, scaledHeight, null);
		g.dispose();
		return scaledBI;
	}
}
