package team.bangbang.common.servlet;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebInitParam;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Tableau报表免登录信任处理
 *
 * 在web.xml中的定义可以传入以下参数：
 * <ul>
 * <li>tableauServerUrl：Tableau Server地址，仅仅包括协议、Host、端口，例如：https://tableau.c.citic/</li>
 * <li>tableauServerUser：Tableau Server用户</li>
 * </ul>
 * 通过URL可以传入以下参数：
 * <ul>
 * <li>view：报表view的地址，默认site以views/开头，例如：/views/_1/sheet0；指定site以t/[site_name]/views/开头，例如：/t/weiapp_lm/views/_0/sheet0</li>
 * </ul>
 *
 * @author 帮帮组
 * @version 1.0 2018-05-14
 */
@WebServlet(urlPatterns = "/common/tableau",
initParams={
        @WebInitParam(name="tableauServerUrl",value="${servlet.tableau.server.url''}"),	// Tableau Server地址
        @WebInitParam(name="tableauServerUser",value="${servlet.tableau.server.user''}") // Tableau Server用户
})
public class TableauServlet extends HttpServlet {
	private static final long serialVersionUID = 8764087666676523513L;
	/** Tableau Server地址 */
	private static String tableauServerUrl = null;
	/** Tableau Server用户 */
	private static String tableauServerUser = null;

	/**
	 * 初始化Servlet，读取配置参数
	 *
	 * @param config
	 *            Servlet配置
	 */
	public void init(ServletConfig config) {
		// Tableau Server地址
		tableauServerUrl = config.getInitParameter("tableauServerUrl");
		if(tableauServerUrl != null) {
			tableauServerUrl = tableauServerUrl.trim();
			if(!tableauServerUrl.endsWith("/")) {
				tableauServerUrl += "/";
			}
		}

		// Tableau Server用户
		tableauServerUser = config.getInitParameter("tableauServerUser");
		if(tableauServerUser != null) {
			tableauServerUser = tableauServerUser.trim();
		}
	}

	@Override
	protected void service(HttpServletRequest request,
			HttpServletResponse response) throws ServletException, IOException {
		if(tableauServerUrl == null || tableauServerUrl.length() == 0) {
			throw new ServletException("The parameter(tableauServerUrl) not found in web.xml");
		}

		if(tableauServerUser == null || tableauServerUser.length() == 0) {
			throw new ServletException("The parameter(tableauServerUser) not found in web.xml");
		}

		String view = request.getParameter("view");

		if(view == null || view.length() == 0) {
			throw new IOException("The parameter(view) not found in this request");
		}

		if(!view.startsWith("/")) {
			view = "/" + view;
		}

		String ip = getRemoteIP(request);

		// 获得target_site
		String targetSite = getTargetSite(view);

		String ticket = getTrustedTicket(tableauServerUrl, tableauServerUser, targetSite, ip);
		if(ticket == null || ticket.indexOf("==") < 0) {
			throw new IOException("Truested ticket is invalid : [" + ticket + "]");
		}

		ticket = ticket.trim();

		String url = tableauServerUrl + "trusted/" + ticket + view + "?:iid=1&:embed=yes&:toolbar=yes";
		response.setStatus(HttpServletResponse.SC_MOVED_TEMPORARILY);
		response.setHeader("Location", url);
	}

	/**
	 * @param view  报表视图地址，例如：/t/weiapp_lm/views/_0/sheet0
	 * @return 报表视图中的siteName
	 */
	private static String getTargetSite(String view) {
		if(view == null) {
			return null;
		}

		// 如果不是以/t/开头
		if(!view.startsWith("/t/")) {
			return null;
		}

		// weiapp_lm/views/_0/sheet0
		view = view.substring(3);

		int nIndex = view.indexOf("/");

		if(nIndex <= 0) {
			return null;
		}

		return view.substring(0,  nIndex);
	}

	private static String getTrustedTicket(String wgserver, String user, String targetSite, String remoteAddr) {
		OutputStreamWriter out = null;
		BufferedReader in = null;
		try {
			StringBuffer data = new StringBuffer();
			data.append(URLEncoder.encode("username", "UTF-8"));
			data.append("=");
			data.append(URLEncoder.encode(user, "UTF-8"));

			if(targetSite != null && targetSite.trim().length() > 0) {
				data.append("&");
				data.append(URLEncoder.encode("target_site", "UTF-8"));
				data.append("=");
				data.append(URLEncoder.encode(targetSite.trim(), "UTF-8"));
			}

			if(remoteAddr != null && remoteAddr.trim().length() > 0) {
				data.append("&");
				data.append(URLEncoder.encode("client_ip", "UTF-8"));
				data.append("=");
				data.append(URLEncoder.encode(remoteAddr.trim(), "UTF-8"));
			}

			URL url = new URL(wgserver + "/trusted");
			URLConnection conn = url.openConnection();
			conn.setDoOutput(true);
			out = new OutputStreamWriter(conn.getOutputStream());

			out.write(data.toString());
			out.flush();

			StringBuffer sb = new StringBuffer();
			in = new BufferedReader(new InputStreamReader(conn.getInputStream()));

			String line = null;
			while((line = in.readLine()) != null) {
				sb.append(line);
			}

			return sb.toString();
		} catch (IOException ioex) {
			ioex.printStackTrace();
		} finally {
			try { if(in != null) in.close(); } catch (IOException ex) {}
			try { if(out != null) out.close(); } catch (IOException ex) {}
		}

		return null;
	}

	/**
	 * @param request HTTP请求
	 * @return 请求来源地址
	 */
	protected String getRemoteIP(HttpServletRequest request) {
		if(request == null) {
			return null;
		}

		if (request.getHeader("x-forwarded-for") == null) {
			return request.getRemoteAddr();
		}

		return request.getHeader("x-forwarded-for");
	}
}
