package team.bangbang.common.servlet;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.config.Config;
import team.bangbang.common.data.StatusCode;
import team.bangbang.common.data.response.DataResponse;
import team.bangbang.common.data.response.ResponseBase;
import team.bangbang.common.data.util.Transport;
import team.bangbang.common.file.FileUtil;
import team.bangbang.common.file.IFileManager;
import team.bangbang.common.utility.LogicUtility;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：文件上传处理器

/**
 * 文件上传处理
 *
 * 参数：
 *
 * file：必选，上传的文件控件名称
 *
 * overwrite 可选，如果文件重名，是否覆盖文件，默认为false
 *
 * rename 可选，默认为true，表示上传后文件名以时间戳重命名；false值表示上传后文件名不变
 *
 * folder 可选，放置在指定目录下，如果目录不存在，则指定的自动创建，默认为当前日期（yyyyMM）目录
 *
 * manager 可选，文件上传下载管理器，通过该参数指定使用的上传下载处理器
 *
 * @author 帮帮组
 * @version 1.0 2018-10-18
 */
// ************************************************************************
@WebServlet(urlPatterns = "/common/upload")
public class UploadServlet extends HttpServlet {
	private static final long serialVersionUID = 8966249419259468129L;

	/**
	 * @param request  HTTP请求
	 * @param response HTTP响应
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		request.setCharacterEncoding("utf-8");
		response.setContentType("text/html;charset=utf-8");
		// 解析数据
		Map<String, Object> datas = new HashMap<String, Object>();
		// 上传文件、传递数据
		Transport.transport(request, datas);

		// 获取参数
		// 放置在指定目录下，如果目录不存在，则指定的自动创建，默认为当前日期（yyyyMM）目录
		String folder = (String) datas.get("folder");
		// 如果文件重名，是否覆盖文件，默认为false
		boolean overwrite = "true".equalsIgnoreCase((String) datas.get("overwrite"));
		// 文件是否要以时间戳重命名，默认为true
		boolean rename = !"false".equalsIgnoreCase((String) datas.get("rename"));
		// 使用的文件上传下载管理器 */
		String fileManager = (String) datas.get("manager");
		// 上传的文件信息(web uploader使用key值) */
		Object[] file = (Object[]) datas.get("file");

		boolean isCkeditor = false;
		if (file == null || file.length != 2) {
			// ckeditor使用key值
			file = (Object[]) datas.get("upload");
			isCkeditor = true;
		}
		if (file == null || file.length != 2) {
			// 上传的数据不对
			ResponseBase rb = new ResponseBase();

			response.getWriter().print(JSONObject.toJSONString(rb));
			response.getWriter().close();

			return;
		}

		String fileName = getFileName((String) file[0]);
		// System.out.println("上传文件 " + fileName);

		File _file = (File) file[1];

		// 远程文件名
		String remoteFile = null;
		ResponseBase rb = null;
		try {
			// 文件上传、下载处理器接口
			IFileManager manager = getFileManager(fileManager);

			// 目标文件片段，不含根目录前缀
			String destSliceFile = getDestSliceFile(manager, fileName, folder, rename, overwrite);
			// 文件上传
			remoteFile = manager.put(_file.getAbsolutePath(), destSliceFile);

			DataResponse<String> dto = new DataResponse<String>(StatusCode.SUCCESS, "文件上传成功");
			dto.setData(remoteFile);

			rb = dto;
		} catch (IOException e) {
			e.printStackTrace();
			rb = new ResponseBase(StatusCode.EXCEPTION_OCCURED, e.getMessage());
		}

		if (!isCkeditor) {
			response.getWriter().print(JSONObject.toJSONString(rb));
		} else if(remoteFile != null){
			// ckeditor 4.11需要返回的数据
			Map<String, Object> mp = new HashMap<String, Object>();
			mp.put("fileName", fileName);
			mp.put("uploaded", 1);
			// 是否是全路径？
			boolean isFullPath = (remoteFile.toLowerCase().startsWith("http://")
					|| remoteFile.toLowerCase().startsWith("https://"));

			mp.put("url", isFullPath ? remoteFile : "../common/stream?file=" + remoteFile);
			
			DataResponse<Map<String, Object>> dr = new DataResponse<Map<String, Object>>();
			dr.setData(mp);
			response.getWriter().print(JSONObject.toJSONString(dr));
		}

		response.getWriter().close();

		return;
	}

	/**
	 * 获取目标文件片段，不含根目录前缀。
	 *
	 * @param manager   文件上传、下载处理器接口
	 * @param fileName  上传的文件原路径，包含文件名
	 * @param folder    放置在指定目录下，如果目录不存在，则指定的自动创建，默认为当前日期（yyyyMM）目录
	 * @param rename    默认为true，表示上传后文件名以时间戳重命名；false值表示上传后文件名不变
	 * @param overwrite 如果文件重名，是否覆盖文件，默认为false
	 *
	 * @return 目标文件片段，不含根目录前缀
	 *
	 * @throws IOException IO异常
	 */
	private String getDestSliceFile(IFileManager manager, String fileName, String folder, boolean rename,
									boolean overwrite) throws IOException {
		if (folder == null || folder.length() == 0) {
			// 当前年月子节点
			folder = LogicUtility.getYearMonth(new Date());
		} else {
			// 如果是汉字，则此处需要进行反向UTF-8转码
			try {
				folder = URLDecoder.decode(folder, "UTF-8");
			} catch (UnsupportedEncodingException e) {
				e.printStackTrace();
			}
		}

		if (rename) {
			// 只使用扩展名
			String ext = FileUtil.getFileExtension(fileName);
			return folder + "/" + System.currentTimeMillis() + "." + ext;
		}

		// 单纯文件名
		fileName = getFileName(fileName);
		String sliceFile = folder + "/" + fileName;

		if (overwrite) {
			return sliceFile;
		}

		// 不覆盖同名文件
		int nIndex = 0;
		// 是否有同名文件
		while (manager.exist(sliceFile)) {
			sliceFile = folder + "/" + getNextFilePath(fileName, ++nIndex);
		}

		return sliceFile;
	}

	/**
	 * 获取文件管理器
	 * 
	 * @param fileManager 文件管理器类名
	 * @return 文件管理器
	 * @throws IOException IO异常
	 */
	public static IFileManager getFileManager(String fileManager) throws IOException {
		// 获得文件上传、下载处理器
		// 优先使用传入参数
		String clazz = fileManager;

		// 其次使用config.properties文件配置参数
		if (clazz == null || clazz.trim().length() == 0) {
			clazz = Config.getProperty("file.attachment.manager");
		}

		// 最后使用默认值
		if (clazz == null || clazz.trim().length() == 0) {
			clazz = "team.bangbang.common.file.LocalFileManager";
		}

		IFileManager manager = null;
		try {
			manager = (IFileManager) Class.forName(clazz).newInstance();
		} catch (Exception ex) {
			throw new IOException("文件上传、下载处理器配置有误！(" + clazz + ")");
		}

		return manager;
	}

	/**
	 * 从路径中获得文件名
	 *
	 * @param p 路径
	 * @return 文件名
	 */
	private String getFileName(String p) {
		if (p == null || p.trim().length() == 0)
			return "";

		int n = p.lastIndexOf("/");
		if (n < 0) {
			n = p.lastIndexOf("\\");
		}

		if (n < 0) {
			return p;
		}

		return p.substring(n + 1);
	}

	private String getNextFilePath(String fileName, int i) {
		if (fileName == null || fileName.trim().length() == 0) {
			return String.valueOf(i);
		}

		// 获得扩展名
		int nIndex = fileName.lastIndexOf(".");
		if (nIndex < 0) {
			return fileName + "(" + i + ")";
		}

		String name = fileName.substring(0, nIndex);
		String ext = fileName.substring(nIndex);

		return name + "(" + i + ")" + ext;
	}
}
