package team.bangbang.common.servlet;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.Random;

import javax.imageio.ImageIO;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebInitParam;
import javax.servlet.annotation.WebServlet;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import team.bangbang.common.config.Constants;
import team.bangbang.common.data.ValidationCode;
import team.bangbang.common.data.ValidationCode.CodeEntity;
import team.bangbang.common.utility.LogicUtility;
import team.bangbang.sso.SSOContext;
import team.bangbang.sso.TokenBinder;

/**
 * 生成验证码的Servlet。在web.xml中的定义可以传入以下参数：
 * <ul>
 * <li>length：整型，验证码字符长度，缺省为4；</li>
 * </ul>
 * 通过URL可以传入以下参数：
 * <ul>
 * <li>fontSize：整型，验证码字符大小，缺省为30；</li>
 * <li>padding：整型，字符之间间隔，缺省为0；</li>
 * <li>chaos：布尔型，是否输出燥点，缺省为true；</li>
 * <li>chaosColor：字符型，格式为#ABCDEF，燥点颜色，缺省为灰色；</li>
 * <li>background：字符型，格式为#ABCDEF，背景颜色，缺省为白色；</li>
 * <li>winding：双精度型，扭曲度，缺省为PI；</li>
 * <li>onlyNumber：布尔型，是否为纯数字，缺省为true；</li>
 * </ul>
 */
@WebServlet(urlPatterns = "/common/validationCode",
		initParams={
				@WebInitParam(name="length",value="4") // 验证码长度
		})
public class ValidationCodeServlet extends HttpServlet {
	private static final long serialVersionUID = -4752126881820789772L;
	/* 字体 */
	private final static String[] fonts = { "Arial", "Georgia", "Times New Roman", "Blue", "Yellow" };
	/* 验证码长度 */
	private static int length = 5;

	@Override
	public void init(ServletConfig config) {
		// 验证码长度，默认为4个字符
		length = LogicUtility.parseInt(config.getInitParameter("length"), 4);
	}

	@Override
	protected void service(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		// 验证码字体大小
		int fontSize = LogicUtility.parseInt(request.getParameter("fontSize"), 30);
		// 边框补
		int padding = LogicUtility.parseInt(request.getParameter("padding"), 0);
		// 是否输出燥点（默认输出）
		boolean chaos = !"false".equalsIgnoreCase(request.getParameter("chaos"));
		// 燥点的颜色（默认灰色）
		String temp = request.getParameter("chaosColor");
		Color chaosColor = LogicUtility.parseColor(temp, Color.lightGray);
		// 背景色（默认白色）
		temp = request.getParameter("background");
		Color background = LogicUtility.parseColor(temp, Color.white);
		// 扭曲度，此值越大，扭曲程度越大，产生波形滤镜效果 ，默认为2PI
		temp = request.getParameter("winding");
		double winding = LogicUtility.parseDouble(temp, 2 * Math.PI);
		// 是否输出纯数字（默认输出纯数字）
		boolean onlyNumber = !"false".equalsIgnoreCase(request.getParameter("onlyNumber"));

		// 生成验证码
		ValidationCode.CodeEntity ce = ValidationCode.getCodeEntity(length, onlyNumber);
		
		// 保存验证码
		saveCode(request, ce);
		// 验证码文字
		String code = ce.letter;

		// 在内存中构建一个Image对象
		// 边缘margin : 2px
		// Image的宽度：字数 * (字体大小 + 间隔) + 间隔 + 4
		int width = length * (fontSize + padding) + padding + 4;
		// Image的高度：字体大小 + 4
		// 由于字体会错位，Image的高度需要再加上 length
		int height = fontSize + padding * 2 + 4 + length;
		BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
		Graphics g = image.getGraphics();
		// 填充背景色
		g.setColor(background);
		g.fillRect(0, 0, width, height);

		// 生成随机类
		Random random = new Random();
		// 给背景添加随机生成的燥点，使图象中的认证码不易被其它程序探测到
		if (chaos) {
			int c = length * 10;
			for (int i = 0; i < c; i++) {
				// 平均每个字符上设置10个燥点
				int x = random.nextInt(width);
				int y = random.nextInt(height);
				// 设置燥点的颜色
				g.setColor(chaosColor);
				// 生成燥点
				g.drawRect(x, y, 1, 1);
			}
		}

		// 随机字体和颜色的验证码字符
		// 字体实际高度
		int fHeight = (int) (fontSize / 1.5);
		// 纵向中心位置
		int vMiddle = height / 2 + fHeight / 2;
		// 纵向字体正中的位置
		for (int i = 0; i < length; i++) {
			// 设置字体属性
			int findex = random.nextInt(fonts.length);
			Font font = new Font(fonts[findex], Font.BOLD, fontSize);
			g.setFont(font);
			// 横向位置
			int left = i * (fontSize + padding) + padding + (fontSize - fHeight) / 2;
			int top = ((i % 2 == 0) ? vMiddle - length / 2 : vMiddle + length / 2);

			g.setColor(getRandomColor(10 + i, 120 - i));
			g.drawString(String.valueOf(code.charAt(i)), left, top);
		}

		// 扭曲效果
		shear(g, width, height, background, winding);

		g.dispose();

		// 设置页面不缓存
		response.setHeader("Pragma", "No-cache");
		response.setHeader("Cache-Control", "no-cache");
		response.setDateHeader("Expires", 0);
		response.setContentType("image/jpeg");

		ImageIO.write(image, "jpg", response.getOutputStream());
		response.getOutputStream().close();
	}

	private void saveCode(HttpServletRequest request, CodeEntity ce) {
		// 是否有token传入？
		String token = (String) SSOContext.getToken();
		if (token != null && token.trim().length() > 0) {
			// 以token保存在Redis中
			TokenBinder.saveValidationCode(token, ce.letter);
			return;
		}

		// 保存到Session中
		request.getSession(true).setAttribute(Constants.KEY_VALIDATION_CODE, ce);
	}

	/**
	 * 获取一个随机颜色
	 *
	 * @param start
	 *            R、G、B三色范围下限（三种颜色R、G、B最大值为0xFF）
	 * @param end
	 *            R、G、B三色范围上限（三种颜色R、G、B最大值为0xFF）
	 * @return 随机颜色
	 */
	private Color getRandomColor(int start, int end) {
		Random random = new Random();
		if (start > 0xFF)
			start = 0xFF;
		if (end > 0xFF)
			end = 0xFF;
		int rang = Math.abs(end - start);
		int r = start + random.nextInt(rang);
		int g = start + random.nextInt(rang);
		int b = start + random.nextInt(rang);
		return new Color(r, g, b);
	}

	/**
	 * 扭曲图片中的内容
	 *
	 * @param g
	 *            绘图对象
	 * @param w1
	 *            图片宽度
	 * @param h1
	 *            图片高度
	 * @param background
	 *            空白区域的颜色
	 * @param winding
	 *            扭曲强度
	 */
	private void shear(Graphics g, int w1, int h1, Color background, double winding) {
		Random random = new Random();
		// 0, 1
		int period = random.nextInt(2);
		boolean borderGap = true;
		int frames = 1;
		// 0, 1
		int phase = random.nextInt(2);
		for (int i = 0; i < h1; i++) {
			double d = (double) (period >> 1)
					* Math.sin((double) i / (double) period + (winding * phase) / (double) frames);
			g.copyArea(0, i, w1, 1, (int) d, 0);
			if (borderGap) {
				g.setColor(background);
				g.drawLine((int) d, i, 0, i);
				g.drawLine((int) d + w1, i, w1, i);
			}
		}

		period = random.nextInt(16);
		frames = 20;
		phase = 3;
		for (int i = 0; i < w1; i++) {
			double d = (double) (period >> 1)
					* Math.sin((double) i / (double) period + (winding * phase) / (double) frames);
			g.copyArea(i, 0, 1, h1, 0, (int) d);
			if (borderGap) {
				g.setColor(background);
				g.drawLine(i, (int) d, i, 0);
				g.drawLine(i, (int) d + h1, i, h1);
			}
		}
	}
}
