package team.bangbang.common.sql;

import java.math.BigDecimal;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

import team.bangbang.common.data.Pagination;
import team.bangbang.common.data.RecordData;
import team.bangbang.common.sql.SQLPool;
import team.bangbang.common.sql.generator.IDGenerator;
import team.bangbang.common.utility.LogicUtility;

//********************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据库操作器
/**
 * 配合team.bangbang.common.data.RecordData使用实现数据库的增删改功能，并且能够<br>
 * 实现单条记录的查询和多条记录的查询，多条记录查询的时候能够根据参数选择是否<br>
 * 进行分页处理，查询结果自动按照字段名和字段值的对应关系保存到<br>
 * team.bangbang.common.data.RecordData实例中。<br>
 *
 * 此外，本类也提供了一个常规的JDBC操作，即直接执行SQL语句实现增删改功能。
 *
 * @author 帮帮组
 * @version 1.0 2009-05-31
 */
// *******************************************************************
public final class SQLHelper {
	/* 数据库连接 */
	private Connection conn = null;
	/* 当前数据库连接在config.properties中配置的数据库别名 */
	private String alias = null;
	/* 数据库名称 */
	private String databaseName = null;

	/**
	 * 使用team.bangbang.common.database.DBManager提供的默认数据库连接<br>
	 * 构造数据库操作实例
	 */
	public SQLHelper() {
		this("application");
	}

	/**
	 * 使用team.bangbang.common.database.DBManager提供的指定数据库连接<br>
	 * 构造数据库操作实例。<br>
	 *
	 * 数据库通过别名进行指定。
	 *
	 * @param alias
	 *            数据库别名，参见config.properties文件中的数据库配置
	 */
	public SQLHelper(String alias) {
		try {
			// 生成数据库连接
			this.conn = SQLPool.getConnection(alias);
			// 保存数据库别名
			this.alias = alias;
		} catch (SQLException ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * 使用指定的数据库连接构造数据库操作器
	 *
	 * @param conn 数据库连接
	 */
	public SQLHelper(Connection conn) {
		this.conn = conn;
	}

	/**
	 * @return 当前数据库连接在config.properties中配置的数据库别名
	 */
	public String getAlias() {
		return alias;
	}

	/**
	 * @return 数据库连接
	 */
	public Connection connection() {
		return this.conn;
	}

	/**
	 * 获得数据库产品的名称，可能的结果有：Oracle、MySQL
	 *
	 * @return 数据库名称
	 */
	public String getDatabaseName() {
		if (databaseName != null) {
			return databaseName;
		}

		try {
			DatabaseMetaData dmd = conn.getMetaData();
			databaseName = dmd.getDatabaseProductName();
		} catch (SQLException ex) {
			databaseName = "";
		}

		return databaseName;
	}

	/**
	 * 设置commit模式
	 *
	 * @param autoCommit
	 *            commit模式
	 * @throws SQLException
	 *             SQL异常
	 */
	public void setAutoCommit(boolean autoCommit) throws SQLException {
		if (conn.getAutoCommit() != autoCommit) {
			conn.setAutoCommit(autoCommit);
		}
	}

	/**
	 * 获得commit模式
	 *
	 * @return commit模式
	 * @throws SQLException
	 *             SQL异常
	 */
	public boolean getAutoCommit() throws SQLException {
		return conn.getAutoCommit();
	}

	/**
	 * 执行commit操作，仅当commit模式为false时有效。
	 *
	 * @throws SQLException
	 *             SQL异常
	 */
	public void commit() throws SQLException {
		if (!conn.getAutoCommit())
			conn.commit();
	}

	/**
	 * 数据回滚，仅当commit模式为false时有效
	 *
	 * @throws SQLException
	 *             SQL异常
	 */
	public void rollback() throws SQLException {
		if (!conn.getAutoCommit())
			conn.rollback();
	}

	/**
	 * 关闭数据库连接和存储过程调用句柄
	 *
	 * @throws SQLException
	 *             SQL异常
	 */
	public void close() throws SQLException {
		// 关闭数据库连接
		if (conn != null) {
			conn.close();
		}
	}

	/**
	 * 执行SELECT查询操作，并把字段名和对应的查询结果自动匹配保存到数据容器中 。<br>
	 *
	 * 查询结果仅保存一条记录，如果查询结果为多条记录，则保存第一条记录。
	 *
	 * @param strSql
	 *            SELECT语句
	 * @param data
	 *            数据容器
	 * @throws SQLException
	 *             SQL异常
	 */
	public void querySingleData(String strSql, RecordData data)
			throws SQLException {
		if (data == null)
			return;

		ResultSet rs = null;
		Statement st = null;
		try {
			// database operation statement
			st = conn.createStatement();
			// 针对数据库进行优化
			String dbName = getDatabaseName();
			if ("Oracle".equalsIgnoreCase(dbName)) {
				// oracle查询一条记录
				/*
				SELECT * FROM (SELECT PAGINATION.*, ROWNUM RN FROM (
					    select * from CUST_EMPLOYEE ORDER BY EMP_CODE
					) PAGINATION WHERE ROWNUM <= 5) WHERE RN >= 0;
				*/
				strSql = "SELECT * FROM (SELECT PAGINATION_.*, ROWNUM RN FROM (" + strSql
						+ ") PAGINATION_ WHERE ROWNUM <= 1) WHERE RN >= 0";
			} else if ("MySQL".equalsIgnoreCase(dbName)) {
				strSql = strSql + " limit 1";
			}

			rs = st.executeQuery(strSql);
			readSingleData(rs, data);
		} catch (SQLException ex) {
			// ex.printStackTrace();
			throw new SQLException(ex.getMessage() + "\n" + strSql);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException ex) {
				}
			}
			if (st != null) {
				try {
					st.close();
				} catch (SQLException ex) {
				}
			}
		}
	}

	/**
	 * 执行SELECT查询操作，并把字段名和对应的查询结果自动匹配保存到数据容器中<br>
	 * 查询结果可以为多条记录。同时保存到数据容器中的还有记录的总条数。<br>
	 *
	 * 本方法中分页操作使用了JDBC通用操作，通用性强，但性能较差。如果需要针对<br>
	 * 具体数据库提供高性能实现，则需要对在此处修改分页相关代码。
	 *
	 * @param strSql
	 *            SELECT语句
	 * @param data
	 *            数据容器
	 * @param blFilter
	 *            是否需要分页 true: 需要分页 false: 不需要分页
	 * @throws SQLException
	 *             SQL异常
	 */
	public void queryMultipleData(String strSql, RecordData data, boolean blFilter)
			throws SQLException {

		if (data == null) {
			return;
		}

		ResultSet rs = null;
		Statement st = null;
		try {
			// 使用jdbc2.0
			st = conn.createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE,
					ResultSet.CONCUR_READ_ONLY);

			// 保存总记录数
			if (data.getPagination() == null) {
				data.setPagination(new Pagination());
			}

			int n = getRecordCount(strSql);
			data.getPagination().setRecordCount(n);

			// 需要分页情况下，针对数据库进行优化
			if (blFilter) {
				// 针对数据库进行优化
				String dbName = getDatabaseName();
				// 开始位置 从0开始
				long nStart = data.getPagination().getStartPosition();
				// 每页最大记录数量
				int maxResult = data.getPagination().getMaxResults();
				if ("Oracle".equalsIgnoreCase(dbName)) {
					// oracle分页
					/*
					SELECT * FROM (SELECT PAGINATION.*, ROWNUM RN FROM (
						    select * from CUST_EMPLOYEE ORDER BY EMP_CODE
						) PAGINATION WHERE ROWNUM <= 5) WHERE RN >= 0;
					*/
					strSql = "SELECT * FROM (SELECT PAGINATION_.*, ROWNUM RN FROM (" + strSql
							+ ") PAGINATION_ WHERE ROWNUM <= " + (nStart + maxResult) + ") WHERE RN > " + nStart;
				} else if ("MySQL".equalsIgnoreCase(dbName)) {
					strSql = strSql + " limit " + nStart + ", " + maxResult;
				}
			}

			rs = st.executeQuery(strSql);

			readMultipleData(rs, data);
		} catch (SQLException ex) {
			throw new SQLException(ex.getMessage() + "\n" + strSql);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException ex) {
				}
			}
			if (st != null) {
				try {
					st.close();
				} catch (SQLException ex) {
				}
			}
		}
	}

	/**
	 * 通过构造SELECT COUNT(FirstField) FROM ... 语句得到指定SQL语句<br>
	 * 查询产生的记录条数
	 *
	 * @param strSql
	 *            指定SQL语句
	 * @return 记录总条数
	 * @throws SQLException SQL异常
	 */
	public int getRecordCount(String strSql) throws SQLException {
		ResultSet rs = null;
		Statement st = null;
		int nCount = 0;
		try {
			st = conn.createStatement();

			// 去除多余空格
			strSql = strSql.replaceAll("\\s+", " ");

			// 去掉 order by 后面的
			String upStr = strSql.toUpperCase();
			// 去除order by
			int nIndex = upStr.indexOf(" ORDER ");
			if (nIndex > 0) {
				strSql = strSql.substring(0, nIndex);
			}

			// 第一个FROM位置
			nIndex = upStr.indexOf(" FROM ");
			if(nIndex < 0) {
				return 1;
			}
			strSql = "SELECT COUNT(*) " + strSql.substring(nIndex);

			// 构造嵌套查询语句SELECT COUNT(*) FROM (SELECT ...) AS DB_TEMP_TABLE
			String dbName = getDatabaseName();
			if ("Oracle".equalsIgnoreCase(dbName)) {
			} else if ("MySQL".equalsIgnoreCase(dbName)) {
			} else if ("Microsoft SQL Server".equalsIgnoreCase(dbName)) {
			} else if ("ACCESS".equalsIgnoreCase(dbName)) {
			} else {
				throw new SQLException(
						"Please config SQL to COUNT records for DB:" + dbName);
			}

			rs = st.executeQuery(strSql);

			if (rs != null && rs.next()) {
				nCount = rs.getInt(1);
			}
		} catch (SQLException ex) {
			throw new SQLException(ex.getMessage() + "\n" + strSql);
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException ex) {
				}
			}
			if (st != null) {
				try {
					st.close();
				} catch (SQLException ex) {
				}
			}
		} // end try

		return nCount;
	}

	/**
	 * 根据数据容器中包含的指定信息删除记录。 数据容器中必须提供的信息有，<br>
	 * 数据库表名和条件限定数据。 本方法根据数据容器的数据进行SQL条件拼合，<br>
	 * 而不使用数据容器的subWhere限定条件。因此本方法 仅适用于单表操作。
	 *
	 * @param ds
	 *            数据容器
	 * @return 删除结果 -1: 删除动作没有影响到任何记录<br>
	 *         其它：删除掉的记录条数
	 * @throws SQLException
	 *             SQL异常
	 */
	public int delete(RecordData ds) throws SQLException {
		String strTable = ds.getTable();
		// return value
		int nRetValue = -1;
		try {
			// 限定条件
			String strWhere = ds.getSubWhere();

			String strField = null;
			Set<String> keys = ds.singleDataKeys();
			// not fields info
			if (strWhere == null && (keys == null || keys.size() == 0)) {
				throw new SQLException("Condition data is expected "
						+ "in SQLHelper.delete(RecordData)");
			}

			if (keys != null) {
				Iterator<String> it = keys.iterator();
				while (it.hasNext()) {
					if (strWhere == null) {
						strWhere = "";
					} else {
						strWhere += " and ";
					}

					strField = it.next();
					// 使用HashMap存储，ds.getSingleData(strField)可能为null；
					// 如果为空，则设置该字段为空
					strWhere += strField + " = "
							+ getSQLValue(ds.getSingleData(strField));
				} // end loop
			}

			// INSERT
			nRetValue = executeUpdate("delete from " + strTable + " where "
					+ strWhere);
		} catch (SQLException ex) {
			throw ex;
		}

		return nRetValue;
	}

	/**
	 * 根据数据中的数据向数据库插入一条记录。 执行本方法之前，必须确定已经通过<br>
	 * RecordData.setSingleData(Table_Field_Name, Field_Value)向数据<br>
	 * 容器中添加相关字段的值。<br>
	 * 本方法会自动调用SQLHelper.getSQLValue()方法来拼接SQL语句，<br>
	 * 然后执行数据库的插入操作。所有在数据容器中不存在的字段-字段值， <br>
	 * 都将会在执行插入过程中设置为null或者默认值。<br>
	 *
	 * @param ds
	 *            数据容器
	 * @return -1: 插入失败<br>
	 *         1: 插入成功
	 *
	 * @throws SQLException
	 *             SQL异常
	 */
	public int insert(RecordData ds) throws SQLException {
		return insert(ds, true);
	}

	/**
	 * 根据数据中的数据向数据库插入一条记录。 执行本方法之前，必须确定已经通过<br>
	 * RecordData.setSingleData(Table_Field_Name, Field_Value)向数据<br>
	 * 容器中添加相关 字段的值。本方法会自动调用SQLHelper.getSQLValue()<br>
	 * 方法来拼接SQL语句，然后执行数据库的插入操作。<br>
	 *
	 * 所有在数据容器中不存在的字段-字段值， 都将会在执行插入过程中设置为null<br>
	 * 或者默认值。
	 *
	 * @param ds
	 *            数据容器
	 * @param autoPK
	 *            自动产生关键字段的值
	 * @return -1: 插入失败 1: 插入成功
	 *
	 * @throws SQLException
	 *             SQL异常
	 */
	public int insert(RecordData ds, boolean autoPK) throws SQLException {
		String strTable = ds.getTable();
		// return value
		int nRetValue = -1;
		try {
			// generate Id
			if (autoPK) {
				IDGenerator.generateId(this, ds);
			}

			// sub sql statement: fields chain and values chain
			String strFields = null;
			String strValues = null;
			String strField = null;
			Set<String> keys = ds.singleDataKeys();
			// not fields info
			if (keys == null || keys.size() == 0) {
				throw new SQLException("Record data is expected "
						+ "SQLHelper.insert(RecordData).");
			}

			Iterator<String> it = keys.iterator();
			while (it.hasNext()) {
				if (strFields == null) {
					strFields = "";
					strValues = "";
				} else {
					strFields += ", ";
					strValues += ", ";
				}

				strField = it.next();
				strFields += strField;

				// 使用HashMap存储，ds.getSingleData(strField)可能为null；
				// 如果为空，则设置该字段为空
				strValues += getSQLValue(ds.getSingleData(strField));
			} // end loop

			// INSERT
			nRetValue = executeUpdate("insert into " + strTable + "("
					+ strFields + ") values(" + strValues + ")");
		} catch (SQLException ex) {
			throw ex;
		}

		return nRetValue;
	}

	/**
	 * 根据数据容器中包含的指定信息修改一条记录。数据容器中必须提供的信息有， <br>
	 * 数据库表名和条件限定语句 执行本方法之前，必须确定已经通过 <br>
	 * RecordData.setSingleData(Table_Field_Name, Field_Value)向 数据容<br>
	 * 器中添加相关 字段的值。本方法会自动调用SQLHelper.getSQLValue()<br>
	 * 方法来拼接SQL语句，然后执行 数据库的修改操作。<br>
	 *
	 * 所有在数据容器中不存在的字段-字段值，都将会在执行修改过程中维持不变。
	 *
	 *
	 * @param ds
	 *            数据容器
	 * @return -1: 修改失败 1: 修改成功
	 *
	 * @throws SQLException SQL异常
	 */
	public int update(RecordData ds) throws SQLException {
		// 数据库表名
		String strTable = ds.getTable();
		// 条件限定语句
		String strCondition = ds.getSubWhere();

		// has sub-where?
		if (strCondition == null || strCondition.trim().length() == 0)
			throw new SQLException("Sub-where statement expected in "
					+ "SQLHelper.update(RecordData)!");

		// return value
		int nRetValue = -1;
		try {
			// update body
			String strBody = null;
			String strField = null;
			Set<String> keys = ds.singleDataKeys();
			// not fields info
			if (keys == null || keys.size() == 0) {
				throw new SQLException(
						"Record data to be updated is expected in "
								+ "SQLHelper.update(RecordData).");
			}

			Iterator<String> it = keys.iterator();
			while (it.hasNext()) {
				if (strBody == null) {
					strBody = "";
				} else {
					strBody += ", ";
				}

				strField = (String) it.next();

				// 使用HashMap存储，ds.getSingleData(strField)可能为null；
				// 如果为null，则设置该字段值为null
				strBody += strField + " = "
						+ getSQLValue(ds.getSingleData(strField));

			} // end loop

			// update
			nRetValue = executeUpdate("UPDATE " + strTable + " SET " + strBody
					+ " WHERE " + strCondition);
		} catch (SQLException ex) {
			throw ex;
		} finally {
		}

		return nRetValue;
	}

	/**
	 * 执行SQL语句，用于记录的增删改
	 *
	 * @param strSql
	 *            SQL语句
	 * @return 影响到的记录条数，如果返回值为-1，则没有任何记录受到影响
	 * @throws SQLException
	 *             SQL异常
	 */
	public int executeUpdate(String strSql) throws SQLException {

		// return value
		int nRetValue = -1;
		Statement st = null;

		try {
			// database operation statement
			st = conn.createStatement();
			// update
			nRetValue = st.executeUpdate(strSql);
		} catch (SQLException ex) {
			throw new SQLException(ex.getMessage() + "\n" + strSql);
		} finally {
			// close Statement
			if (st != null) {
				try {
					st.close();
				} catch (SQLException e) {
				}
			}
		}

		return nRetValue;
	}

	/**
	 * 从结果记录集中读取第一条记录，按照字段名和字段值的对应关系保存到<br>
	 * 数据容器中。<br>
	 *
	 * 本方法只供SQLHelper.querySingleData()调用。
	 *
	 * @param sourceData
	 *            结果记录集
	 * @param data
	 *            数据容器
	 * @throws SQLException
	 *             SQL异常
	 */
	private void readSingleData(ResultSet sourceData, RecordData data)
			throws SQLException {

		// no data
		if (sourceData == null || data == null) {
			return;
		}

		// read column name
		ResultSetMetaData col = sourceData.getMetaData();
		String arrColName[] = new String[col.getColumnCount()];

		for (int i = 0; i < arrColName.length; i++) {
			// getColumnLabel()可以获得字段别名
			// getColumnName()获得字段名
			// 2009-08-22 Xianjun
			arrColName[i] = col.getColumnLabel(i + 1);
		}

		Object objValue = null;

		// read data and save
		if (sourceData.next()) {
			// save, if the arrColName[i] is null, an exception will occur
			for (int j = 0; j < arrColName.length; j++) {
				switch (col.getColumnType(j + 1)) {
				case Types.DATE: // date: yyyy-mm-dd
				case Types.TIME: // time: yyyy-mm-dd hh:mm:ss
				case Types.TIMESTAMP: // time: yyyy-mm-dd hh:mm:ss
					objValue = sourceData.getTimestamp(arrColName[j]);
					break;
				case Types.CLOB:
					objValue = sourceData.getClob(arrColName[j]);
					break;
				case Types.BLOB:
					objValue = sourceData.getBlob(arrColName[j]);
					break;
				default:
					objValue = sourceData.getObject(arrColName[j]);
					break;
				}
				// store
				data.setSingleData(arrColName[j].toLowerCase(), objValue);
			}
		}
	}

	/**
	 * 从结果记录集中读取记录，按照字段名和字段值的对应关系保存到数据容器中。 <br>
	 *
	 * 本方法只供SQLHelper.queryMultipleData()调用。
	 *
	 * @param sourceData
	 *            结果记录集
	 * @param data
	 *            数据容器
	 * @throws SQLException
	 *             SQL异常
	 */
	private void readMultipleData(ResultSet sourceData, RecordData data)
			throws SQLException {
		// read column name
		ResultSetMetaData col = sourceData.getMetaData();
		String arrColName[] = new String[col.getColumnCount()];

		// collumn name
		for (int i = 0; i < arrColName.length; i++) {
			// getColumnLabel()可以获得字段别名
			// getColumnName()获得字段名
			// 2009-08-22 Xianjun
			arrColName[i] = col.getColumnLabel(i + 1);
		}

		Object objValue = null;

		while (sourceData.next()) {
			// read data and store
			HashMap<String, Object> row = new HashMap<String, Object>();
			// save, if the arrColName[i] is null, an exception will occur
			for (int j = 0; j < arrColName.length; j++) {
				switch (col.getColumnType(j + 1)) {
				case Types.DATE: // date: yyyy-mm-dd
				case Types.TIME: // time: yyyy-mm-dd hh:mm:ss
				case Types.TIMESTAMP: // time: yyyy-mm-dd hh:mm:ss
					objValue = sourceData.getTimestamp(arrColName[j]);
					break;
				case Types.CLOB:
					objValue = sourceData.getClob(arrColName[j]);
					break;
				case Types.BLOB:
					objValue = sourceData.getBlob(arrColName[j]);
					break;
				default:
					objValue = sourceData.getObject(arrColName[j]);
					break;
				}
				row.put(arrColName[j].toLowerCase(), objValue);
			}

			data.addMultipleData(row);
		}
	}

	/**
	 * 根据传入的数据库字段值类型来得到用于DML的SQL值
	 *
	 * @param obj
	 *            字段值
	 * @return 用于DML的SQL值
	 * @throws SQLException
	 *             传入的字段值类型不在常规处理类型范围之内
	 */
	private String getSQLValue(Object obj) throws SQLException {
		if (obj == null) {
			return "null";
		}

		// 获得数据库名称
		String dbName = getDatabaseName();
		// 字符串
		if (obj instanceof String) {
			return "'"
					+ DbUtil.getDataString((String) obj,
							getDatabaseName()) + "'";
		}
		// 数值
		if (obj instanceof Integer || obj instanceof Short
				|| obj instanceof Byte || obj instanceof Long
				|| obj instanceof BigDecimal) {
			return obj.toString();
		}

		// 数值，Float和Double的数据可能会使用科学计数法。
		// 此处转为BigDecimal就是为了防止科学计数法的出现
		if (obj instanceof Float || obj instanceof Double) {
			String str = obj.toString();
			return (new BigDecimal(str)).toString();
		}
		// 布尔
		if (obj instanceof Boolean) {
			return ((Boolean)obj).booleanValue() ? "1" : "0";
		}

		if ("Oracle".equalsIgnoreCase(dbName)) {
			// Oracle
			return getOracleSQLValue(obj);
		}

		if ("MySQL".equalsIgnoreCase(dbName)) {
			// MySQL
			return getMySQLValue(obj);
		}

		if ("Microsoft SQL Server".equalsIgnoreCase(dbName)) {
			// Microsoft SQL Server
			return getSQLServerValue(obj);
		}

		if ("ACCESS".equalsIgnoreCase(dbName)) {
			// Microsoft SQL Server
			return getSQLServerValue(obj);
		}

		// 其它数据库，待实现
		throw new SQLException("请在" + getClass().getName() + "中配置" + dbName
				+ "数据库的SQL转换！");
	}

	/**
	 * 使用Oracle方言根据传入的数据库字段值类型来得到用于DML的SQL值
	 *
	 * @param obj
	 *            字段值
	 * @return 用于DML的SQL值
	 * @throws SQLException
	 *             传入的字段值类型不在常规处理类型范围之内
	 */
	private String getMySQLValue(Object obj) throws SQLException {
		// 日期
		if (obj instanceof Date) {
			return getDBTimeValue((Date) obj);
		}

		throw new SQLException("An unkown type of field value is encountered.");
	}

	/**
	 * 使用Oracle方言根据传入的数据库字段值类型来得到用于DML的SQL值
	 *
	 * @param obj
	 *            字段值
	 * @return 用于DML的SQL值
	 * @throws SQLException
	 *             传入的字段值类型不在常规处理类型范围之内
	 */
	private String getOracleSQLValue(Object obj) throws SQLException {
		// 日期
		if (obj instanceof Date) {
			return getDBTimeValue((Date) obj);
		}

		// Clob
		if (obj instanceof Clob) {
			return "EMPTY_CLOB()";
		}

		throw new SQLException("An unkown type of field value is encountered.");
	}

	/**
	 * 使用Microsoft SQL Server方言根据传入的数据库字段值类型来得到用于DML的SQL值
	 *
	 * @param obj
	 *            字段值
	 * @return 用于DML的SQL值
	 * @throws SQLException
	 *             传入的字段值类型不在常规处理类型范围之内
	 */
	private String getSQLServerValue(Object obj) throws SQLException {
		// 日期
		if (obj instanceof Date) {
			return getDBTimeValue((Date) obj);
		}

		throw new SQLException("An unkown type of field value is encountered.");
	}

	/**
	 * 得到数据库的日期类型数据值
	 *
	 * @param dt
	 *            日期
	 * @return 日期类型数据值
	 */
	public String getDBDateValue(Date dt) {
		// 数据库名称
		String dbName = getDatabaseName();
		return getDBDateValue(dbName, dt);
	}

	/**
	 * 得到数据库的日期类型数据值
	 *
	 * @param dbName
	 *            数据库名称
	 * @param dt
	 *            日期
	 * @return 日期类型数据值
	 */
	public static String getDBDateValue(String dbName, Date dt) {
		if ("Oracle".equalsIgnoreCase(dbName)) {
			return "TO_DATE('" + LogicUtility.getDateAsString(dt)
					+ "', 'yyyy-mm-dd')";
		}

		return "'" + LogicUtility.getDateAsString(dt) + "'";
	}

	/**
	 * 得到数据库的日期时间类型数据值
	 *
	 * @param dt
	 *            日期时间
	 * @return 日期时间类型数据值
	 */
	public String getDBTimeValue(Date dt) {
		// 数据库名称
		String dbName = getDatabaseName();

		return getDBTimeValue(dbName, dt);
	}

	/**
	 * 得到数据库的日期时间类型数据值
	 *
	 * @param dbName
	 *            数据库名称
	 * @param dt
	 *            日期时间
	 * @return 日期时间类型数据值
	 */
	public static String getDBTimeValue(String dbName, Date dt) {
		if ("Oracle".equalsIgnoreCase(dbName)) {
			return "TO_DATE('" + LogicUtility.getTimeAsString(dt)
					+ "', 'yyyy-mm-dd hh24:mi:ss')";
		}

		return "'" + LogicUtility.getTimeAsString(dt) + "'";
	}

	/**
	 * 使用日期函数将DATE、DATETIME类型的字段进行格式化，格式化为yyyy-MM-dd的字符串
	 *
	 * @param fieldName
	 *            字段名称
	 * @return 字段格式化函数
	 */
	public String formatDateField(String fieldName) {
		// 获得数据库名称
		String dbName = getDatabaseName();
		return DbUtil.formatDateField(dbName, fieldName);
	}
}
