package team.bangbang.common.sql;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import com.alibaba.druid.pool.DruidDataSource;

import team.bangbang.common.config.Config;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：数据库连接获得器
/**
 * 通过配置文件config.properties中的相关配置，构造Proxool连接池
 *
 * @author 帮帮组
 * @version 1.0
 */
// ************************************************************************
public final class SQLPool {
	/* 判断是否使用了SQL数据库 */
	private static boolean useDatabase = (Config
			.getProperty("application.druid.url") != null);
	/* 是否自动提交 */
	private static boolean autocommit = false;
	/* 数据库名称 */
	private static String databaseName = null;
	/* 数据库数据源<别名, 数据源> */
	private static Map<String, DataSource> dsMap = null;
	/** 数据库别名 */
	public static final String DB_ALIAS = "ApplicationName";

	static {
		try {
			// 构造连接池
			if(useDatabase) configure();
		} catch (SQLException sqle) {
			sqle.printStackTrace();
		}
	}

	/**
	 * 获得数据库连接
	 *
	 * @return 数据库连接
	 * @throws SQLException
	 *             数据库异常
	 */
	public static Connection getConnection() throws SQLException {
		return getConnection("application");
	}

	/**
	 * 获得数据库数据源
	 *
	 * @return 数据库数据源
	 * @throws SQLException
	 *             数据库异常
	 */
	public static DataSource getDataSource() throws SQLException {
		return getDataSource("application");
	}

	/**
	 * 可能的结果有：Oracle、MySQL
	 *
	 * @return 数据库名称
	 */
	public static String getDatabaseName() {
		if(!useDatabase) {
			return null;
		}

		if (databaseName != null) {
			return databaseName;
		}

		Connection conn = null;
		try {
			conn = getConnection();
			DatabaseMetaData dmd = conn.getMetaData();
			databaseName = dmd.getDatabaseProductName();
		} catch (SQLException ex) {
			databaseName = "";
		} finally {
			// 关闭数据库连接
			if (conn != null) {
				try {
					conn.close();
				} catch (SQLException ex) {
				}
			}
		}

		return databaseName;
	}

	/**
	 * 获得数据库连接
	 *
	 * @param alias
	 *            数据库别名
	 *
	 * @return 数据库连接
	 * @throws SQLException
	 *             数据库异常
	 */
	public static Connection getConnection(String alias) throws SQLException {
		if(!useDatabase) {
			return null;
		}

		DataSource ds = getDataSource(alias);
		if (ds == null) {
			return null;
		}

		Connection c = ds.getConnection();
		if (c.getAutoCommit() != autocommit) {
			c.setAutoCommit(autocommit);
		}

		// 设置事务级别
		c.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
		return c;
	}

	/**
	 * 获得数据库数据源
	 *
	 * @param alias
	 *            数据库别名
	 * @return 数据源
	 */
	public static DataSource getDataSource(String alias) {
		if(!useDatabase) {
			return null;
		}

		if (dsMap == null) {
			dsMap = new HashMap<String, DataSource>();
		}

		DataSource ds = dsMap.get(alias);

		return ds;
	}

	/**
	 * 根据配置文件参数配置Proxool数据库连接池
	 *
	 * @throws SQLException
	 *             数据库异常
	 */
	private static void configure() throws SQLException {
		// 读取配置信息
		List<Properties> lstConfig = getDatabaseProperties();

		if (dsMap == null) {
			dsMap = new HashMap<String, DataSource>();
		}

		// 循环注册所有的数据库配置信息
		for (int i = 0; i < lstConfig.size(); i++) {
			Properties dc = (Properties) lstConfig.get(i);

			DruidDataSource dds = new DruidDataSource();
			dds.configFromPropety(dc);

			// 别名
			String alias = dc.getProperty("druid.name");
			dsMap.put(alias, dds);
		}
	}

	/**
	 * 得到配置文件中的所有数据库连接配置信息
	 *
	 * @return 所有数据库连接的配置信息，形式是List<DatabaseConfig>
	 */
	private static List<Properties> getDatabaseProperties() {
		List<Properties> lstConfig = new ArrayList<Properties>();
		Enumeration<String> er = Config.keys();
		// 数据库别名集合
		List<String> lstAlias = new ArrayList<String>();
		while (er != null && er.hasMoreElements()) {
			String key = (String) er.nextElement();
			int nIndex = key.indexOf(".druid.");
			if (nIndex < 0)
				continue;
			// 保存Alias
			String alias = key.substring(0, nIndex);
			if (!lstAlias.contains(alias))
				lstAlias.add(alias);
		}

		// 获得每个Alias表示的数据库配置
		for (int i = 0; i < lstAlias.size(); i++) {
			// 数据库别名
			String alias = lstAlias.get(i);
			Properties dc = new Properties();
			dc.setProperty("druid.name", alias);

			dc.setProperty("druid.url", Config.getProperty(alias + ".druid.url"));
			dc.setProperty("druid.driverClassName", Config.getProperty(alias + ".druid.driver-class"));
			dc.setProperty("druid.username", Config.getProperty(alias + ".druid.username"));
			dc.setProperty("druid.password", Config.getProperty(alias + ".druid.password"));

			dc.setProperty("druid.initialSize", Config.getProperty(alias + ".druid.initial-size"));
			dc.setProperty("druid.minIdle", Config.getProperty(alias + ".druid.min-idle"));
			dc.setProperty("druid.maxActive", Config.getProperty(alias + ".druid.max-active"));
			dc.setProperty("druid.maxWaitThreadCount", Config.getProperty(alias + ".druid.maxWait"));

			dc.setProperty("druid.timeBetweenEvictionRunsMillis", Config.getProperty(alias + ".druid.timeBetweenEvictionRunsMillis"));
			dc.setProperty("druid.minEvictableIdleTimeMillis", Config.getProperty(alias + ".druid.minEvictableIdleTimeMillis"));
			dc.setProperty("druid.validationQuery", Config.getProperty(alias + ".druid.validationQuery"));
			dc.setProperty("druid.testWhileIdle", Config.getProperty(alias + ".druid.testWhileIdle"));

			dc.setProperty("druid.testOnBorrow", Config.getProperty(alias + ".druid.testOnBorrow"));
			dc.setProperty("druid.testOnReturn", Config.getProperty(alias + ".druid.testOnReturn"));

			// save
			lstConfig.add(dc);
		}

		return lstConfig;
	}
}
