package team.bangbang.common.sql.generator;

import java.sql.SQLException;
import java.util.UUID;

import team.bangbang.common.data.RecordData;
import team.bangbang.common.sql.SQLHelper;
import team.bangbang.common.sql.generator.TableKey;
import team.bangbang.common.sql.generator.TableKeyMetaData;
import team.bangbang.common.utility.LogicUtility;


//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：关键字段编号创建器
/**
 * 能够创建关键字段编号，本类可能存在并发创建的冲突。<br>
 * 本类只允许被SQLHelper在新增记录的时候自动调用，不允许其他调用。
 *
 * @author 帮帮组
 * @version 1.24
 */
// ************************************************************************
public final class IDGenerator {
	/**
	 * 通过数据库表的字段信息，获得数据库表的第一个字段<br>
	 * （设计时务必设置第一个字段为唯一的关键字段，而且该字段类型必须为char型），<br>
	 * 计算该字段在数据库中存在的最大值，在该最大值上顺序加1，<br>
	 * 即为下一个关键字段的编号
	 *
	 * @param db
	 *            数据库操作对象
	 * @param wd
	 *            数据库容器，其中包含数据库表名
	 * @throws SQLException
	 *             数据库操作出错
	 */
	public static synchronized void generateId(SQLHelper db, RecordData wd)
			throws SQLException {
		String tableName = wd.getTable();
		// 获得该表的关键字段信息
		team.bangbang.common.sql.generator.TableKey tk = TableKeyMetaData.getTableKey(db.getAlias(), tableName);
		// 主键名称
		String pkName = tk.getKeyName();

		if (wd.getSingleData(pkName) != null) {
			// 用户已经指定了关键字段的值
			return;
		}

		// 下一个主键值
		/***********************************************************************
		 * ********************* version 1.0
		 *
		 * <pre>
		 * Object objNextId = tk.getNextKeyValue();
		 * if (objNextId != null) {
		 * 	// 下一个主键值有可能为空，该种情况为适应MySQL等数据库的自增长而设定
		 * 	// 把该编号字符串保存在关键字段对应的值中
		 * 	wd.addSingleData(pkName, objNextId);
		 * } else {
		 * 	throw new SQLException(&quot;关键字的值已经超出范围！&quot;);
		 * }
		 * </pre>
		 **********************************************************************/
		// max计算下一个关键字的值
		Object objNextId = getNextId(db, tk);
		if (objNextId != null) {
			// 把该编号字符串保存在关键字段对应的值中
			wd.setSingleData(pkName, objNextId);
		} else {
			throw new SQLException("关键字的值已经超出范围！");
		}
	}

	private static synchronized Object getNextId(SQLHelper db, TableKey tk)
			throws SQLException {
		// 如果关键字段是char(36)的，自动使用java.util.UUID
		if ("char".equalsIgnoreCase(tk.getKeyType()) && tk.getKeyLength() == 36) {
			return UUID.randomUUID().toString();
		}

		try {
			// 关键字段
			String pkName = tk.getKeyName();
			// 表名称
			String tableName = tk.getTableName();
			// 查询数据库，得到当前最大编号
			String sql = "SELECT MAX(" + pkName + ") AS MaxId FROM "
					+ tableName;
			RecordData wdTemp = new RecordData();
			db.querySingleData(sql, wdTemp);

			// 最大编号
			String maxId = wdTemp.getSingleString("MaxId");

			if (maxId == null || maxId.trim().length() == 0) {
				tk.setCurrentKeyNumber(0L);
			} else if (tk.getKeyType().indexOf("CHAR") >= 0) {
				// 字符型关键字
				tk.setCurrentKeyNumber(LogicUtility.parseLong(maxId.substring(tk
						.getPrefix().length()), 0L));
			} else {
				// 数字型关键字
				tk.setCurrentKeyNumber(LogicUtility.parseLong(maxId, 0L));
			}
		} catch (SQLException ex) {
			throw ex;
		}

		return tk.getNextKeyValue();
	}
}
