package team.bangbang.common.sql.generator;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import team.bangbang.common.config.Config;
import team.bangbang.common.exception.BizException;
import team.bangbang.common.sql.SQLPool;

/**
 * 使用数据库记录获得数据库关键字段的属性记录，仅处理别名为default的数据库
 *
 *
 * @author 帮帮组
 * @version 1.24 2007-2-28
 */
public class TableKeyMetaData {
	// JDBC数据库字段属性：
	// TABLE_CAT
	// TABLE_SCHEM
	// TABLE_NAME
	// COLUMN_NAME
	// DATA_TYPE
	// TYPE_NAME
	// COLUMN_SIZE
	// BUFFER_LENGTH
	// DECIMAL_DIGITS
	// NUM_PREC_RADIX
	// NULLABLE
	// REMARKS
	// COLUMN_DEF
	// SQL_DATA_TYPE
	// SQL_DATETIME_SUB
	// CHAR_OCTET_LENGTH
	// ORDINAL_POSITION
	// IS_NULLABLE

	/* 数据库关键字段的记录Map<数据库别名, Map<表名, 结构对象>> */
	private static Map<String, Map<String, TableKey>> keys = null;

	/**
	 * 将数据库关键字段的属性记录保存到Map中，Key为表名，Value为TableKey对象
	 *
	 * @param alias
	 *            数据库别名
	 */
	private static void initialize(String alias) {
		// 建立数据库关键字段的记录Map<数据库别名, Map<表名, 结构对象>>
		if (keys == null) {
			keys = new HashMap<String, Map<String, TableKey>>();
		}

		// 重新建立别名代表的数据库表结构
		Map<String, TableKey> mp = new HashMap<String, TableKey>();
		keys.put(alias, mp);

		// 数据库链接
		Connection conn = null;
		// 结果记录集
		ResultSet rs = null;
		// SQL句柄
		// Statement st = null;
		try {
			// 获得数据库的连接
			conn = SQLPool.getConnection(alias);

			String catalog = conn.getCatalog();
			DatabaseMetaData dmd = conn.getMetaData();
			// 读取设定的Schema
			String schema = getSchema(dmd, alias);

			// 获得数据库表
			String[] types = { "TABLE" };
			rs = dmd.getTables(catalog, schema, "%", types);
			// TABLE_CAT、TABLE_SCHEM、TABLE_NAME、TABLE_TYPE、REMARKS
			while (rs != null && rs.next()) {
				// 获得表的名字
				TableKey tk = new TableKey();
				String tableName = rs.getString("TABLE_NAME");

				tk.setTableName(tableName);
				// System.out.println("\t" + tableName);
				mp.put(tableName.toUpperCase(), tk);
			}
			if (rs != null) rs.close();

			if (mp.isEmpty()) {
				throw new BizException("没有找到有效的数据库Schema对象（" + schema
						+ "）！可以设定系统参数{[alia].druid.schema}"
						+ "或者更改数据库连接用户名指定Schema。");
			}

			// 获得关键字段值前缀、关键字段、关键字段的长度和当前最大的关键字段数字
			// 循环每个数据库表
			Iterator<String> it = mp.keySet().iterator();
			while (it != null && it.hasNext()) {
				Object objKey = it.next();
				TableKey tk = (TableKey) mp.get(objKey);
				// 表名
				String tableName = tk.getTableName();
				// 关键字的前缀
				String prefix = getPrefix(tableName);
				tk.setPrefix(prefix);

				// 获得关键字段
				// 获得所有数据库表的关键字段名称
				rs = dmd.getPrimaryKeys(catalog, schema, tableName);
				// 关键字的ResultSet含有的名称
				// TABLE_CAT、TABLE_SCHEM、TABLE_NAME、COLUMN_NAME、KEY_SEQ、PK_NAME
				if (rs != null && rs.next()) {
					// 获得关键字段
					String pkName = rs.getString("COLUMN_NAME");
					tk.setKeyName(pkName);
				}
				if (rs != null)  rs.close();

				// 关键字段名
				String pkName = tk.getKeyName();
				// 关键关键字段的类型、长度
				rs = dmd.getColumns(catalog, schema, tableName, pkName);
				if (rs != null && rs.next()) {
						tk.setKeyType(rs.getString("TYPE_NAME").toUpperCase());
						tk.setKeyLength(rs.getInt("COLUMN_SIZE"));
				}
				if (rs != null)  rs.close();
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException se) {
				}
			}
			if (conn != null) {
				try {
					conn.close();
				} catch (SQLException se) {
				}
			}
		}
	}

	/**
	 * 获得指定数据库表的关键字段信息
	 *
	 * @param alias
	 *            数据库连接在config.properties中配置的数据库别名
	 * @param tableName
	 *            数据库表名称
	 * @return 关键字段信息
	 */
	public static TableKey getTableKey(String alias, String tableName) {
		if (keys == null || keys.isEmpty() || keys.get(alias) == null) {
			initialize(alias);
		}

		// 获得指定别名的数据库表结构
		Map<String, TableKey> mp = keys.get(alias);
		if (mp == null) {
			return null;
		}

		return (TableKey) mp.get(tableName.toUpperCase());
	}

	/**
	 * 得到当前数据库中所有表的名称集合
	 * @param alias
	 *            数据库连接在config.properties中配置的数据库别名
	 *
	 * @return 当前数据库中所有表的名称集合
	 */
	public static Set<String> getTables(String alias) {
		if (keys == null || keys.isEmpty() || keys.get(alias) == null) {
			initialize(alias);
		}

		// 获得指定别名的数据库表结构
		Map<String, TableKey> mp = keys.get(alias);
		if (mp == null) {
			return null;
		}

		return mp.keySet();
	}

	/**
	 * 获得表关键字的前缀字母<br>
	 * 表名称为module_entity_suffix<br>
	 * 默认取suffix的首字母，但如果suffix为master或者base，<br>
	 * 则关键字的前缀字母取entity名称的首字母
	 *
	 * @param tableName
	 *            表名称
	 * @return 表关键字的前缀字母
	 */
	private static String getPrefix(String tableName) {
		// 得到表名后缀suffix的名称
		int nIndex = tableName.lastIndexOf("_");
		if (nIndex < 0)
			nIndex = tableName.lastIndexOf("8");
		String suffix = tableName.substring(nIndex + 1).toUpperCase();
		if ("BASE".equals(suffix) || "MASTER".equals(suffix)) {
			// 如果suffix为master或者base，取entity名称的首字母
			nIndex = tableName.indexOf("_");
			if (nIndex < 0)
				nIndex = tableName.indexOf("8");

			// 得到关键字段编号的前缀
			return tableName.substring(nIndex + 1, nIndex + 2).toUpperCase();
		}

		// 使用suffix的首字母
		return String.valueOf(suffix.charAt(0));
	}

	/**
	 * 获得指定数据库的schema名称
	 *
	 * @param dmd
	 *            数据库连接元数据
	 *
	 * @param alias
	 *            数据库别名
	 * @return 数据库的schema名称
	 */
	private static String getSchema(DatabaseMetaData dmd, String alias) {
		// 读取设定的Schema
		String schema = Config.getProperty(alias + ".druid.schema");

		// 如果没有设定Schema，使用用户名
		if (schema == null) {
			String username = Config
					.getProperty(alias + ".druid.username");
			// 获取数据库的schema
			ResultSet rs = null;
			try {
				rs = dmd.getSchemas();
				while (rs != null && rs.next()) {
					String temp = rs.getString(1);
					if (temp != null && temp.equalsIgnoreCase(username)) {
						schema = temp;
						break;
					}
				}
			} catch (Exception exp) {
				exp.printStackTrace();
			} finally {
				if (rs != null) {
					try {
						rs.close();
					} catch (SQLException e) {
						e.printStackTrace();
					}
				}
			}
		}

		// 这种情况发生在SQL Server中使用sa访问，schema为dbo的时候
		if (schema != null && schema.trim().length() == 0) {
			schema = null;
		}

		return schema;
	}
}
