package team.bangbang.common.utility;

import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.cipher.BangbangCipher;

/**
 * CheckValue的签名和验签
 *
 * 签名过程：<br>
 * 1. 将所有的参数按照字典排序，以&amp;间隔，拼合成key1=value1&amp;key2=value2&amp;key3=value3&amp;...格式的queryString<br>
 * 2. 在queryString后面追加keyCode，形成新的字符串s<br>
 * 3. 对字符串s进行md5编码，该编码即为签名数据<br>
 *
 * 验签过程：<br>
 * 1. 将所有的参数（除签名字段）按照字典排序，以&amp;间隔，拼合成key1=value1&amp;key2=value2&amp;key3=value3&amp;...格式的queryString<br>
 * 2. 在queryString后面追加keyCode，形成新的字符串s<br>
 * 3. 对字符串s进行md5编码，该编码即为签名数据sign1<br>
 * 4. 如果sign1与请求传递过来的签名数据相同，则表示合法<br>
 *
 * @author 帮帮组
 * @version 1.0 2016年4月29日
 */
public class CheckValueUtil {

	/**
	 * 生成签名数据
	 *
	 * @param data 请求数据，不包含签名数据，如含有签名数据，需要在调用此方法前去除
	 * @param keyCode KeyCode
	 * @return 签名数据
	 */
	public static String sign(Map<String, String> data, String keyCode) {
		// 按照字典排序构造queryString
		String sb = makeQueryString(data);

		// 按照字典排序后添加keyCode
		String s = sb + keyCode;
		try {
			byte[] bt = BangbangCipher.digest(s.getBytes("UTF-8"), BangbangCipher.MD5);
			String stringSign = BangbangCipher.byte2hex(bt);

			return stringSign;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * 生成签名数据
	 *
	 * @param data 请求数据，不包含签名数据，如含有签名数据，需要在调用此方法前去除
	 * @param keyCode KeyCode
	 * @return 签名数据
	 */
	public static String sign(JSONObject data, String keyCode) {
		// 按照字典排序构造queryString
		String sb = makeQueryString(data);

		// 按照字典排序后添加keyCode
		String s = sb + keyCode;
		try {
			byte[] bt = BangbangCipher.digest(s.getBytes("UTF-8"), BangbangCipher.MD5);

			return BangbangCipher.byte2hex(bt);
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * 按照字典排序构造queryString，空值会被忽略
	 *
	 * @param data 传递的参数
	 * @return queryString
	 */
	public static String makeQueryString(Map<String, String> data) {
		// TreeMap的KEY按照字典排序
		Map<String, String> treeData = new TreeMap<String, String>();
		for (String key : data.keySet()) {
			String value = (String) data.get(key);
			if (value == null || value.length() == 0) {
				continue;
			}

			// 对value值进行去除前后空处理
			treeData.put(key, value);
		}

		// 使用URL键值对的格式（即key1=value1&key2=value2…）拼接
		StringBuffer sb = getStringBuffer(treeData);

		return sb.toString();
	}

	/**
	 * 按照字典排序构造queryString，空值会被忽略
	 *
	 * @param data 传递的参数
	 * @return queryString
	 */
	public static String makeQueryString(JSONObject data) {
		// TreeMap的KEY按照字典排序
		Map<String, String> treeData = new TreeMap<String, String>();
		for (Object key : data.keySet()) {
			Object value = data.get((String)key);

			if (value == null) {
				value = "";
			}

			// 对value值进行去除前后空处理
			treeData.put((String)key, value.toString());
		}

		StringBuffer sb = getStringBuffer(treeData);

		return sb.toString();
	}

	private static StringBuffer getStringBuffer(Map<String, String> treeData) {
		// 使用URL键值对的格式（即key1=value1&key2=value2…）拼接
		StringBuffer sb = new StringBuffer();
		Iterator<Map.Entry<String, String>> it = treeData.entrySet().iterator();
		while (it.hasNext()) {
			Map.Entry<String, String> entry = it.next();
			String key = entry.getKey();
			String value = entry.getValue();
			if (value == null || value.length() == 0) {
				continue;
			}
			if (sb.length() > 0) {
				sb.append("&");
			}
			sb.append(key + "=" + value);
		}
		return sb;
	}
}
