package team.bangbang.common.utility;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.cipher.BangbangCipher;
import team.bangbang.common.config.Config;
import team.bangbang.common.config.Constants;

/**
 * Cookie的操作
 *
 * @author 帮帮组
 */
public class CookieUtility {
	/* 域名（格式示例：test.com） */
	private final static String DOMAIN = Config.getProperty("cookie.document.domain");
	/* 加密KEY */
	private final static byte[] KEY_DES = "77-09-19".getBytes();

	/**
	 * 添加一个ROOT下面的Cookie
	 *
	 * @param response
	 *            HTTP响应对象
	 * @param name
	 *            Cookie的名称
	 * @param value
	 *            Cookie的值
	 * @param expireSeconds
	 *            失效时间。单位是秒<br>
	 *            -1表示存储在内存中，浏览器关闭即消失。<br>
	 *            0表示不作任何存储。<br>
	 */
	public static void addCookie(HttpServletResponse response, String name, String value, int expireSeconds) {
		addCookie(response, "/", name, value, expireSeconds);
	}

	/**
	 * 添加一个Cookie
	 *
	 * @param response
	 *            HTTP响应对象
	 * @param contextPath
	 *            上下文路径
	 * @param name
	 *            Cookie的名称
	 * @param value
	 *            Cookie的值
	 * @param expireSeconds
	 *            失效时间。单位是秒<br>
	 *            -1表示存储在内存中，浏览器关闭即消失。<br>
	 *            0表示不作任何存储。<br>
	 */
	public static void addCookie(HttpServletResponse response, String contextPath, String name, String value,
			int expireSeconds) {
		// 1. 创建Cookie对象
		Cookie cookie = null;

		// 域名（格式示例：.test.com）
		if (DOMAIN != null && DOMAIN.length() > 5) {
			if(!name.startsWith(DOMAIN + ".")) name = DOMAIN + "." + name;

			cookie = new Cookie(name, value);
			cookie.setDomain(DOMAIN);
		} else {
			cookie = new Cookie(name, value);
		}

		cookie.setMaxAge(expireSeconds);

		// 2. 设定工作路径
		// 使用默认工作路径
		cookie.setPath(contextPath);

		// 3. 添加Cookie
		response.addCookie(cookie);
	}

	/**
	 * 获取指定Cookie的值
	 *
	 * @param request
	 *            HTTP请求
	 * @param name
	 *            Cookie的名字
	 * @return 参数name对应的Cookie值
	 */
	public static String getCookieValue(HttpServletRequest request, String name) {
		// 1. 取出所有Cookie
		Cookie[] cookies = request.getCookies();

		if(DOMAIN != null && DOMAIN.length() >= 4) {
			if(!name.startsWith(DOMAIN + ".")) name = DOMAIN + "." + name;
		}

		// 2. 遍历循环所有的Cookie
		for (int i = 0; cookies != null && i < cookies.length; i++) {
			Cookie c = cookies[i];

			if (c.getName().equals(name)) {
				return c.getValue();
			}
		} // end for

		return null;
	}

	/**
	 * 将当前用户信息加密变换后写入Cookie中
	 *
	 * @param response
	 *            HTTP响应对象
	 * @param user
	 *            当前用户信息，JSON格式，使用fastjson
	 * @param expireSeconds
	 *            失效时间，单位是秒<br>
	 *            -1表示存储在内存中，浏览器关闭即消失。<br>
	 *            0表示不作任何存储。<br>
	 */
	public static void putUser(HttpServletResponse response, JSONObject user, int expireSeconds) {
		if (user == null || user.keySet().isEmpty()) {
			return;
		}

		try {
			// 将当前登录的用户信息序列化为字节数组
			byte[] data = user.toString().getBytes("UTF-8");

			// 使用今天的日期(yy-MM-dd)进行DES加密
			// String ymd = LogicUtility.getDateAsString().substring(2);
			data = BangbangCipher.encryptDES(data, KEY_DES);
			// 转化为字符串
			String s = BangbangCipher.byte2hex(data);

			// 分成小段保存到Cookie中，每小段1024字节
			// 字符串总长度
			int len = s.length();
			// 分成多少段？
			int blockCount = len / 1024;
			if (len % 1024 > 0)
				blockCount++;

			for (int i = 0; i < blockCount; i++) {
				// 该小段的字符串
				int nStart = i * 1024;
				int nEnd = (i + 1) * 1024;
				if (nEnd > len) {
					nEnd = len;
				}
				String temp = s.substring(nStart, nEnd);
				// -1表示存储在内存中，浏览器关闭即消失。
				addCookie(response, Constants.KEY_CURRENT_ACCOUNT + "_" + i, temp, expireSeconds);
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		}
	}

	/**
	 * 从Cookie中获取当前用户信息。
	 *
	 * @param request
	 *            HTTP请求
	 *
	 * @return 当前用户信息
	 */
	public static JSONObject getUser(HttpServletRequest request) {
		StringBuffer sb = new StringBuffer();
		int i = 0;
		String temp = null;
		while((temp = CookieUtility.getCookieValue(request, Constants.KEY_CURRENT_ACCOUNT + "_" + i)) != null) {
			sb.append(temp);
			i++;
		}

		// 用户加密数据
		if (sb.length() == 0) {
			return null;
		}

		JSONObject json = null;
		try {
			// 转化为字节数组
			byte[] data = BangbangCipher.hex2byte(sb.toString());
			// 使用今天的日期(yy-MM-dd)进行DES加密
			// String ymd = LogicUtility.getDateAsString().substring(2);

			data = BangbangCipher.decryptDES(data, KEY_DES);

			// 形成JSON
			String s = new String(data, "UTF-8");
			json = JSONObject.parseObject(s);
		} catch (Exception ex) {
			ex.printStackTrace();
		}

		// 返回
		return json;
	}
}
