package team.bangbang.common.utility;

import java.awt.Color;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.NumberFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Random;
import java.util.StringTokenizer;
import java.util.UUID;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：静态工具类
/**
 * 所有方法均为静态工具方法，主要包括数据转换、日期操作、字符串加工以及其它在应用开发中<br>
 * 可能用到的静态工具方法
 *
 * @author 帮帮组
 * @version 1.0
 */
// ************************************************************************
public final class LogicUtility {
	/* 数字格式化实例 */
	private static NumberFormat nf = NumberFormat
			.getNumberInstance(Locale.CHINA);

	/**
	 * 判断传入的字符串是否表示一个数字
	 *
	 * @param strValue 传入的字符串
	 *
	 * @return true:可以表示一个数字 false:不可以表示一个数字
	 */
	public static boolean isNumber(String strValue) {
		if (strValue == null || strValue.trim().length() == 0)
			return false;
		// 整型？
		try {
			Integer.parseInt(strValue);
		} catch (Exception ex1) {
			return false;
		}

		// 双精度？
		try {
			Double.parseDouble(strValue);
		} catch (Exception ex1) {
			return false;
		}

		return true;
	}

	/**
	 * 得到传入的字符串字面表示的十进制整型数据，如果该字符串字面表示的不是数字，<br>
	 * 则取用默认整型数值
	 *
	 * @param strValue
	 *            字符串
	 * @param nDefaultValue
	 *            默认的整型数值
	 * @return 字面表示的十进制整型数据
	 */
	public static int parseInt(String strValue, int nDefaultValue) {
		// parse
		return (int) parseDouble(strValue, (double) nDefaultValue);
	}

	/**
	 * 得到传入的字符串字面表示的十进制长整型数据，如果该字符串字面表示的不是数字，<br>
	 * 则取用默认长整型数值
	 *
	 * @param strValue
	 *            字符串
	 * @param lDefaultValue
	 *            默认长整型
	 * @return 字面表示的十进制双精度型数据
	 */
	public static long parseLong(String strValue, long lDefaultValue) {
		// parse
		try {
			lDefaultValue = Long.parseLong(strValue);
		} catch (Exception ex) {
		}

		return lDefaultValue;
	}

	/**
	 * 得到传入的字符串字面表示的十进制双精度型数据，如果该字符串字面表示的不是数字，<br>
	 * 则取用默认双精度型数值
	 *
	 * @param strValue
	 *            字符串
	 * @param dblDefaultValue
	 *            默认双精度型
	 * @return 字面表示的十进制双精度型数据
	 */
	public static double parseDouble(String strValue, double dblDefaultValue) {
		// parse
		try {
			dblDefaultValue = Double.parseDouble(strValue);
		} catch (Exception ex) {
		}

		return dblDefaultValue;
	}

	/**
	 * 根据颜色字符串，产生对应的Color对象。
	 * @param strRgb 格式为#ABCDEF的颜色字符串
	 * @param defaultColor 如果转换失败，使用默认的颜色
	 *
	 * @return Color对象
	 */
	public static Color parseColor(String strRgb, Color defaultColor) {
		if(strRgb == null || strRgb.length() != 7) {
			return defaultColor;
		}
		// 截取#号后面的16进制数字
		strRgb = strRgb.substring(1);
		// 进行16进制转换
		int rgb = -1;
		try {
			rgb = Integer.parseInt(strRgb, 16);
		} catch (NumberFormatException nfex) {
		}

		return (rgb < 0) ? defaultColor : new Color(rgb);
	}

	/**
	 * 获得当前时间字符串，格式为yyyy-mm-dd HH24:mi:ss
	 *
	 * @return 当前时间字符串
	 */
	public static String getTimeAsString() {
		return getTimeAsString(new Date());
	}

	/**
	 * 获得时间字符串，格式为yyyy-mm-dd HH24:mi:ss
	 *
	 * @param date
	 *            时间参数
	 *
	 * @return 当前时间字符串
	 */
	public static String getTimeAsString(Date date) {
		if (date == null) {
			return "";
		}

		Calendar objDate = Calendar.getInstance();
		if (date != null) {
			objDate.setTime(date);
		}

		String strDate = objDate.get(Calendar.YEAR) + "-"
				+ to2Digits(objDate.get(Calendar.MONTH) + 1) + "-"
				+ to2Digits(objDate.get(Calendar.DATE)) + " "
				+ to2Digits(objDate.get(Calendar.HOUR_OF_DAY)) + ":"
				+ to2Digits(objDate.get(Calendar.MINUTE)) + ":"
				+ to2Digits(objDate.get(Calendar.SECOND));

		objDate = null;

		return strDate;
	}

	/**
	 * 将整数格式化到两位数字，不足两位前面补0
	 *
	 * @param n
	 *            待格式化的数字
	 * @return 格式化后产生的字符串
	 */
	private static String to2Digits(int n) {
		return (n < 10) ? "0" + n : String.valueOf(n);
	}

	/**
	 * 把日期字符串转换为java.util.Date实例，然后得到该日期距离1970年1月1日0点整的毫秒数。
	 *
	 * @param strTime
	 *            日期字符串，格式为yyyy-mm-dd或者yyyy/dd/mm *
	 * @return 毫秒数
	 */
	public static long getTimeInMillis(String strTime) {
		long lTime = 0L;
		try {
			strTime = strTime.trim();
			if (strTime.indexOf(" ") > 0)
				throw new Exception(
						"Time format must be yyyy-mm-dd or yyyy/dd/mm");
			String str = "-";
			int nIndex = strTime.indexOf(str);
			if (nIndex < 0) {
				str = "/";
				nIndex = strTime.indexOf(str);
			}
			int nYear = Integer.parseInt(strTime.substring(0, nIndex));
			strTime = strTime.substring(nIndex + 1);
			nIndex = strTime.indexOf(str);
			int nMonth = Integer.parseInt(strTime.substring(0, nIndex));
			if (--nMonth < 0) {
				nYear--;
				nMonth = 11;
			}
			int nDay = Integer.parseInt(strTime.substring(nIndex + 1));
			Calendar objDate = Calendar.getInstance();
			objDate.clear();
			objDate.set(nYear, nMonth, nDay);
			lTime = objDate.getTime().getTime();
			objDate = null;
		} catch (Exception exception) {
		}
		return lTime;
	}

	/**
	 * 得到当前日期的格式化字符串，格式为yyyy-MM-dd
	 *
	 * @return yyyy-MM-dd格式的当前日期字符串
	 */
	public static String getDateAsString() {
		return getDateAsString(new Date());
	}

	/**
	 * 得到yyyy-MM-dd格式的日期字符串
	 *
	 * @param date
	 *            传入的日期对象
	 * @return yyyy-MM-dd格式的日期字符串
	 */
	public static String getDateAsString(Date date) {
		if (date == null) {
			return "";
		}

		Calendar objDate = Calendar.getInstance();
		if (date != null) {
			objDate.setTime(date);
		}

		String strDate = objDate.get(Calendar.YEAR) + "-"
				+ to2Digits(objDate.get(Calendar.MONTH) + 1) + "-"
				+ to2Digits(objDate.get(Calendar.DATE));

		objDate = null;

		return strDate;
	}

	/**
	 * 得到yyyy-MM格式的日期字符串
	 *
	 * @param date
	 *            传入的日期对象
	 * @return yyyy-MM格式的日期字符串
	 */
	public static String getYearMonth(Date date) {
		Calendar objDate = Calendar.getInstance();
		objDate.setTime(date);

		String strDate = String.valueOf(objDate.get(Calendar.YEAR))
				+ to2Digits(objDate.get(Calendar.MONTH) + 1);

		return strDate;
	}

	/**
	 * 根据一段日期时间字符串，转换得到日期时间对象。日期时间字符串可以是以下格式：
	 * 1. yyyy-MM
	 * 2. yyyy-MM-dd
	 * 3. HH:mm
	 * 4. HH:mm:ss
	 * 5. yyyy-MM-dd HH:mm
	 * 6. yyyy-MM-dd HH:mm:ss
	 *
	 * @param str
	 *            日期时间字符串
	 * @return 日期时间对象
	 */
	public static Date parseDate(String str) {
		// 没有输入值
		if (str == null)
			return null;
		str = str.trim();
		if (str.length() == 0)
			return null;

		str = str.replace('/', '-');
		str = str.replace('\\', '-');

		// yyyy-MM-dd HH:mm:ss
		String[] sa = str.split("\\s+");
		if (sa == null || sa.length == 0)
			return null;

		// 年月日
		String s1 = null;
		// 时分秒
		String s2 = null;

		if (sa.length == 1) {
			// 可能是 年月日，也可能是 时分秒
			if (sa[0].indexOf(":") > 0) {
				s2 = sa[0];
			} else {
				s1 = sa[0];
			}
		}

		if (sa.length >= 2) {
			for (int i = 0; sa != null && i < 2 && i < sa.length; i++) {
				if (sa[i].indexOf(":") > 0 || s1 != null) {
					s2 = sa[i];
				} else {
					s1 = sa[i];
				}
			}
		}

		if (s1 == null)
			s1 = "1900-01-01";
		if (s2 == null)
			s2 = "00:00:00";

		int[] ymd = getYmd(s1);
		int[] hms = getHms(s2);

		Calendar cal = Calendar.getInstance();
		cal.set(ymd[0], ymd[1] - 1, ymd[2], hms[0], hms[1], hms[2]);
		return cal.getTime();
	}

	private static int[] getHms(String s2) {
		int[] hms = { 0, 0, 0 };
		int nEnd = s2.indexOf(":");
		if (nEnd < 0) {
			// 只有小时
			// 第1个数字
			hms[0] = LogicUtility.parseInt(s2, hms[0]);
			return hms;
		}

		int nStart = 0;
		// 时分秒
		// 第1个数字
		hms[0] = LogicUtility.parseInt(s2.substring(nStart, nEnd), hms[0]);
		// 第2个数字
		nStart = nEnd + 1;
		if (nStart >= s2.length())
			return hms;
		nEnd = s2.indexOf(":", nStart);
		if (nEnd < 0) {
			// 只有1个 -
			hms[1] = LogicUtility.parseInt(s2.substring(nStart), hms[1]);
			nEnd = s2.length();
		} else {
			hms[1] = LogicUtility.parseInt(s2.substring(nStart, nEnd), hms[1]);
		}
		// 第3个数字
		nStart = nEnd + 1;
		if (nStart >= s2.length())
			return hms;
		hms[2] = LogicUtility.parseInt(s2.substring(nStart), hms[2]);

		return hms;
	}

	private static int[] getYmd(String s1) {
		int[] ymd = { 1900, 1, 1 };
		int nEnd = s1.indexOf("-");
		if (nEnd < 0) {
			// 只有小时
			// 第1个数字
			ymd[0] = LogicUtility.parseInt(s1, ymd[0]);
			return ymd;
		}

		int nStart = 0;
		// 年月日
		// 第1个数字
		ymd[0] = LogicUtility.parseInt(s1.substring(nStart, nEnd), ymd[0]);
		// 第2个数字
		nStart = nEnd + 1;
		if (nStart >= s1.length())
			return ymd;
		nEnd = s1.indexOf("-", nStart);
		if (nEnd < 0) {
			// 只有1个 -
			ymd[1] = LogicUtility.parseInt(s1.substring(nStart), ymd[1]);
			nEnd = s1.length();
		} else {
			ymd[1] = LogicUtility.parseInt(s1.substring(nStart, nEnd), ymd[1]);
		}

		// 第3个数字
		nStart = nEnd + 1;
		if (nStart >= s1.length())
			return ymd;
		ymd[2] = LogicUtility.parseInt(s1.substring(nStart), ymd[2]);

		return ymd;
	}

	/**
	 * 判定两个日期是否是同一天
	 *
	 * @param dateA
	 *            日期A
	 * @param dateB
	 *            日期B
	 * @return true: 是同一天 false: 不是同一天
	 */
	public static boolean isSameDay(Date dateA, Date dateB) {
		// 无效日期
		if (dateA == null || dateB == null) {
			return false;
		}

		Calendar objDateA = Calendar.getInstance();
		objDateA.setTime(dateA);

		Calendar objDateB = Calendar.getInstance();
		objDateB.setTime(dateB);

		// 比较年
		if (objDateA.get(Calendar.YEAR) != objDateB.get(Calendar.YEAR))
			return false;
		// 比较月
		if (objDateA.get(Calendar.MONTH) != objDateB.get(Calendar.MONTH))
			return false;
		// 比较日
		if (objDateA.get(Calendar.DAY_OF_MONTH) != objDateB
				.get(Calendar.DAY_OF_MONTH))
			return false;

		// 年月日相等
		return true;
	}
	/**
	 * 把指定的字符串重复指定的次数，拼接一个新的字符串
	 *
	 * @param strValue
	 *            需要重复拼接的字符串
	 * @param nUnit
	 *            需要重复拼接的次数
	 * @return 拼接后的字符串
	 */
	public static String duplicateString(String strValue, int nUnit) {
		StringBuffer sb = new StringBuffer("");
		for (int i = 0; i < nUnit; i++) {
			sb.append(strValue);
		}

		return sb.toString();
	}

	/**
	 * 把指定的字符串按照分割字符串进行拆分，拆分后的所有字符串形成字符串数组。<br>
	 * 本方法支持所有Java版本，自Java 1.4版本之后，String类提供了<br>
	 * String.split(String regex)实现了该功能。
	 *
	 * @param strValue
	 *            需要拆分的字符串
	 * @param strDelim
	 *            分割字符串
	 * @return 拆分所得字符串数组
	 */
	public static String[] splitString(String strValue, String strDelim) {
		if (strValue == null || strDelim == null)
			return null;
		StringTokenizer st = new StringTokenizer(strValue,
				strDelim);
		String arrResult[] = new String[st.countTokens()];

		for (int i = 0; i < arrResult.length; i++) {
			arrResult[i] = st.nextToken();
		}

		return arrResult;
	}

	/**
	 * 用字符串表示格式化后的双精度数值，使用四舍五入
	 *
	 * @param dbValue
	 *            双精度数值
	 * @param nFraction
	 *            小数位数
	 * @return 双精度数据格式化输出的字符串
	 */
	public static String formatNumber(double dbValue, int nFraction) {
		nf.setMaximumFractionDigits(nFraction);
		return nf.format(dbValue);
	}

	/**
	 * 用字符串表示格式化后的双精度数值，使用四舍五入
	 *
	 * @param strValue
	 *            字符串性表示的双精度数值
	 * @param nFraction
	 *            小数位数
	 * @return 双精度数据（字符串表示）格式化输出的字符串
	 */
	public static String formatNumber(String strValue, int nFraction) {
		return formatNumber(parseDouble(strValue, 0.0), nFraction);
	}

	/**
	 * 把字符串转变成适合在HTML输入框、下拉框中显示的字符串
	 *
	 * @param strValue
	 *            待转变字符串
	 * @return 转变后的字符串
	 */
	public static String getQuotedString(String strValue) {
		StringBuffer stbResult = new StringBuffer("");
		if (strValue != null && strValue.length() > 0) {
			for (int i = 0; i < strValue.length(); i++) {

				switch (strValue.charAt(i)) {
				case '&':
					stbResult.append("&amp;");
					break;
				case '<':
					stbResult.append("&lt;");
					break;
				case '>':
					stbResult.append("&gt;");
					break;
				case '"':
					stbResult.append("&quot;");
					break;
				case '\\':
					stbResult.append("\\\\");
					break;
				default:
					stbResult.append(strValue.charAt(i));
					break;
				} // end switch
			} // end for

			return stbResult.toString();
		} else { // null
			return "";
		}
	}

	/**
	 * 把字符串转变成适合在HTML页面显示的字符串
	 *
	 * @param strValue
	 *            待转变字符串
	 * @return 转变后的字符串
	 */
	public static String getHTMLString(String strValue) {
		StringBuffer stbResult = new StringBuffer("");
		char cLast = ' ';

		for (int i = 0; strValue != null && i < strValue.length(); i++) {

			switch (strValue.charAt(i)) {

			case '&':
				stbResult.append("&amp;");
				break;
			case '<':
				stbResult.append("&lt;");
				break;
			case '>':
				stbResult.append("&gt;");
				break;
			case ' ':
				if (cLast == ' ') {
					stbResult.append("&nbsp;");
				} else {
					stbResult.append(" ");
				}

				break;
			case '"':
				stbResult.append("&quot;");
				break;
			case '\r':
				stbResult.append("<br/>");
				break;
			case '\n':
				// in Windows, a break line might be "\r\n"
				if (cLast != '\r')
					stbResult.append("<br>");
				break;
			case '\t':
				stbResult.append("&nbsp;&nbsp;&nbsp;&nbsp;");
				break;
			default:
				stbResult.append(strValue.charAt(i));
				break;
			} // end switch
			cLast = strValue.charAt(i);
		} // end for

		return stbResult.toString();
	}

	/**
	 * 把字符串中的空格和换行转变成HTML页面中的空格和换行
	 *
	 * @param strValue
	 *            待转变字符串
	 * @return 转变后的字符串
	 */
	public static String changeHTMLLineSpace(String strValue) {
		StringBuffer stbResult = new StringBuffer("");
		char cLast = ' ';

		if (strValue != null && strValue.length() > 0) {
			for (int i = 0; i < strValue.length(); i++) {

				switch (strValue.charAt(i)) {
				case ' ':
					if (cLast == ' ') {
						stbResult.append("&nbsp;");
					} else {
						stbResult.append(" ");
					}

					break;
				case '\r':
					if (cLast != '>')
						stbResult.append("<br/>");
					break;
				case '\n':
					// in Windows, a break line might be "\r\n"
					if (cLast != '\r' && cLast != '>')
						stbResult.append("<br/>");
					break;
				case '\t':
					stbResult.append("&nbsp;&nbsp;&nbsp;&nbsp;");
					break;
				default:
					stbResult.append(strValue.charAt(i));
					break;
				} // end switch
				cLast = strValue.charAt(i);
			} // end for
		} else { // null
			stbResult.append("&nbsp;");
		}

		return stbResult.toString();
	}

	/**
	 * 把字符串从ISO-8859-1字符集转到UTF-8字符集
	 *
	 * @param strValue
	 *            字符串
	 * @return 转变后的字符串
	 */
	public static String getCNString(String strValue) {

		if (strValue == null)
			return null;
		try {
			byte[] temp = strValue.getBytes("ISO-8859-1");
			return new String(temp, "UTF-8");
		} catch (Exception ex) {
			ex.printStackTrace();
		}

		return null;
	}

	/**
	 * 从输入流中读取字节数组
	 *
	 * @param is 输入流
	 * @return 字节数组
	 * @throws IOException IO异常
	 */
	public static byte[] readInputStream(InputStream is) throws IOException {
		ByteArrayOutputStream baos = null;
		byte[] result = null;
		try {
			baos = new ByteArrayOutputStream();

			// buffer
			byte[] buff = new byte[512];
			int nLen = 0;
			while ((nLen = is.read(buff)) > 0) {
				baos.write(buff, 0, nLen);
			}

			result = baos.toByteArray();
		} catch (IOException ex) {
			throw ex;
		} finally {
			try {
				if (baos != null)
					baos.close();
			} catch (Exception ex1) {
			}
		}

		return result;
	}

	/**
	 * 对给定的字段进行截取，截取的字段后添加省略号。 用于HTML位置不足时候的所略显示
	 *
	 * @param strValue
	 *            给定字符串
	 * @param nMaxLen
	 *            截取长度，半角字符长度为1，全角字符长度为2
	 * @return 经过截取加工的字符串
	 */
	public static String getSegmentString(String strValue, int nMaxLen) {
		if (strValue == null || nMaxLen <= 0) {
			return "";
		}
		// MaxLen >= strValue.length
		StringBuffer sb = new StringBuffer();
		// 字符串长度
		int nLen = strValue.length();
		int i = 0;
		char c = ' ';
		// 已经截取的字符长度
		int hasGet = 0;
		for (; i < nLen; i++) {
			c = strValue.charAt(i);

			if (c > 255) {
				hasGet += 2;
			} else {
				hasGet++;
			}

			// 判断是否长度到了截取的长度
			if (hasGet > nMaxLen) {
				break;
			} else {
				sb.append(c);
			}
		}

		if (i < nLen) {
			// 字符串有省略部分
			sb.append("...");
		}

		return sb.toString();
	}

	/**
	 * 将整数格式化为指定长度的字符串，不足位数的前面补0
	 *
	 * @param n
	 *            待格式化的数字
	 * @param len 指定长度
	 *
	 * @return 格式化后产生的字符串
	 */
	public static String toDigitString(int n, int len) {
		String s = String.valueOf(n);
		while(s.length() < len) {
			s = "0" + s;
		}
		return s;
	}

	/**
	 * 将整数格式化为指定长度的字符串，不足位数的前面补0
	 *
	 * @param n
	 *            待格式化的数字
	 * @param len 指定长度
	 *
	 * @return 格式化后产生的字符串
	 */
	public static String toDigitString(long n, int len) {
		String s = String.valueOf(n);
		while(s.length() < len) {
			s = "0" + s;
		}
		return s;
	}

	/**
	 * 将字符串转变为Ascii码，以便于HTTP传输。
	 * @param v 字符串
	 * @return Ascii码表示的字符串
	 */
	public static String toAsciiString(String v) {
		if(v == null || v.length() == 0) {
			return v;
		}

		// 转换后的字符串
		StringBuffer sb = new StringBuffer();

		char[] cs = v.toCharArray();
		for(int i = 0; i < cs.length; i++) {
			if(cs[i] < 128) {
				sb.append(cs[i]);
				continue;
			}

			// 双字节
			String temp = Integer.toHexString(cs[i] & 0xFFFFFFFF);
			while(temp.length() < 4) temp = "0" + temp;

			sb.append("\\u").append(temp);
		}

		return sb.toString();
	}

	/**
	 * 获取随机数
	 * @param length 随机数长度
	 * @return 随机数字
	 */
	public static String getRandomNumber(int length) {
		String serial = "0123456789";
		Random random = new Random();
		int n = serial.length();
		StringBuffer code = new StringBuffer();

		for(int i = 0; i < length; ++i) {
			code.append(serial.charAt(random.nextInt(n)));
		}

		return code.toString();
	}

	/**
	 * 获取UUID
	 * @return UUID字符串
	 */
	public static synchronized String getUUID() {
		return UUID.randomUUID().toString();
	}

	/**
	 * 获取随机字符串
	 * @param length 随机字符串长度
	 * @return 随机字符串
	 */
	public static String getRandomString(int length) {
		String codeSerial = "1234567890abcdefghkmnpqrstwxyABCDEFGHKLMNPQRSTWXY";
		Random random = new Random();
		int n = codeSerial.length();
		StringBuffer code = new StringBuffer();

		for(int i = 0; i < length; ++i) {
			code.append(codeSerial.charAt(random.nextInt(n)));
		}

		return code.toString();
	}

	/**
	 * 判断给定的字符串是否为空
	 *
	 * @param test 给定的字符串
	 * @return 是否为空
	 */
	public static boolean isEmpty(String test) {
		return test == null || test.trim().length() == 0;
	}

	/**
	 * 判断给定的字符串是否为非空
	 *
	 * @param test 给定的字符串
	 * @return 是否为非空
	 */
	public static boolean isNotEmpty(String test) {
		return test != null && test.trim().length() != 0;
	}
}
