package team.bangbang.common.utility;

import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import team.bangbang.common.file.ResourceLoader;

//************************************************************************
//系统名称：帮帮WEB开发辅助类库
//class名称：模板套印
/**
 * <pre>
 * 按照给定的模板，比对已有的符合该模板格式的文档，将模板中定义的标记在已有文档中的
 * 对应字符串。将模板中的标记和抽取到的对应字符信息匹配保存到Map中。
 * 
 * 该类用于抽取固定格式的邮件、文档中各属性信息。比如抽取简历邮件中应聘者的
 * 姓名、生日、专业、毕业院校、证件号码、联系方法、工作经历等等信息。
 * 
 * 模板中所有的标记以bb开头，比如&lt;bb:name/&gt;、&lt;bb:birthday/&gt;、&lt;bb:major/&gt;等等。
 *</pre>
 *
 * @author 帮帮组
 * @version 1.0 2007-10-13
 */
// ************************************************************************
public class TemplateDraw {
	/* 模板文件的内容 */
	private String templateContent = null;

	/**
	 * 获得模板文件的内容。
	 *
	 * @param templateFile
	 *            模板文件的名称
	 * @throws IOException IO异常
	 */
	public TemplateDraw(String templateFile) throws IOException {
		InputStream is = ResourceLoader.getResourceAsStream("template/"
				+ templateFile);
		try {
			byte[] bts = LogicUtility.readInputStream(is);
			templateContent = new String(bts);
		} catch (IOException ioe) {
			throw ioe;
		} finally {
			try {
				is.close();
			} catch (IOException ioe2) {
			}
		} // end try
	}

	/**
	 * 抽取指定文档内容中各标记代表的数据。
	 *
	 * @param doc
	 *            指定的文档内容
	 * @return 标记与标记对应的数据一一匹配，组成Map对象返回。
	 */
	public Map<String, String> drawDatas(String doc) {
		Map<String, String> mp = new HashMap<String, String>();

		// 得到模板中所有标记的位置、名称、长度
		Enumeration<TagInfo> er = getTags();
		TagInfo ti = null;
		// 上一个标记结束位置
		int index = 0;
		while ((ti = er.nextElement()) != null) {
			// 如果是第一个标记，获得在模板中该标记前边界
			if (index == 0) {
				String preBdy = templateContent.substring(0, ti.getPosition());
				// 得到这个前边界在实际文档中的位置
				int nTemp = doc.indexOf(preBdy);
				if (nTemp > 0) {
					ti.setPosition(nTemp + ti.getPosition());
				}
			}

			// 当前标记开始位置
			int nStart = index;
			if (nStart == 0) {
				nStart += ti.getPosition();
			}

			if (ti.getBoundary() == null || ti.getBoundary().length() == 0) {
				mp.put(ti.getName(), doc.substring(nStart));
				break;
			}

			// 当前标记结束位置
			int nEnd = doc.indexOf(ti.getBoundary(), nStart);

			// 标记对应内容
			String tagValue = "";
			if (nEnd > nStart) {
				// 标记名称，对应内容
				tagValue = doc.substring(nStart, nEnd);
			}
			mp.put(ti.getName(), tagValue);

			index = nEnd + ti.getBoundary().length();
		}
		return mp;
	}

	/**
	 * @return 标记在模板中出现的位置、标记的名称、标记的长度（含左右括号<>）
	 */
	private Enumeration<TagInfo> getTags() {
		return new Enumeration<TagInfo>() {
			// 当前标签的位置
			int index = -1;

			// 标记前缀
			String prefix = "<bb:";
			// 标记前缀长度
			int prefixLen = prefix.length();
			// 标记结束标记
			String suffix = "/>";
			// 标记结束标记长度
			int suffixLen = suffix.length();

			public boolean hasMoreElements() {
				int tempIndex = index;
				if (tempIndex < 0) {
					tempIndex = templateContent.indexOf(prefix);
				} else {
					tempIndex = templateContent.indexOf(prefix, tempIndex);
				}

				return (tempIndex >= 0);
			}

			// 获得下一个标记的位置、标记名称、标记长度
			public TagInfo nextElement() {
				int tempIndex = index;
				if (tempIndex < 0) {
					tempIndex = templateContent.indexOf(prefix);
				} else {
					tempIndex = templateContent.indexOf(prefix, tempIndex);
				}

				// 没有下一个标记
				if (tempIndex < 0) {
					return null;
				}

				// 有下一个标记
				TagInfo ti = new TagInfo();
				// 标记在模板中出现的位置
				ti.setPosition(tempIndex);

				// 标记结束标记
				int tagEnd = templateContent.indexOf(suffix, tempIndex);
				// 标记名称
				if (tagEnd > tempIndex + prefixLen) {
					ti.setName(templateContent.substring(tempIndex + prefixLen,
							tagEnd));
				}
				// 标记长度
				ti.setLength(tagEnd + suffixLen - tempIndex);

				// 移动标签位置指针
				index = tempIndex + ti.getLength();

				// 该标记后的分割字符串
				tempIndex = templateContent.indexOf(prefix, index);
				if (tempIndex > index) {
					// 后面还有标记
					ti.setBoundary(templateContent.substring(index, tempIndex));
				} else {
					// 后面没有标记
					ti.setBoundary(templateContent.substring(index));
				}

				// 返回下一个标签信息
				return ti;
			}
		};
	}
}

/**
 * 记录标记在模板中出现的位置、标记的名称、标记的长度（含左右括号<>）。
 *
 * @author 帮帮组
 * @version 1.0 2007-10-13
 */
class TagInfo {
	/* 标记在模板中出现的位置 */
	private int position = 0;
	/* 标记名称 */
	private String name = null;
	/* 标记长度 */
	private int length = 0;
	/* 标记之后的分界字符串 */
	private String boundary = null;

	/**
	 * @return 标记在模板中出现的位置
	 */
	public int getPosition() {
		return position;
	}

	/**
	 * @param position
	 *            标记在模板中出现的位置
	 */
	public void setPosition(int position) {
		this.position = position;
	}

	/**
	 * @return 标记名称
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name
	 *            标记名称
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return 标记长度
	 */
	public int getLength() {
		return length;
	}

	/**
	 * @param length
	 *            标记长度
	 */
	public void setLength(int length) {
		this.length = length;
	}

	/**
	 * @return 标记之后的分界字符串
	 */
	public String getBoundary() {
		return boundary;
	}

	/**
	 * @param boundary
	 *            标记之后的分界字符串
	 */
	public void setBoundary(String boundary) {
		this.boundary = boundary;
	}
}
