package team.bangbang.common.utility;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

/**
 * Tree树的操作集合
 *
 * 概念定义：<br>
 * Ancestor：处于父路径上的所有节点；<br>
 * Parent：处于父路径上的最近父节点，即上级节点<br>
 * Node：当前节点 Family：当前节点及所辖子节点<br>
 * Son：所辖子节点<br>
 *
 * @author 帮帮组
 * @version 2.0 2010-04-15
 */
public final class TreeUtil {
	/* 属性值 临时标识 的KEY值，存储Object型Value，用于树运算过程中的标记 */
	private final static String TEMPORARY_FLAG = TreeUtil.class.getName()
			+ " TEMPORARY FLAG";

	/**
	 * 禁止实例化
	 */
	private TreeUtil() {
	}

	/*
	 * ========================================================================
	 * 公有方法区
	 * ======================================================================
	 */
	/**
	 * 把一组节点列表梳理成树
	 *
	 * @param nodes
	 *            节点列表，元素为TreeNode类型
	 *
	 * @return 树
	 */
	public static TreeNode[] toTrees(List<TreeNode> nodes) {
		if (nodes == null) {
			return null;
		}

		// 得到顶级节点
		List<TreeNode> lstTop = new ArrayList<TreeNode>();
		// 将集合转换为数组
		TreeNode[] temp = new TreeNode[nodes.size()];
		nodes.toArray(temp);

		for (int i = 0; i < temp.length; i++) {
			//i=225;
			TreeNode node = temp[i];
			boolean blTop = true;
			for (int j = 0; j < nodes.size(); j++) {
				if (i == j) {
					continue;
				}

				TreeNode node2 = temp[j];
				if (node.getParentId() != null
						&& node.getParentId().equals(node2.getId())) {
					blTop = false;
				}

				node2 = null;
				if (!blTop)
					break;
			}
			if (blTop) {
				lstTop.add(node);
				// 删除顶级节点，在余下节点中找子节点 xxm已经注释
				//nodes.remove(node);
			}
			node = null;
		}
          // 删除顶级节点 xxm
		for (int i=0;i< lstTop.size(); i++) {
			nodes.remove(lstTop.get(i));
		}

		// 形成树
		TreeNode sons[] = new TreeNode[lstTop.size()];
		for (int i = 0; i < lstTop.size(); i++) {
			sons[i] = (TreeNode) lstTop.get(i);
			sons[i].addSons(getSons(nodes, sons[i].getId()));
		}
		lstTop = null;

		return sons;
	}

	/**
	 * 从源树中找到匹配指定编号的节点
	 *
	 * @param tree
	 *            源树
	 * @param objId
	 *            指定的节点编号
	 * @return 匹配指定编号的节点
	 */
	public static TreeNode findNode(TreeNode tree, Object objId) {
		if (tree == null || tree.getId() == null) {
			return null;
		}

		if (tree.getId().equals(objId)) {
			return tree;
		}

		if (!tree.hasSon()) {
			return null;
		}

		TreeNode[] sons = tree.getSons();
		for (int i = 0; i < sons.length; i++) {
			TreeNode t = findNode(sons[i], objId);
			if (t != null) {
				return t;
			}
		}

		return null;
	}

	/**
	 * 获得指定节点及所有子节点的编号集合
	 *
	 * @param tree 指定节点
	 * @return 当前节点及所有子节点的编号集合
	 */
	public static Collection<Object> getSelfAndSonIds(TreeNode tree) {
		if(tree == null) {
			return null;
		}

		Collection<Object> ids = new HashSet<Object>();
		saveSelfAndSonIds(tree, ids);

		return ids;
	}

	/**
	 * 在一棵树下找出目标编号范围内的节点，获得这些节点及其子节点的编号集合
	 *
	 * @param tree 一棵树
	 * @param ids 目标编号范围内的节点
	 * @return 目标节点及其子节点的编号集合
	 */
	public static Collection<Object> getSelfAndSonIds(TreeNode tree, Collection<Object> ids) {
		if (tree == null || ids == null || ids.size() == 0) {
			return null;
		}

		Collection<Object> resultIds = new HashSet<Object>();
		for (Object id : ids) {
			if (id == null) continue;

			if (id instanceof Long && ((Long)id) == 0L) {
				continue;
			}
			if (id instanceof Integer && ((Integer)id) == 0) {
				continue;
			}

			TreeNode node = TreeUtil.findNode(tree, id);
			if (node == null) {
				continue;
			}

			// 添加当前节点
			resultIds.add(id);

			// 该节点及子节点ID
			Collection<Object> objs = getSelfAndSonIds(node);
			if (objs == null || objs.isEmpty()) {
				continue;
			}

			resultIds.addAll(objs);
		}

		return resultIds;
	}

	/**
	 * 把一个树节点根据Id-ParentId关系插入到一棵树中
	 *
	 * @param tree
	 *            目标树
	 * @param node
	 *            等待放入目标树的节点
	 * @return 节点插入操作后的树
	 */
	public static TreeNode insertNode(TreeNode tree, TreeNode node) {
		if (node == null) {
			return tree;
		}

		if (tree == null) {
			return node;
		}

		// 遍历整棵树，找到它的父节点
		TreeNode father = findNode(tree, node.getParentId());
		if (father == null) {
			// 没有找到插入的位置，不执行插入
			return tree;
		}
		// 在父节点下添加该节点
		TreeNode[] temp = new TreeNode[1];
		temp[0] = node;
		father.addSons(temp);
		// 将各子节点进行重新排序
		father.sortSons();

		return tree;
	}

	/**
	 * 从整棵树中根据Id找到一个树节点并更新之。
	 *
	 * 更新会影响子节点的有效标识，但不会影响子节点的状态标识。
	 *
	 * @param tree
	 *            目标树
	 * @param node
	 *            等待更新到目标树中的节点
	 * @return 节点更新操作后的树
	 */
	public static TreeNode updateNode(TreeNode tree, TreeNode node) {
		if (tree == null || node == null) {
			return tree;
		}

		// 遍历整棵树，找到它的原有节点
		TreeNode old = findNode(tree, node.getId());
		if (old == null) {
			// 没有找到，不执行更新
			return tree;
		}

		// 找到了，但tree就是node要更新的节点
		if (tree.equals(old)) {
			// 直接返回node
			return node;
		}

		node.addSons(old.getSons());

		// 判断是否和原有的节点有效标识相同
		Boolean obj1 = old.getActiveFlag();
		Boolean obj2 = node.getActiveFlag();
		if (obj1 == null || !obj1.equals(obj2)) {
			// 如果有效标识也变化了，则影响当前节点的所有子节点
			setActiveFlag(node, (Boolean) obj2);
		}

		// 找到原有节点的父节点
		TreeNode oldFather = findNode(tree, old.getParentId());
		// 删除原有子节点
		oldFather.removeSon(old);

		// 重新添加该节点
		insertNode(tree, node);

		return tree;
	}

	/**
	 *
	 * 从整棵树中根据Id找到一个树节点并删除之。
	 *
	 * @param tree
	 *            目标树
	 * @param node
	 *            等待从目标树中删除的节点
	 * @return 节点删除操作后的树
	 */
	public static TreeNode deleteNode(TreeNode tree, TreeNode node) {
		if (tree == null || node == null) {
			return tree;
		}

		// 遍历整棵树，找到它的原有节点
		TreeNode old = findNode(tree, node.getId());
		if (old == null) {
			// 没有找到，不执行删除
			return tree;
		}

		// 找到了，但tree就是node要删除的节点
		if (tree.equals(old)) {
			// 删除整个树
			return null;
		}

		// 找到原有节点的父节点
		TreeNode oldFather = findNode(tree, old.getParentId());
		// 删除原有子节点
		oldFather.removeSon(old);

		return tree;
	}

	/**
	 * 从一棵树中剔除允许范围列表之外的所有节点，仅保留允许范围内节点及其子节点。
	 *
	 * 注意保留允许范围内节点及其子节点，不包括允许节点的祖先节点。
	 *
	 * @param tree
	 *            需要过滤的树
	 * @param ids
	 *            允许的节点Id集合
	 *
	 * @return 剔除操作完成之后的树数组，因为Without Ancestor，所以此时有可能是多棵树。
	 */
	public static TreeNode[] filterWithoutAncestor(TreeNode tree,
                                                                         Collection<? extends Object> ids) {
		// 不需要保留任何节点
		if (tree == null || ids == null || ids.size() == 0) {
			return null;
		}

		TreeNode[] result = null;
		// 最高顶节点是否在允许的列表中
		if (ids.contains(tree.getId())) {
			result = new TreeNode[1];
			result[0] = tree;
			return result;
		}

		// 当前节点不在允许的组织机构列表中，检查子节点
		if (tree.hasSon()) {
			TreeNode[] sons = tree.getSons();
			for (int i = 0; i < sons.length; i++) {
				// 处理子节点
				result = appendArray(result,
						filterWithoutAncestor(sons[i], ids));
			}
		}

		return result;
	}

	/**
	 * 从一棵树中剔除允许范围列表之外的所有节点，保留允许范围内节点（及子节点）及其父节点。
	 *
	 * 注意此方法返回结果包括允许节点的所有父路径上的节点，可以用于构造登录后的功能菜单。
	 *
	 * @param tree
	 *            需要过滤的树
	 * @param ids
	 *            允许的节点Id集合
	 *
	 * @return 剔除操作完成之后的树，因为With Ancestor，所以此时仍然是一棵树。
	 */
	public static TreeNode filterWithAncestor(TreeNode tree,
                                                                    Collection<? extends Object> ids) {
		return filterWithAncestor(tree, ids, true);
	}

	/**
	 * 从一棵树中剔除允许范围列表之外的所有节点，保留允许范围内节点（可以指定是否含子节点）及其父节点。
	 *
	 * 注意此方法返回结果包括允许节点的所有父路径上的节点。
	 *
	 * @param tree
	 *            需要过滤的树
	 * @param ids
	 *            允许的节点Id集合
	 * @param retainSon
	 *            是否保留子节点
	 *
	 * @return 剔除操作完成之后的树，因为With Ancestor，所以此时仍然是一棵树。
	 */
	public static TreeNode filterWithAncestor(TreeNode tree,
                                                                    Collection<? extends Object> ids, boolean retainSon) {
		// 不需要保留任何节点
		if(tree == null || ids == null || ids.size() == 0) {
			return null;
		}

		if(retainSon) {
			// 在树上对指定节点及其子节点进行标记
			markFamily(null, tree, ids);
		} else {
			// 在树上对指定节点进行标记
			for(Object id : ids) {
				TreeNode node = findNode(tree, id);
				if(node != null) {
					node.setAttribute(TEMPORARY_FLAG, new Boolean(true));
				}
			}
		}

		// 将标记后的节点，反向标记祖先
		markAncestor(tree, ids);

		// 删除所有未标记的节点
		tree = removeUnmarkedNodes(tree);

		// 去除临时标记
		removeMark(tree);

		return tree;
	}

	/*
	 * ========================================================================
	 * 私有方法区（供上述公有方法调用）
	 * ======================================================================
	 */
	/**
	 * 获得指定节点下的子节点
	 *
	 * @param nodes
	 *            节点列表，元素为TreeNode类型
	 * @param parentid
	 *            父节点编号，父节点编号为null或者""时，表示获取顶节点
	 * @return 指定节点下的子节点
	 */
	private static TreeNode[] getSons(List<TreeNode> nodes,
                                                            Object parentid) {
		if(nodes == null || nodes.isEmpty()) {
			return new TreeNode[0];
		}

		// clone一个容器
		List<TreeNode> leftNodes = new ArrayList<TreeNode>();
		for(int i = 0; i < nodes.size(); i++) {
			TreeNode node = nodes.get(i);
			leftNodes.add(node);
		} // end while

		return getSons_(leftNodes, parentid);
	}

	/**
	 * 获得指定节点下的子节点
	 *
	 * @param nodes
	 *            节点列表，元素为TreeNode类型
	 * @param parentid
	 *            父节点编号，父节点编号为null或者""时，表示获取顶节点
	 * @return 指定节点下的子节点
	 */
	private static TreeNode[] getSons_(List<TreeNode> nodes,
                                                             Object parentid) {
		TreeNode sons[] = null;
		if (parentid == null) {
			parentid = "";
		}

		TreeNode[] tns = new TreeNode[nodes.size()];

		tns = nodes.toArray(tns);

		for(int i = 0; i < tns.length; i++) {
			TreeNode node = (TreeNode)tns[i];
			// 判断该节点的父编号是否和要求的父编号一致
			if (parentid.equals(node.getParentId())) {
				// 添加
				if (sons == null) {
					sons = new TreeNode[1];
					sons[0] = node;
				} else {
					TreeNode temp[] = sons;
					sons = new TreeNode[temp.length + 1];
					for (int j = 0; j < temp.length; j++) {
						sons[j] = temp[j];
					}
					sons[temp.length] = node;
					temp = null;
				}

				// 删除已经归位的TreeNode
				nodes.remove(node);
			}
		}

		if(sons == null || sons.length == 0) {
			return new TreeNode[0];
		}

		// 给子节点添加孙子
		for(int i = 0; i < sons.length; i++) {
			TreeNode node = sons[i];
			// 添加子Node的子Node
			node.addSons(getSons_(nodes, node.getId()));
		}

		return sons;
	}

	/**
	 * 对一棵树中的节点进行反向标记，如果该节点是指定的节点祖先，则标记该节点。
	 *
	 * 此方法不增加、不删除目标树上的任何节点。
	 *
	 * @param tree
	 *            需要过滤的树
	 * @param ids
	 *            允许的节点编号集合
	 */
	private static void markAncestor(TreeNode tree,
                                     Collection<? extends Object> ids) {
		if (tree == null || ids == null || ids.size() == 0) {
			return;
		}

		// 检查子节点
		boolean hasMarkedSon = false;
		if (tree.hasSon()) {
			TreeNode[] sons = tree.getSons();
			for (int i = 0; i < sons.length; i++) {
				markAncestor(sons[i], ids);
				// 子节点临时标记
				Boolean bl = (Boolean) sons[i].getAttribute(TEMPORARY_FLAG);
				hasMarkedSon = hasMarkedSon
						|| (bl != null && bl.booleanValue());
			}
		}

		// 根据子节点的临时标记，标记父节点
		if (hasMarkedSon) {
			tree.setAttribute(TEMPORARY_FLAG, new Boolean(true));
		}
	}

	/**
	 * 对一棵树中的节点进行标记，如果该节点编号在指定的编号（包括其子节点编号）范围内，<br>
	 * 则标记该节点。
	 *
	 * 此方法不增加、不删除目标树上的任何节点。
	 *
	 * @param parentMarkFlag
	 *            父节点的临时标记
	 * @param tree
	 *            需要过滤的树
	 * @param ids
	 *            允许的节点编号集合
	 */
	private static void markFamily(Boolean parentMarkFlag, TreeNode tree,
			Collection<? extends Object> ids) {
		if (tree == null || ids == null || ids.size() == 0) {
			return;
		}

		// 当前节点ID
		Object currentId = tree.getId();
		// 最高顶节点是否在允许的列表中
		if ((parentMarkFlag != null && parentMarkFlag.booleanValue())
				|| ids.contains(currentId)) {
			tree.setAttribute(TEMPORARY_FLAG, new Boolean(true));
		}

		// 检查子节点
		if (tree.hasSon()) {
			TreeNode[] sons = tree.getSons();
			for (int i = 0; i < sons.length; i++) {
				// 处理子节点
				markFamily((Boolean) tree.getAttribute(TEMPORARY_FLAG),
						sons[i], ids);
			}
		}
	}

	/**
	 * 检查一个节点是否已经被标记
	 *
	 * @param node
	 *            待检查的节点
	 * @return true: 被标记 false: 没有标记
	 */
	public static boolean isMarked(TreeNode node) {
		Boolean bl = (Boolean) node.getAttribute(TEMPORARY_FLAG);
		return (bl != null && bl.booleanValue());
	}

	/**
	 * 删除所有未标记的节点
	 *
	 * @param tree
	 *            等待处理的树
	 * @return 删除所有未标记的节点后的树
	 */
	private static TreeNode removeUnmarkedNodes(TreeNode tree) {
		if (tree == null) {
			return null;
		}

		// 检查子节点
		if (tree.hasSon()) {
			TreeNode[] sons = tree.getSons();
			for (int i = 0; i < sons.length; i++) {
				removeUnmarkedNodes(sons[i]);

				if (canDelete(sons[i])) {
					// 该叶子节点未标记
					tree.removeSon(sons[i]);
				}
			}
		}

		return tree;
	}

	/**
	 * @param node
	 *            等待检查的节点
	 * @return true: 当前节点没有子节点，(没有临时标记，或者临时标记为false)
	 */
	private static boolean canDelete(TreeNode node) {
		boolean bl1 = !node.hasSon();
		// 当前节点的临时标记
		Boolean bl = (Boolean) node.getAttribute(TEMPORARY_FLAG);
		boolean bl2 = (bl == null || !bl.booleanValue());

		return bl1 && bl2;
	}

	/**
	 * 去除一棵树上的所有节点的所有临时标记
	 *
	 * @param tree
	 *            等待去除临时标记的树
	 */
	private static void removeMark(TreeNode tree) {
		if (tree == null) {
			return;
		}

		// 去除当前节点的临时标记
		tree.setAttribute(TEMPORARY_FLAG, null);

		// 去除子节点的临时标记
		if (tree.hasSon()) {
			TreeNode[] sons = tree.getSons();
			for (int i = 0; i < sons.length; i++) {
				// 处理子节点
				removeMark(sons[i]);
			}
		}
	}

	/**
	 * 刷新给定的节点及其下所有子节点的有效标识
	 *
	 * @param node
	 *            给定的树节点
	 * @param bl
	 *            新的有效标识
	 */
	private static void setActiveFlag(TreeNode node, Boolean bl) {
		node.setActiveFlag(bl);
		// 处理子节点
		if (node.hasSon()) {
			TreeNode[] sons = node.getSons();
			for (int i = 0; i < sons.length; i++) {
				setActiveFlag(sons[i], bl);
			}
		}
	}

	/**
	 * 将一个对象追加到一个对象数组的最后，形成一个新的对象数组后返回。
	 *
	 * @param objs
	 *            对象数组
	 * @param toAdd
	 *            要加入对象数组的对象
	 * @return 新的对象数组
	 */
	private static TreeNode[] appendArray(TreeNode[] objs, TreeNode[] toAdd) {
		if (toAdd == null) {
			return objs;
		}

		if (objs == null) {
			return toAdd;
		}

		int count1 = objs.length;
		int count2 = toAdd.length;
		TreeNode[] newArray = new TreeNode[count1 + count2];
		// read old values
		for (int i = 0; i < count1; i++) {
			newArray[i] = objs[i];
		}

		// add new value
		for (int i = 0; i < count2; i++) {
			newArray[count1 + i] = toAdd[i];
		}

		return newArray;
	}

	/**
	 * 获得指定节点及所有子节点的编号集合
	 *
	 * @param tree 指定节点
	 * @param ids 存储当前节点及所有子节点的编号集合
	 */
	private static void saveSelfAndSonIds(TreeNode tree, Collection<Object> ids) {
		if(tree == null || tree.getId() == null) {
			return;
		}

		Object id = tree.getId();
		if(id instanceof String && ((String)id).trim().length() == 0) {
			id = null;
		}

		if(id != null) ids.add(id);

		// 处理子节点
		TreeNode[] sons = tree.getSons();
		for(int i = 0; sons != null && i < sons.length; i++) {
			saveSelfAndSonIds(sons[i], ids);
		}
	}
}
