package team.bangbang.config.service;

import java.util.Collections;
import java.util.List;

import org.ehcache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.CommonMPI;
import team.bangbang.common.data.CustomizeCache;
import team.bangbang.common.net.http.HttpClient;
import team.bangbang.common.net.http.ResponseHandler;
import team.bangbang.config.data.Item;

/**
 * 选项配置 - Service
 *
 * @author 帮帮组
 * @version 1.0 2018-10-13
 */
public final class ItemService {
	/* 日志对象 */
	private  final static Logger logger = LoggerFactory.getLogger(ItemService.class);
	/* 缓存 */
	private static Cache<String, Object> cache = CustomizeCache.createCache("ItemService", 300);
	/* http请求客户端 */
	private static HttpClient hc = new HttpClient();

	/**
	 * 查询多条选项配置，并转化为相应的POJO对象列表
	 *
	 * @param category   所属分类
	 * @param parentCode 父级选项编码
	 *
	 * @return 返回结果记录，并转化为相应的POJO对象列表
	 */
	public static List<Item> getItemList(String category, String parentCode) {
		String cacheKey = "ItemList:" + category + ":" + parentCode;
		// 检查缓存是否存在
		if(cache != null && cache.containsKey(cacheKey)) {
			try {
				String s = (String)cache.get(cacheKey);
				return JSONArray.parseArray(s).toJavaList(Item.class);
			} catch (Exception ex) {
				return null;
			}
		}

		// 权限管理系统的根地址，包括context-path，以 / 符号结尾
		String consoleRoot = CommonMPI.getConsoleRoot();
		if (consoleRoot == null || consoleRoot.length() < 5) {
			return Collections.emptyList();
		}

		if (category == null || category.trim().length() == 0) {
			return Collections.emptyList();
		}

		// 请求服务
		String service = consoleRoot + "api/config/getItemList";

		try {
			String data = "category=" + category;
			if (parentCode != null) {
				data += "&parentCode=" + parentCode;
			}

			ResponseHandler rh = hc.post(service, data, HttpClient.FormMime);
			JSONObject json = rh.toJSONObject();

			if(rh.getHttpStatus() != 200) {
				logger.error("调用 " + service + " 出现异常：" + json);
				return null;
			}

			// 数据
			JSONArray arr = json.containsKey("dto") ? json.getJSONArray("dto") : null;

			if(cache != null && arr != null) {
				cache.put(cacheKey, arr != null ? arr.toJSONString() : null);
			}

			return (arr != null) ? arr.toJavaList(Item.class) : null;
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("调用 " + service + " 出现异常：" + e.getMessage());
		}

		return Collections.emptyList();
	}

	/**
	 * 获取一个选项
	 *
	 * @param category 所属分类，必填
	 * @param itemCode 选项编码，必填
	 *
	 * @return 一个选项
	 */
	public static Item getItem(String category, String itemCode) {
		return getItem(category, itemCode, null);
	}

	/**
	 * 通过选项编码或者选项名称获取一个选项
	 *
	 * @param category 所属分类，必填
	 * @param itemCode 选项编码，选项编码与选项名称条件二选一必填
	 * @param itemName 选项名称，选项编码与选项名称条件二选一必填
	 *
	 * @return 一个选项
	 */
	public static Item getItem(String category, String itemCode, String itemName) {
		if (category == null || category.trim().length() == 0) {
			return null;
		}

		if ((itemCode == null || itemCode.trim().length() == 0)
				&& (itemName == null || itemName.trim().length() == 0)) {
			return null;
		}

		if(itemCode == null) itemCode = "";
		if(itemName == null) itemName = "";

		String cacheKey = "Item:" + category + ":" + itemCode + ":" + itemName;
		// 检查缓存是否存在
		if(cache != null && cache.containsKey(cacheKey)) {
			try {
				String s = (String)cache.get(cacheKey);
				return JSONObject.parseObject(s).toJavaObject(Item.class);
			} catch (Exception ex) {
				return null;
			}
		}

		// 权限管理系统的根地址，包括context-path，以 / 符号结尾
		String consoleRoot = CommonMPI.getConsoleRoot();
		if (consoleRoot == null || consoleRoot.length() < 5) {
			return null;
		}

		// 请求服务
		String service = consoleRoot + "api/config/getItem";

		try {
			String data = "category=" + category + "&itemCode=" + itemCode + "&itemName=" + itemName;

			ResponseHandler rh = hc.post(service, data, HttpClient.FormMime);
			JSONObject json = rh.toJSONObject();

			if(rh.getHttpStatus() != 200) {
				logger.error("调用 " + service + " 出现异常：" + json);
				return null;
			}

			// 数据
			json = json.containsKey("dto") ? json.getJSONObject("dto") : null;

			if(cache != null && json != null) {
				cache.put(cacheKey, json != null ? json.toJSONString() : null);
			}

			return (json != null) ? json.toJavaObject(Item.class) : null;
		} catch (Exception e) {
			e.printStackTrace();
			logger.error("调用 " + service + " 出现异常：" + e.getMessage());
		}

		return null;
	}
}
