package team.bangbang.sso;

import java.lang.reflect.Constructor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import team.bangbang.common.config.Config;
import team.bangbang.common.config.Constants;
import team.bangbang.common.data.ThreadVariable;
import team.bangbang.common.utility.CookieUtility;
import team.bangbang.common.utility.LogicUtility;

/**
 * 单点登录上下文，用于生成、获取IApplicationSSO、IAccountSSO、IDataLimitSSO、IFunctionLimitSSO对象
 * 
 * 本类下面的所有static方法必须在SSOFilter启用的情况下才有意义<br>
 * 
 * @see team.bangbang.common.filter.SSOFilter
 *
 * @author Bangbang
 * @version 1.0 2020年12月22日
 * @version 1.1 2020-12-25
 *          从仅仅支持Header传入token，扩大为支持header、cookie、parameter传入token
 *          获取token的优先顺序：header 优于 cookie 优于 parameter，以上3种方式前端未传入token时，使用后端生成的token
 *          SSOContext支持前端3种方式传入token，目的是为了兼容不同架构的业务系统对接。为安全起见，推荐使用header方式传入token
 */
public class SSOContext {
	/* 日志对象 */
	private final static Logger logger = LoggerFactory.getLogger(SSOContext.class);
	/** HTTP请求的KEY */
	public final static String KEY_HTTP_SERVLET_REQUEST = "KEY_HTTP_SERVLET_REQUEST";
	/** HTTP响应的KEY */
	public final static String KEY_HTTP_SERVLET_RESPONSE = "KEY_HTTP_SERVLET_RESPONSE";

	/**
	 * 创建一个单点登录上下文
	 * 
	 * @param request  HTTP请求
	 * @param response HTTP响应
	 */
	public SSOContext(HttpServletRequest request, HttpServletResponse response) {
		ThreadVariable.setVariable(KEY_HTTP_SERVLET_REQUEST, request);
		ThreadVariable.setVariable(KEY_HTTP_SERVLET_RESPONSE, response);

		// 清除原有数据
		ThreadVariable.removeVariable(Constants.KEY_SSO_ACCOUNT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_APPLICATION);
		ThreadVariable.removeVariable(Constants.KEY_SSO_DATA_LIMIT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_FUNCTION_LIMIT);
		ThreadVariable.removeVariable(Constants.KEY_SSO_TOKEN);

		// 允许通过header向客户端传递token
		response.setHeader("Access-Control-Allow-Headers", "content-type,token");
		response.setHeader("Access-Control-Expose-Headers", "token");

		// 通过header段返回给客户端
		// 返回token
		String token = getToken();
		response.setHeader("token", token);
		// 延长Token票据有效期
		TokenBinder.refreshToken(token);
	}

	/**
	 * @return 获得HTTP请求
	 */
	public static HttpServletRequest getHttpRequest() {
		return (HttpServletRequest) ThreadVariable.getVariable(KEY_HTTP_SERVLET_REQUEST);
	}

	/**
	 * @return 获得HTTP响应
	 */
	public static HttpServletRequest getHttpResponse() {
		return (HttpServletRequest) ThreadVariable.getVariable(KEY_HTTP_SERVLET_RESPONSE);
	}

	/**
	 * 获取当前应用系统的编号
	 * 
	 * 首先从http请求中获取，如果http请求中没有传入的话，则从配置文件中读取
	 * 
	 * @return 当前应用系统的编号
	 */
	public static String getApplicationId() {
		HttpServletRequest request = getHttpRequest();
		if (request == null) return null;
		
		String appId = request.getParameter("applicationId");
		if (appId == null || appId.trim().length() == 0) {
			appId = Config.getProperty("sso.application.id");
		}
		
		return appId;
	}

	/**
	 * @return Token票据
	 */
	public static String getToken() {
		String token = (String) ThreadVariable.getVariable(Constants.KEY_SSO_TOKEN);
		if (token == null || token.trim().length() == 0) {
			HttpServletRequest request = getHttpRequest();
			if (request == null) return null;
			
			// 检查前端传入的Token
			// 优先从header获取
			token = request.getHeader("token");
			// 其次从cookie获取
			if (token == null || token.trim().length() == 0) {
				token = CookieUtility.getCookieValue(request, "token");
			}
			// 再次从parameter获取
			if (token == null || token.trim().length() == 0) {
				token = request.getParameter("token");
			}

			// 前端未传入token，从后端生成
			if (token == null || token.trim().length() == 0) {
				token = LogicUtility.getUUID();
			}
			
			// 保存到线程变量中
			ThreadVariable.setVariable(Constants.KEY_SSO_TOKEN, token);
		}

		return token;
	}

	/**
	 * @return 获取IApplicationSSO对象
	 */
	public static IApplicationSSO getApplicationSSO() {
		Object sso = ThreadVariable.getVariable(Constants.KEY_SSO_APPLICATION);
		if (sso == null) {
			// 创建相应的SSO对象
			sso = newSSO("sso.application.class", Constants.KEY_SSO_APPLICATION);
		}

		return (IApplicationSSO) sso;
	}

	/**
	 * @return 获取IAccountSSO对象
	 */
	public static IAccountSSO getAccountSSO() {
		Object sso = ThreadVariable.getVariable(Constants.KEY_SSO_ACCOUNT);
		if (sso == null) {
			// 创建相应的SSO对象
			sso = newSSO("sso.account.class", Constants.KEY_SSO_ACCOUNT);
		}

		return (IAccountSSO) sso;
	}

	/**
	 * @return 获取IDataLimitSSO对象
	 */
	public static IDataLimitSSO getDataLimitSSO() {
		Object sso = ThreadVariable.getVariable(Constants.KEY_SSO_DATA_LIMIT);
		if (sso == null) {
			// 创建相应的SSO对象
			sso = newSSO("sso.data-limit.class", Constants.KEY_SSO_DATA_LIMIT);
		}

		return (IDataLimitSSO) sso;
	}

	/**
	 * @return 获取IFunctionLimitSSO对象
	 */
	public static IFunctionLimitSSO getFunctionLimitSSO() {
		Object sso = ThreadVariable.getVariable(Constants.KEY_SSO_FUNCTION_LIMIT);
		if (sso == null) {
			// 创建相应的SSO对象
			sso = newSSO("sso.function-limit.class", Constants.KEY_SSO_FUNCTION_LIMIT);
		}

		return (IFunctionLimitSSO) sso;
	}

	/**
	 * 创建一个SSO对象
	 * 
	 * @param clazzNameKey 该SSO对象的类名在application-*.properties文件中的配置KEY
	 * @param key          生成的SSO对象，保存在线程变量中的KEY值
	 * @return SSO对象
	 */
	private static Object newSSO(String clazzNameKey, String key) {
		// 创建相应的SSO对象
		// 实现类
		String clazz = Config.getProperty(clazzNameKey);
		if (clazz == null || clazz.trim().length() == 0)
			return null;

		try {
			Constructor<?> cst = Class.forName(clazz).getDeclaredConstructor();
			Object sso = cst.newInstance();
			ThreadVariable.setVariable(key, sso);

			return sso;
		} catch (Exception ex) {
			logger.error("使用无参数构造方法实例化 " + clazz + " 类失败：" + ex.getMessage());
		}

		return null;
	}
}
