package team.bangbang.sso;

import com.alibaba.fastjson.JSONObject;

import team.bangbang.common.config.Config;
import team.bangbang.common.config.Constants;
import team.bangbang.common.redis.RedisUtil;
import team.bangbang.common.utility.LogicUtility;
import team.bangbang.sso.data.Account;
import team.bangbang.sso.data.DataLimit;

import java.util.Set;

/**
 * 单点登录服务端 - Token绑定
 *
 * @author ICode Studio
 * @version 1.0  2020年12月10日
 */
public class TokenBinder {
	/* TOKEN票据过期秒数，默认1800，半小时，可以在配置文件中使用“sso.token.expire.seconds”修改数值 */
	/* 该数值可以动态修改，当SSO服务器修改会话策略的过期秒数时，可以同步修改本参数 */
	private static int SSO_TOKEN_EXPIRE_SECONDS = LogicUtility.parseInt(Config.getProperty("sso.token.expire.seconds"), 1800);

	/**
	 * 设置当前登录信息超期时间，单位：秒
	 *
	 * 当使用分布式微服务时，本方法可能不会及时刷新
	 *
	 * @param seconds 当前登录信息超期时间，单位：秒
	 */
	public static void setTimeOut(int seconds) {
		if(seconds > 0) {
			SSO_TOKEN_EXPIRE_SECONDS = seconds;
		}
	}

	/**
	 * 将验证码与token票据绑定并保存
	 * 
	 * @param token  token票据
	 * @param vCode 验证码
	 */
	public static void saveValidationCode(String token, String vCode) {
		if (token == null || vCode == null || vCode.length() == 0) {
			return;
		}
		String key = token + "_VALIDATION_CODE";
		
		// 将Account以token为key值，放入Redis中
		RedisUtil.setString(key, vCode, SSO_TOKEN_EXPIRE_SECONDS);
	}

	/**
	 * 根据token票据获取绑定的验证码
	 * 
	 * @param token token票据
	 * @return 验证码
	 */
	public static String getValidationCode(String token) {
		String key = token + "_VALIDATION_CODE";
		// 从Redis中获取登录验证码
		String v = RedisUtil.getString(key);
		if (v == null || v.trim().length() == 0) {
			return null;
		}
		
		return v;
	}
	
	/**
	 * 删除Token票据绑定的验证码
	 * 
	 * @param token  token票据
	 */
	public static void deleteValidationCode(String token) {
		// 验证码
		String key = token + "_VALIDATION_CODE";
		RedisUtil.setString(key, null, 0);
	}
	
	/**
	 * 将账户信息与token票据绑定并保存
	 * 
	 * @param token  token票据
	 * @param account 账户信息
	 */
	public static void saveAccount(String token, Account account) {
		if (token == null || account == null) {
			return;
		}
		
		// 将Account以token为key值，放入Redis中
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token;
		RedisUtil.setString(key, JSONObject.toJSONString(account), SSO_TOKEN_EXPIRE_SECONDS);
	}

	/**
	 * 根据token票据获取绑定的账户信息
	 * 
	 * @param token token票据
	 * @return 账户信息
	 */
	public static Account getAccount(String token) {
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token;
		// 从Redis中获取登录帐号信息
		String v = RedisUtil.getString(key);
		if (v == null || v.trim().length() == 0) {
			return null;
		}
		
		// 账户信息
		Account acc = null;
		try {
			JSONObject json = JSONObject.parseObject(v);
			
			acc = JSONObject.toJavaObject(json, Account.class);
		} catch (Exception ex) {
		}
		
		if (acc != null) {
			return acc;
		}
		
		return null;
	}
	
	/**
	 * 删除token票据绑定的账户信息
	 * 
	 * @param token  token票据
	 */
	public static void deleteAccount(String token) {
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token;
		// 将账户信息
		RedisUtil.setString(key, null, 0);
	}

	/**
	 * 将数据权限与token票据绑定并保存
	 * 
	 * @param token  token票据
	 * @param dataLimit 数据权限
	 */
	public static void saveDataLimit(String token, DataLimit dataLimit) {
		if (token == null || dataLimit == null) {
			return;
		}
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token + "_DataLimit";
		
		// 将Account以token为key值，放入Redis中
		RedisUtil.setString(key, JSONObject.toJSONString(dataLimit), SSO_TOKEN_EXPIRE_SECONDS);
	}

	/**
	 * 根据token票据获取绑定的数据权限
	 * 
	 * @param token token票据
	 * 
	 * @return 数据权限DataLimit
	 */
	public static DataLimit getDataLimit(String token) {
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token + "_DataLimit";
		// 从Redis中获取数据权限信息
		String v = RedisUtil.getString(key);
		
		// 账户信息
		DataLimit dl = null;
		try {
			if (v != null && v.trim().length() > 0) {
				JSONObject json = JSONObject.parseObject(v);
				
				dl = JSONObject.toJavaObject(json, DataLimit.class);
			}
		} catch (Exception ex) {
		}
		
		return dl;
	}
	
	/**
	 * 删除token票据绑定的数据权限
	 * 
	 * @param token  token票据
	 */
	public static void deleteDataLimit(String token) {
		String key = Constants.KEY_CURRENT_ACCOUNT + ":" + token + "_DataLimit";
		RedisUtil.setString(key, null, 0);
	}
	
	/**
	 * 延长Token有效期
	 * @param token Token票据
	 */
	static void refreshToken(String token) {
		if (token != null && token.trim().length() > 0) {
			RedisUtil.expire(token, SSO_TOKEN_EXPIRE_SECONDS);
		}
	}

	/**
	 * @return 当前有效会话的数量
	 */
	public static int getSessionCount() {
		String pattern = Constants.KEY_CURRENT_ACCOUNT + ":*";
		Set<String> ss = RedisUtil.keys(pattern);

		return (ss == null ? 0 : ss.size());
	}
}
