package tools;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import team.bangbang.common.config.Config;
import team.bangbang.common.sql.SQLPool;
import team.bangbang.common.sql.generator.TableKeyMetaData;

/**
 * 获得数据库中所有的表，按字母顺序排列后输出
 *
 * @author 帮帮组
 * @version 1.0 2018年4月6日
 */
public class DBCompare {
	/* 基准数据库别名 */
	private static final String ALIAS_DEFAULT = "application";
	/* 和基准数据库比较的数据库别名 */
	private static final String ALIAS_OTHER = "other";

	/**
	 * @param args 命令行参数
	 */
	public static void main(String[] args) {
		Set<String> tbls1 = TableKeyMetaData.getTables(ALIAS_DEFAULT);
		Set<String> tbls2 = TableKeyMetaData.getTables(ALIAS_OTHER);

		// default与other中共同含有的数据库表
		List<String> joinList = new ArrayList<String>();

		// 以default为基准，xuejava缺少的表
		System.out.print("## 1. 以default为基准，other缺少的表 ");
		// other中缺少的表
		List<String> noList = new ArrayList<String>();

		for(String table : tbls1) {
			if(!tbls2.contains(table)) {
				noList.add(table);
			} else {
				joinList.add(table);
			}
		}
		System.out.println(noList.size() + " 张");
		Collections.sort(noList);
		for (int i = 0; i < noList.size(); i++) {
			System.out.println("\t" + (i + 1) + "  - " + noList.get(i));
		}

		System.out.println("\n");

		Collections.sort(joinList);

		System.out.println("## 2. 以default为基准，other缺少（-）、修改（*）、增加（+）的字段");
		// 对共同含有的表
		for(int i = 0; i < joinList.size(); i++) {
			String table = joinList.get(i);
			Map<String, Field> mp1 = getFields(ALIAS_DEFAULT, table);
			Map<String, Field> mp2 = getFields(ALIAS_OTHER, table);

			StringBuffer sb = new StringBuffer();;
			// 以default为基准，xuejava缺少的字段
			String s1 = compareFields1(table, mp1, mp2);
			if(s1 != null && s1.trim().length() > 0) {
				if(sb.length() > 0) sb.append("\n");
				sb.append(s1);
			}

			// 以default为基准，xuejava修改的字段
			String s2 = compareFields2(table, mp1, mp2);
			if(s2 != null && s2.trim().length() > 0) {
				if(sb.length() > 0) sb.append("\n");
				sb.append(s2);
			}

			// 以default为基准，xuejava增加的字段
			String s3 = compareFields3(table, mp1, mp2);
			if(s3 != null && s3.trim().length() > 0) {
				if(sb.length() > 0) sb.append("\n");
				sb.append(s3);
			}

			if(sb.length() > 0) {
				sb.insert(0, "\t " + table + "\n");
				System.out.println(sb);
				System.out.println();
			}
		}
	}

	private static String compareFields3(String table, Map<String, Field> mp1, Map<String, Field> mp2) {
		StringBuffer sb = new StringBuffer();
		for(String name : mp2.keySet()) {
			Field f1 = mp1.get(name);
			Field f2 = mp2.get(name);

			if(f1 == null) {
				if(sb.length() > 0) sb.append("\n");
				sb.append("\t  + " + f2.name + " " + f2.type + "(" + f2.size + ")");
			}
		}

		return sb.toString();
	}

	private static String compareFields2(String table, Map<String, Field> mp1, Map<String, Field> mp2) {
		StringBuffer sb = new StringBuffer();
		for(String name : mp1.keySet()) {
			Field f1 = mp1.get(name);
			Field f2 = mp2.get(name);

			if(f2 == null) {
				continue;
			}

			if(!f1.type.equals(f2.type) || f1.size != f2.size) {
				if(sb.length() > 0) sb.append("\n");
				sb.append("\t  * " + f1.name + " " + f1.type + getTypeSize(f1.size) + " ==> " +  f2.name + " " + f2.type + getTypeSize(f2.size));
			}
		}

		return sb.toString();
	}

	private static String getTypeSize(int size) {
		return (size <= 0 ? "" : "(" + size + ")");
	}

	// 以default为基准，xuejava缺少的字段
	private static String compareFields1(String table, Map<String, Field> mp1, Map<String, Field> mp2) {
		StringBuffer sb = new StringBuffer();
		for(String name : mp1.keySet()) {
			Field f1 = mp1.get(name);
			Field f2 = mp2.get(name);

			if(f2 == null) {
				if(sb.length() > 0) sb.append("\n");
				sb.append("\t  - " + f1.name);
			}
		}

		return sb.toString();
	}


	private static Map<String, Field> getFields(String alias, String table) {
		Map<String, Field> mp = new HashMap<String, Field>();
		// 获取default数据库表字段
		// 数据库链接
		Connection conn = null;
		// 结果记录集
		ResultSet rs = null;

		try {
			// 获得数据库的连接
			conn = SQLPool.getConnection(alias);

			String catalog = conn.getCatalog();
			DatabaseMetaData dmd = conn.getMetaData();
			// 读取设定的Schema
			String schema = getSchema(dmd, alias);
			// 关键关键字段的类型、长度
			rs = dmd.getColumns(catalog, schema, table, null);
			if (rs != null) {
				while (rs.next()) {
					Field f = new Field();
					f.name = rs.getString("COLUMN_NAME");
					f.type = rs.getString("TYPE_NAME").toLowerCase();

					if(f.type.equals("bigint") || f.type.equals("tinyint") || f.type.indexOf("date") > 0 || f.type.indexOf("time") > 0) {
						// 省略长度
					} else {
						f.size = rs.getInt("COLUMN_SIZE");
					}

					mp.put(f.name, f);
				}
				rs.close();
			}
		} catch (Exception ex) {
			ex.printStackTrace();
		} finally {
			if (rs != null) {
				try {
					rs.close();
				} catch (SQLException se) {
				}
			}

			if (conn != null) {
				try {
					conn.close();
				} catch (SQLException se) {
				}
			}
		}

		return mp;
	}


	/**
	 * 获得指定数据库的schema名称
	 *
	 * @param dmd
	 *            数据库连接元数据
	 *
	 * @param alias
	 *            数据库别名
	 * @return 数据库的schema名称
	 */
	private static String getSchema(DatabaseMetaData dmd, String alias) {
		// 读取设定的Schema
		String schema = Config.getProperty(alias + ".connection.schema");

		// 如果没有设定Schema，使用用户名
		if (schema == null) {
			String username = Config
					.getProperty(alias + ".connection.username");
			// 获取数据库的schema
			ResultSet rs = null;
			try {
				rs = dmd.getSchemas();
				while (rs != null && rs.next()) {
					String temp = rs.getString(1);
					if (temp != null && temp.equalsIgnoreCase(username)) {
						schema = temp;
						break;
					}
				}
			} catch (Exception exp) {
				exp.printStackTrace();
			} finally {
				if (rs != null) {
					try {
						rs.close();
					} catch (SQLException e) {
						e.printStackTrace();
					}
				}
			}
		}

		// 这种情况发生在SQL Server中使用sa访问，schema为dbo的时候
		if (schema != null && schema.trim().length() == 0) {
			schema = null;
		}

		return schema;
	}
}

class Field {
	public String name = null;
	public String type = null;
	public int size = 0;

	public int hashCode() {
		return (name + ":" + type + ":" + size).hashCode();
	}

	public boolean equals(Object o) {
		if(!(o instanceof Field) || o == null) {
			return false;
		}

		Field of = (Field)o;

		return (name + ":" + type + ":" + size).equals(of.name + ":" + of.type + ":" + of.size);
	}
}
