package tech.figure.objectstore.gateway.client

import io.grpc.Deadline
import io.grpc.Metadata
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder
import io.grpc.stub.AbstractStub
import io.grpc.stub.MetadataUtils
import io.provenance.scope.sdk.toPublicKeyProto
import io.provenance.scope.util.toByteString
import kotlinx.coroutines.flow.Flow
import tech.figure.objectstore.gateway.GatewayGrpc
import tech.figure.objectstore.gateway.GatewayGrpcKt
import tech.figure.objectstore.gateway.GatewayOuterClass
import tech.figure.objectstore.gateway.GatewayOuterClass.BatchGrantObjectPermissionsRequest
import tech.figure.objectstore.gateway.GatewayOuterClass.BatchGrantObjectPermissionsResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.BatchGrantScopePermissionRequest
import tech.figure.objectstore.gateway.GatewayOuterClass.BatchGrantScopePermissionResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.GrantObjectPermissionsRequest
import tech.figure.objectstore.gateway.GatewayOuterClass.GrantObjectPermissionsResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.GrantScopePermissionRequest
import tech.figure.objectstore.gateway.GatewayOuterClass.GrantScopePermissionResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.PutObjectResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.RegisterExistingObjectResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.RevokeObjectPermissionsResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.RevokeScopePermissionRequest
import tech.figure.objectstore.gateway.GatewayOuterClass.RevokeScopePermissionResponse
import tech.figure.objectstore.gateway.GatewayOuterClass.ScopeGrantee
import tech.figure.objectstore.gateway.admin.Admin.DataStorageAccount
import tech.figure.objectstore.gateway.admin.Admin.FetchDataStorageAccountRequest
import tech.figure.objectstore.gateway.admin.Admin.PutDataStorageAccountRequest
import tech.figure.objectstore.gateway.admin.GatewayAdminGrpc
import java.io.Closeable
import java.security.PublicKey
import java.time.Duration
import java.time.OffsetDateTime
import java.util.concurrent.TimeUnit

class GatewayClient(val config: ClientConfig) : Closeable {
    private val channel = NettyChannelBuilder.forAddress(config.gatewayUri.host, config.gatewayUri.port)
        .apply {
            if (config.gatewayUri.scheme == "grpcs") {
                useTransportSecurity()
            } else {
                usePlaintext()
            }
        }
        .executor(config.executor)
        .maxInboundMessageSize(config.inboundMessageSize)
        .idleTimeout(config.idleTimeout.first, config.idleTimeout.second)
        .keepAliveTime(config.keepAliveTime.first, config.keepAliveTime.second)
        .keepAliveTimeout(config.keepAliveTimeout.first, config.keepAliveTimeout.second)
        .also { builder -> config.channelConfigLambda(builder) }
        .build()

    private val gatewayStub = GatewayGrpc.newFutureStub(channel)
    private val gatewayCoroutineStub = GatewayGrpcKt.GatewayCoroutineStub(channel)
    private val gatewayAdminStub = GatewayAdminGrpc.newFutureStub(channel)

    /**
     * Fetch scope data from gateway, using an existing JWT as authentication
     * @param scopeAddress the scope's address
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun requestScopeData(
        scopeAddress: String,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): GatewayOuterClass.FetchObjectResponse = gatewayStub
        .withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .fetchObject(
            GatewayOuterClass.FetchObjectRequest.newBuilder()
                .setScopeAddress(scopeAddress)
                .build()
        )
        .get()

    /**
     * Write an object to object store via the gateway. The object will be encrypted by the server's key and the address in the JWT will be permissioned
     * to retrieve the object.
     *
     * @param objectBytes The raw data to store
     * @param objectType (Optional) the type of data that this represents. This is for reference at the time of retrieval if needed
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     * @param additionalAudienceKeys An optional list of additional public keys that should have access to this object via the gateway
     * @param useRequesterKey An optional flag to indicate that you want to store this object in object store using your own private key
     *          (only allowed if the jwt in use was generated by a key registered with the connected gateway instance)
     *
     * @return A proto containing the hash of the stored object. This hash can be used for future retrieval via [getObject].
     *  Note that this is not the hash of the provided objectBytes, but rather the sha256 hash of a serialized proto containing the provided objectBytes and objectType
     */
    fun putObject(
        objectBytes: ByteArray,
        objectType: String? = null,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
        additionalAudienceKeys: List<PublicKey> = listOf(),
        useRequesterKey: Boolean = false,
    ): PutObjectResponse {
        return gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
            .interceptJwt(jwt, timeout)
            .putObject(
                GatewayOuterClass.PutObjectRequest.newBuilder()
                    .apply {
                        objectBuilder.objectBytes = objectBytes.toByteString()
                        if (objectType != null) {
                            objectBuilder.type = objectType
                        }
                    }
                    .addAllAdditionalAudienceKeys(additionalAudienceKeys.map { it.toPublicKeyProto() })
                    .setUseRequesterKey(useRequesterKey)
                    .build()
            ).get()
    }

    /**
     * Register an existing object for access control that you have already written to object store outside of this gateway instance.
     * This will grant access to the designated addresses so the data may be fetched via the gateway
     *
     * Note: this will only succeed if the private key that generated the jwt in use is registered as a private key with the gateway instance
     * and can fetch the object from object store by hash
     *
     * @param objectHash the hash of the object to grant access to
     * @param granteeAddresses a list of provenance addresses to grant access to for this object hash
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     *
     * @return a response containing the request parameters
     */
    fun registerExistingObject(
        objectHash: String,
        granteeAddresses: List<String>,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT
    ): RegisterExistingObjectResponse {
        return gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
            .interceptJwt(jwt, timeout)
            .registerExistingObject(
                GatewayOuterClass.RegisterExistingObjectRequest.newBuilder()
                    .setHash(objectHash)
                    .addAllGranteeAddress(granteeAddresses)
                    .build()
            ).get()
    }

    /**
     * Retrieve an object from object store via the gateway. The object will only be returned if the address contained within the authenticated jwt
     * has been granted access via the gateway.
     *
     * @param hash The hash of the object to retrieve (as returned by [putObject])
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     *
     * @return A proto containing an object that holds the provided objectBytes and objectType as provided by [putObject]
     */
    fun getObject(
        hash: String,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): GatewayOuterClass.FetchObjectByHashResponse {
        return gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
            .interceptJwt(jwt, timeout)
            .fetchObjectByHash(
                GatewayOuterClass.FetchObjectByHashRequest.newBuilder()
                    .setHash(hash)
                    .build()
            ).get()
    }

    /**
     * Creates an object permission grant from the caller to the grantee.  If the caller does not have access to the
     * target object, an ACCESS_DENIED error will be emitted.
     *
     * @param hash The hash of the object to receive (as returned by [putObject])
     * @param granteeAddress The bech32 address of the account that will receive access to the object.
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     *
     * @return A proto containing the successfully-granted account's address and the hash of the granted object.
     */
    fun grantObjectPermissions(
        hash: String,
        granteeAddress: String,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): GrantObjectPermissionsResponse = gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .grantObjectPermissions(GrantObjectPermissionsRequest.newBuilder().setHash(hash).setGranteeAddress(granteeAddress).build())
        .get()

    /**
     * Creates object permission grants from the caller to the grantee for all, or a specific number of hashes.  If the
     * caller does not have access to a target hash, that requested grant will be ignored.  This returns a coroutine Flow
     * that will emit protos containing the grantee address and granted object hash for each grant that is created
     * throughout this process. The flow will terminate after all grants have been created.
     *
     * @param granteeAddress The bech32 address of the account that will receive access to objects.
     * @param providedObjectHashes If null, all objects owned by the granter will be granted to the grantee. If provided,
     * only the hashes in this value will be granted to the grantee.
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request.
     * @param jwtExpireAfter The time at which the JWT should be validated as expired.
     */
    fun batchGrantObjectPermissions(
        granteeAddress: String,
        providedObjectHashes: Collection<String>? = null,
        jwt: GatewayJwt,
        jwtExpireAfter: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): Flow<BatchGrantObjectPermissionsResponse> = gatewayCoroutineStub.interceptJwt(jwt, jwtExpireAfter)
        .batchGrantObjectPermissions(
            request = BatchGrantObjectPermissionsRequest.newBuilder().also { request ->
                if (providedObjectHashes != null) {
                    request.specifiedHashesBuilder.granteeAddress = granteeAddress
                    request.specifiedHashesBuilder.addAllTargetHashes(providedObjectHashes)
                } else {
                    request.allHashesBuilder.granteeAddress = granteeAddress
                }
            }.build(),
        )

    /**
     * Revoke access for a particular object hash to a list of addresses
     *
     * Note: if another address has granted access to the same hash, that access will not be removed
     *
     * @param objectHash the hash of the object to revoke access to
     * @param granteeAddresses a list of provenance addresses to revoke access to for this object hash
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun revokeObjectPermissions(
        objectHash: String,
        granteeAddresses: List<String>,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT
    ): RevokeObjectPermissionsResponse {
        return gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
            .interceptJwt(jwt, timeout)
            .revokeObjectPermissions(
                GatewayOuterClass.RevokeObjectPermissionsRequest.newBuilder()
                    .setHash(objectHash)
                    .addAllGranteeAddress(granteeAddresses)
                    .build()
            ).get()
    }

    /**
     * Grants permission to the grantee to view the records associated with the given Provenance Blockchain Scope
     * address.  The caller of this function has to be either the value owner of the given scope, or the administrator
     * of the gateway application.
     *
     * @param scopeAddress The bech32 Provenance Blockchain Scope address for which to grant access
     * @param granteeAddress The bech32 Provenance Blockchain Account address to which access will be granted
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param grantId A free-form grant identifier that will be appended to the record created in the scope_permissions
     * table for targeted revokes.  If omitted, the record created will have a null grant id
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun grantScopePermission(
        scopeAddress: String,
        granteeAddress: String,
        jwt: GatewayJwt,
        grantId: String? = null,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): GrantScopePermissionResponse = gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .grantScopePermission(
            GrantScopePermissionRequest.newBuilder().also { request ->
                request.scopeAddress = scopeAddress
                request.granteeAddress = granteeAddress
                grantId?.also { request.grantId = it }
            }.build()
        )
        .get()

    /**
     * Grants permission to all grantees to view the records associated with the given Provenance Blockchain Scope
     * address.  The caller of this function has to be either the value owner of the given scope, or the administrator
     * of the gateway application.
     *
     * @param scopeAddress The bech32 Provenance Blockchain Scope address for which to grant access
     * @param grantees All Provenance Blockchain accounts to which access will be granted
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun batchGrantScopePermission(
        scopeAddress: String,
        grantees: Collection<ScopeGrantee>,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): BatchGrantScopePermissionResponse = gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .batchGrantScopePermission(
            BatchGrantScopePermissionRequest.newBuilder().also { request ->
                request.scopeAddress = scopeAddress
                request.addAllGrantees(grantees)
            }.build()
        )
        .get()

    /**
     * Revokes permission from the grantee to view the records associated with the given Provenance Blockchain Scope
     * address.  The caller of this function has to be one of the following to avoid request rejection:
     * - The associated scope's value owner
     * - The master administrator of the gateway application
     * - The grantee (accounts that have been granted permissions to a scope can revoke their own permissions if desired)
     *
     * @param scopeAddress The bech32 Provenance Blockchain Scope address for which to revoke access
     * @param granteeAddress The bech32 Provenance Blockchain Account address for which access will be revoked
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param grantId A free-form grant identifier that will be used to query for existing scope_permissions records.
     * If this value is omitted, all grants for the given scope and grantee combination will be revoked, versus targeting
     * a singular unique record with the given id.
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun revokeScopePermission(
        scopeAddress: String,
        granteeAddress: String,
        jwt: GatewayJwt,
        grantId: String? = null,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): RevokeScopePermissionResponse = gatewayStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .revokeScopePermission(
            RevokeScopePermissionRequest.newBuilder().also { request ->
                request.scopeAddress = scopeAddress
                request.granteeAddress = granteeAddress
                grantId?.also { request.grantId = it }
            }.build()
        )
        .get()

    /**
     * Adds a new data storage account or updates an existing data storage account in the gateway service with the
     * given address and additional properties.  This will allow an account access to the fetchObject and putObject
     * functions when signing those requests with a JWT.
     *
     * THIS ROUTE MANDATES THE USE OF THE GATEWAY MASTER KEY.  ALL OTHER ADDRESSES' REQUESTS WILL BE REJECTED.
     *
     * @param address The Provenance Blockchain bech32 address of the account that will be added (or updated) as an
     * authorized user for using object storage routes
     * @param enabled If true, this account will be enabled for object storage routes.  If false, the account will be
     * barred from this functionality
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun adminPutDataStorageAccount(
        address: String,
        enabled: Boolean,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): DataStorageAccount = gatewayAdminStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .putDataStorageAccount(
            PutDataStorageAccountRequest.newBuilder().also { request ->
                request.address = address
                request.enabled = enabled
            }.build()
        )
        .get()
        .account

    /**
     * Fetches a data storage account existing in the gateway.  If an account does not exist in the server for this
     * address, a NOT_FOUND error will be emitted.
     *
     * THIS ROUTE MANDATES THE USE OF THE GATEWAY MASTER KEY.  ALL OTHER ADDRESSES' REQUESTS WILL BE REJECTED.
     *
     * @param address The Provenance Blockchain bech32 address of the account for which to fetch a storage record
     * @param jwt Any instance of GatewayJwt for use in generating the proper JWT metadata for the request
     * @param timeout An optional timeout for the request that also controls the timeout for any generated jwt
     */
    fun adminFetchDataStorageAccount(
        address: String,
        jwt: GatewayJwt,
        timeout: Duration = GatewayJwt.DEFAULT_TIMEOUT,
    ): DataStorageAccount = gatewayAdminStub.withDeadline(Deadline.after(timeout.seconds, TimeUnit.SECONDS))
        .interceptJwt(jwt, timeout)
        .fetchDataStorageAccount(FetchDataStorageAccountRequest.newBuilder().setAddress(address).build())
        .get()
        .account

    override fun close() {
        channel.shutdown()
    }

    /**
     * A helper extension function to dynamically append header metadata that includes a generated JWT to the
     * gatewayStub's requests
     */
    private fun <S : AbstractStub<S>> S.interceptJwt(jwt: GatewayJwt, timeout: Duration): S = this
        .withInterceptors(
            MetadataUtils.newAttachHeadersInterceptor(
                Metadata().apply {
                    this.put(
                        Constants.JWT_GRPC_HEADER_KEY,
                        jwt.createJwt(mainNet = config.mainNet, expiresAt = OffsetDateTime.now() + timeout),
                    )
                }
            )
        )
}
